/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
/* \paragraph{Error messages} tell the operator that a computation has */
/* terminated abnormally, and has failed to produce an acceptable result. */
/* Normally this is associated with assigning a non-zero */
/* \verb@statusCode@; an error message is printed automatically whenever */
/* a function exits with non-zero \verb@statusCode@. */

/* \paragraph{Warning messages} tell the user that a computation is */
/* working, but with unusual behaviour that might indicate an unreliable */
/* or meaningless result.  Warnings do not normally result in a non-zero */
/* \verb@statusCode@. */

/* \paragraph{Information messages} tell the operator that the */
/* computation is proceeding as expected, and simply provide additional */
/* information about its progress. */

/* \subsubsection{Abnormal return from a function} */
/* \idx{ABORT()} */

/* The standard method to terminate a function unsuccessfully is with the */
/* \verb@ABORT()@ macro, which takes three arguments: the status pointer, */
/* the status code, and the status description string.  Normally the */
/* various error codes and descriptions will be constants defined in the */
/* function's header file \verb@MyHeader.h@: */
/* \begin{verbatim} */
/* ABORT( stat, MYHEADERH_EMYERR, MYHEADERH_MSGEMYERR ); */
/* \end{verbatim} */
/* where the error code \verb@MYHEADERH_EMYERR@ and the error message */
/* \verb@MYHEADERH_MSGEMYERR@ are defined in \verb@MyHeader.h@.  This */
/* standard LAL naming convention for error messages prevents namespace */
/* conflicts between different header files.  Like \verb@RETURN()@, */
/* \verb@ABORT()@ correctly handles any status logging required by the */
/* implementation and the \verb@lalDebugLevel@.  Note that \verb@ABORT()@ */
/* does \emph{not} raise a \verb@SIGABRT@ signal, but instead returns */
/* control to the calling routine. */

/* \subsubsection{Error checking within a function} */
/* \idx{ASSERT()} */

/* Another way to indicate an unsuccessful termination is with the macro */
/* \verb@ASSERT()@, which takes as arguments a test statement, a status */
/* pointer, a status code, and a status description.  The statement */
/* \verb@ASSERT( assertion, ... );@ is in all ways equivalent to the */
/* statement \verb@if ( !assertion ) ABORT( ... );@, except on a failure */
/* the \verb@ASSERT()@ macro will also report the failed assertion.  In */
/* the above example, one might have: */
/* \begin{verbatim} */
/* ASSERT( assertion, stat, MYHEADERH_EMYERR, MYHEADERH_MSGEMYERR ); */
/* \end{verbatim} */

/* One subtle but important point is that the \verb@ASSERT()@ should be */
/* used only to trap coding errors, rather than runtime errors, which */
/* would be trapped using \verb@ABORT()@.  In other words, the assertion */
/* should always test true in the final debugged program.  This is vital */
/* because certain compilation flags will remove all \verb@ASSERT()@ */
/* macros at compile time, in order to speed execution of the final code. */
/* See Sec.~\ref{ss:compilation-flags}, below. */

/* Programmers should also be aware that using \verb@ASSERT()@ to exit a */
/* function in normal runtime can have serious side effects.  For */
/* example, it is an error to allocate dynamic memory to local variables */
/* in a function and then fail to free it before returning.  Thus, if you */
/* have dynamically allocated memory, you cannot then use \verb@ASSERT()@ */
/* for runtime error checking, as this does not permit you to free the */
/* memory before returning.  Instead, you must explicitly check the */
/* assertion, and, if it fails, free the memory and call \verb@ABORT()@. */

#include <stdlib.h>
#include <string.h>

#ifndef _BCC_MACROS_H
#define _BCC_MACROS_H

#ifdef  __cplusplus
extern "C" {
#endif

#ifdef NDEBUG 
#ifndef BCC_NDEBUG
#define BCC_NDEBUG
#endif
#endif
  
  extern int bccDebugLevel;
  extern const int bccNoDebug;
  
  enum { BCCInfoLength = 1024 };
  
  enum
    {
      BCCNDEBUG  = 0,
      BCCERROR   = 1,
      BCCWARNING = 2,
      BCCINFO    = 4,
    };
  
  /* composite bccDebugLevels: */
  
  enum { BCCMSGLVL1  = BCCERROR };
  enum { BCCMSGLVL2  = BCCERROR | BCCWARNING };
  enum { BCCMSGLVL3  = BCCERROR | BCCWARNING | BCCINFO };
  
  
#define INITSTATUSPTR( statusptr )                                             \
  if ( (statusptr) )                                                           \
  {                                                                            \
    memset( (statusptr), 0, sizeof( bcc_status ) ); /* possible memory leak */ \
    SETSTATUSFILEFUNCLINE( statusptr );                                        \
  }                                                                            \
  else                                                                         \
    bcc_abort( "Abort: function %s, file %s, line %d\n"                        \
                  "       Null status pointer passed to function\n",           \
                  __func__, __FILE__, __LINE__ )

#define CHECKSTATUSPTR( statusptr )                                           \
  if ( ! (statusptr) )                                                        \
    bcc_abort( "Abort: function %s, file %s, line %d\n"                       \
                  "       Null status pointer passed to function\n",          \
                  __func__, __FILE__, __LINE__ );                             \
  SETSTATUSFILEFUNCLINE( statusptr );                                         \


#define RETURN( statusptr )                                                   \
  if ( 1 )                                                                    \
  {                                                                           \
    SETSTATUSFILEFUNCLINE( statusptr );                                       \
    if ( (statusptr)->code)                                                   \
      (void) bcc_error( statusptr, "RETURN:" );                               \
    return;                                                                   \
  }                                                                           \
  else (void)(0)

#define ABORT( statusptr, code, mesg )                                        \
  if ( 1 )                                                                    \
  {                                                                           \
    SETSTATUS( statusptr, code, mesg );                                       \
    if ( code )                                                               \
      (void) bcc_error( statusptr, "ABORT:" );                                \
    return;                                                                   \
  }                                                                           \
  else (void)(0)

#ifdef BCC_NDEBUG
#define ASSERT( assertion, statusptr, code, mesg )
#else
#define ASSERT( assertion, statusptr, code, mesg )                            \
  if ( !(assertion) )                                                         \
  {                                                                           \
    SETSTATUS( statusptr, code, mesg );                                       \
    (void) bcc_error( statusptr, "Assertion \"" #assertion "\" failed:" );    \
    return;                                                                   \
  }                                                                           \
  else (void)(0)
#endif

#define TRY( func, statusptr )                                                \
  if ( (func), (statusptr)->code )                                            \
  {                                                                           \
    SETSTATUS( statusptr, -1, "Recursive error" );                            \
    (void) bcc_error( statusptr, "Function call \"" #func "\" failed:" );     \
    return;                                                                   \
  }                                                                           \
  else (void)(0)

#define SETSTATUSFILEFUNCLINE( statusptr )                                    \
  ( ( void ) ( (statusptr)->file = __FILE__,                                  \
      (statusptr)->function = __func__,                                       \
      (statusptr)->line = __LINE__ ) )

#define SETSTATUS( statusptr, label, mesg )                                   \
  ( SETSTATUSFILEFUNCLINE( statusptr ),                                       \
    (statusptr)->description = (mesg),                                        \
    (statusptr)->code = (label) )

#define SUB( func, statusptr )                                                \
  if ( (func), (statusptr)->code )                                            \
  {                                                                           \
    SETSTATUS( statusptr, -1, "Recursive error" );                            \
    (void) bcc_error( statusptr, "Function call \"" #func "\" failed:" );     \
    exit(EXIT_FAILURE);                                                       \
  }                                                                           \
  else (void)(0)

#ifdef NDEBUG

#define bcc_error( statusptr, statement ) 0
#define bcc_warning( statusptr, warning ) 0
#define bcc_info( statusptr, info )       0
#define bcc_log( statusptr, info )        0

#else

#define bcc_error( statusptr, statement )                                    \
  ( bccDebugLevel & BCCERROR ?                                               \
    bcc_print_error( "Error %d: function %s, file %s, line %d\n"             \
        "        %s %s\n", (statusptr)->code,                                \
        (statusptr)->function, (statusptr)->file, (statusptr)->line,         \
        (statement) ? (statement) : "",                                      \
        (statusptr)->description ) : 0 )

#define bcc_warning( statusptr, warning )                                   \
  ( bccDebugLevel & BCCWARNING ?                                            \
    bcc_print_error( "Warning: function %s, file %s, line %d\n"             \
        "        %s\n", (statusptr)->function,                              \
        (statusptr)->file, (statusptr)->line, (warning) )                   \
      : 0 )

#define bcc_info( statusptr, info )                                         \
  ( bccDebugLevel & BCCINFO ?                                               \
    bcc_print_error( "Info: function %s, file %s, line %d\n"                \
        "        %s\n", (statusptr)->function,                              \
        (statusptr)->file, (statusptr)->line, (info) )                      \
      : 0 )

#define bcc_log( statusptr, log )                                           \
  ( bccDebugLevel & BCCINFO ? bcc_print_error( "Log: %s\n", (log) ) : 0 )

#endif /* NDEBUG */

#ifdef  __cplusplus
}
#endif

#endif /* _BCC_MACROS_H */
