/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "filtering.h"

#ifdef  __cplusplus
extern "C" {
#endif
    
  void filtering_init_buffer(bcc_status* status, filtering_buffers* b, const int N, const int Nf, const int Nw)
  {
    int k;
    
    CHECKSTATUSPTR(status);
    
    ASSERT((Nf>0)&(is_power_of_two(Nf)==1),status,FLT_EFRQZ,FLT_MSGEFRQZ);
    ASSERT((Nw>=0),status,FLT_EOVLZ,FLT_MSGEOVLZ);

    TRY(create_vector(status,&b->in,Nf),status);
    b->out = (fftw_complex*) fftw_malloc((Nf/2+1)*sizeof(fftw_complex));
    b->fwd = fftw_plan_dft_r2c_1d (Nf, b->in->data, b->out, FFTW_ESTIMATE);
    b->bck = fftw_plan_dft_c2r_1d (Nf, b->out, b->in->data, FFTW_ESTIMATE);
    TRY(create_vector(status,&b->win,Nw),status);
    for (k=0 ; k<Nw ; k++)
      b->win->data[k] = 0.5 * (1.0 + cos(M_PI*(k+1.0)/(Nw+1.0)));

    TRY(create_vector(status,&b->raw,N),status);
    TRY(create_vector(status,&b->proc,Nf),status);

    RETURN(status);
  }

  void filtering_clear_buffer(bcc_status* status, filtering_buffers* b)
  {
    CHECKSTATUSPTR(status);

    ASSERT(b,status,FLT_ENULL,FLT_MSGENULL);

    TRY(reset_vector(status,b->raw),status);
    TRY(reset_vector(status,b->proc),status);

    RETURN(status);
  }

  static void filter_one_block(bcc_status* status, vector *proc, const vector *in, const vector *left_in, const vector *filter, const filtering_buffers *b, const int enable_surrogate)
  {
    int N,Nf,Nw,k;
    double phase;

    CHECKSTATUSPTR(status);

    ASSERT(proc,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(proc->data,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(in,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(in->data,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(filter,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(filter->data,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(left_in,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(left_in->data,status,FLT_ENULL,FLT_MSGENULL);

    if (proc->length!=b->in->length)
      {ABORT(status,FLT_EBLKZ,FLT_MSGEBLKZ);}

    if (filter->length!=b->in->length/2+1)
      {ABORT(status,FLT_EBLKZ,FLT_MSGEBLKZ);}

    N=in->length;
    Nw=b->win->length;
    Nf=b->in->length;

    TRY(reset_vector(status,b->in),status);

    /* make a local copy of the data */
    memcpy(b->in->data,in->data,N*sizeof(double));
    
    /* multiply right edge by Tukey window */
    for (k=0 ; k<Nw ; k++)
      b->in->data[N-Nw+k] *= b->win->data[k];
    
    /* glue left edge, multiply by Tukey window */
    for (k=0 ; k<Nw ; k++)
      b->in->data[Nf-1-k] = left_in->data[N-1-k]* b->win->data[k];
    
    /* take FFT */
    fftw_execute(b->fwd);
    
    /* filtering */    
    for (k=0 ; k< filter->length ; k++)
      {
	b->out[k][0] *= filter->data[k];
	b->out[k][1] *= filter->data[k];
      }

    if (enable_surrogate==BCC_TRUE)
      {
	for (k=0 ; k< filter->length ; k++)
	  {
	    if (filter->data[k]>0)
	      {
		phase = 2.0*M_PI* (double) (0.5 - rand()/(RAND_MAX+1.0));
		b->out[k][0] *= cos(phase);
		b->out[k][1] *= sin(phase);
	      }
	  }
      }
    
    /* take inverse FFT */
    fftw_execute(b->bck);

    /* copy the result to output */
    memcpy(proc->data,b->in->data,Nf*sizeof(double));

    RETURN(status);
  }
  
  static void overlap_and_add(bcc_status* status, vector *out, vector *next, const vector *current, const int overlap)
  {
    int N,Nf,k;

    CHECKSTATUSPTR(status);

    ASSERT(out,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(out->data,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(next,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(next->data,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(current,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(current->data,status,FLT_ENULL,FLT_MSGENULL);

    N=out->length;
    Nf=next->length;
    
    /* copy current block to output */
    memcpy(out->data,current->data,N*sizeof(double));

    /* overlap and add next to out (right side) */
    for (k=1; k<overlap ; k++)
      out->data[N-k] += next->data[Nf-k];

    /* overlap and add next to next (left side) */
    for (k=0; k<overlap ; k++)
      next->data[k] += current->data[N+k];

    RETURN(status);
  }

  void filtering(bcc_status *status, time_series *out, filtering_buffers *b, const time_series *in, const vector *filter, const filtering_params *p)
  {

    int k,N,j;
    vector *curr_proc=NULL;
    static vector block_in, block_left, block_out;

    CHECKSTATUSPTR(status);

    ASSERT(in,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(in->vec,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(b,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(out,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(out->vec,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(out->vec->length==in->vec->length,status,FLT_EVECZ,FLT_MSGEVECZ);
    ASSERT(filter,status,FLT_ENULL,FLT_MSGENULL);
    ASSERT(p,status,FLT_ENULL,FLT_MSGENULL);

    ASSERT(p->block_length>0,status,FLT_EBLKZ,FLT_MSGEBLKZ);
    ASSERT(in->vec->length%p->block_length==0,status,FLT_EBLKZ,FLT_MSGEBLKZ);

    N=p->block_length;

    TRY(create_vector(status,&curr_proc,p->n_fft),status);
	
    block_in.length=N;
    block_in.data=in->vec->data;

    TRY(filter_one_block(status,curr_proc,&block_in,b->raw,filter,b,p->use_surrogate_data),status);

    block_out.length=N;
    block_out.data=out->vec->data;

    TRY(overlap_and_add(status,&block_out,curr_proc,b->proc,p->overlap_length),status);

    block_left.length=N;

    k=N; 
    while (k<in->vec->length)
      {
	block_in.data=in->vec->data+k;
	block_left.data=block_in.data-N;

	TRY(filter_one_block(status,b->proc,&block_in,&block_left,filter,b,p->use_surrogate_data),status);

	block_out.data=out->vec->data+k;

	TRY(overlap_and_add(status,&block_out,b->proc,curr_proc,p->overlap_length),status);

	TRY(copy_vector(status,curr_proc,b->proc),status);

	k+=N;
      }

    TRY(copy_vector(status,b->raw,&block_in),status);
    
    out->epoch=in->epoch - (double) N * in->deltaT;
    out->deltaT=in->deltaT;

    TRY(destroy_vector(status,&curr_proc),status);

    RETURN(status);
  }
    
#ifdef  __cplusplus
}
#endif
