/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "dwv.h"

#ifdef  __cplusplus
extern "C" {
#endif
  
  static inline int 
  get_min ( int x, int y)
  {
    return x < y ? x : y;
  }

  static int 
  is_power_of_two(int n)
  {
    while(!(n & 1))
      n = n>>1;
    return(n);
  }
  
  void create_FFTW_plan(bcc_status* status, fftw_plan** p, const matrix * m, const unsigned flag)
  {

    const fftw_r2r_kind kind=FFTW_REDFT00;

    CHECKSTATUSPTR(status);

    ASSERT( p != NULL, status, DWV_EVPTR, DWV_MSGEVPTR );
    ASSERT( *p == NULL, status, DWV_EUPTR, DWV_MSGEUPTR );
    ASSERT( m, status, DWV_ENULL, DWV_MSGENULL );
    ASSERT( m->data1, status, DWV_ENULL, DWV_MSGENULL );
    ASSERT(is_power_of_two(m->rows-1)==1,status,DWV_EFRQZ,DWV_MSGEFRQZ);

    *p=(fftw_plan*) malloc(m->columns*sizeof(fftw_plan));
    if ( NULL == *p )
      {
	ABORT( status, DWV_EMALLOC, DWV_MSGEMALLOC );
      }
	
    /* fftw_plan fftw_plan_many_r2r(int rank, const int *n, int howmany, */
    /*                                   double *in, const int *inembed, */
    /*                                   int istride, int idist, */
    /*                                   double *out, const int *onembed, */
    /*                                   int ostride, int odist, */
    /*                                   const fftw_r2r_kind *kind, unsigned flags); */

    **p= fftw_plan_many_r2r(1, &m->rows , m->columns,
			    m->data1, NULL,
			    1, m->rows,
			    m->data1, NULL,
			    1, m->rows,
			    &kind, FFTW_ESTIMATE);
    
    RETURN(status);
  }

  void destroy_FFTW_plan(bcc_status* status, fftw_plan** p)
  {

    CHECKSTATUSPTR(status);
    
    ASSERT( p != NULL, status, DWV_EVPTR, DWV_MSGEVPTR );
    ASSERT( *p == NULL, status, DWV_EUPTR, DWV_MSGEUPTR );
    
    fftw_free(*p);
    *p=NULL;
    
    RETURN(status);
  }

  void dwv(bcc_status* status, matrix * tfd, const vector * in, const int n0, const fftw_plan *p)
  {
    int n,k,t,dt;

    CHECKSTATUSPTR(status);
    
    ASSERT(tfd,status,DWV_ENULL,DWV_MSGENULL);
    ASSERT(tfd->data1,status,DWV_ENULL,DWV_MSGENULL);
    ASSERT(in,status,DWV_ENULL,DWV_MSGENULL);
    ASSERT(in->data,status,DWV_ENULL,DWV_MSGENULL);
    ASSERT((n0>=0)&(n0<in->length),status,DWV_ETIDX,DWV_MSGETIDX);
    ASSERT(p,status,DWV_ENULL,DWV_MSGENULL);

    for (n=0 ; n<tfd->columns ; n++)
      {
	t=n0+n;
	dt=get_min(2*t,2*in->length-1-2*t);
	for (k=0 ; k<=dt ; k++)
	  tfd->data2[n][k]=in->data[(int) (t+k/2.0)]*in->data[(int) (t-k/2.0)];
	memset(tfd->data2[n]+(dt+1),0,(tfd->rows-dt)*sizeof(double));
      }

    fftw_execute(*p);

    RETURN(status);
  }
  
#ifdef  __cplusplus
}
#endif
