/* 
*  This file is part of BCC.
*
*  BCC is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCC is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "PSD.h"

#ifdef  __cplusplus
extern "C" {
#endif

  
  void PSD_init_buffer(bcc_status* status, PSD_buffers* b, const int N_fft, const int N)
  {
    int k;
    double norm=0.0;

    CHECKSTATUSPTR(status);

    ASSERT(b,status,PSD_ENULL,PSD_MSGENULL);
    ASSERT((N_fft>0) & (is_power_of_two(N_fft)==1),status,PSD_EFRQZ,PSD_MSGEFRQZ);
    ASSERT(N>=0,status,PSD_EBLKZ,PSD_MSGEBLKZ);
    
    TRY(create_vector(status,&b->in,N_fft),status);
    b->out = (fftw_complex*) fftw_malloc(N_fft*sizeof(fftw_complex));
    b->plan = fftw_plan_dft_r2c_1d (N_fft, b->in->data, b->out, FFTW_ESTIMATE);
    TRY(create_vector(status,&b->win,N),status);

    for (k=0; k<N; k++)
      {
	b->win->data[k] = 0.5 - 0.5 * cos( 2.0 * M_PI * k/(N-1.0));
	norm += b->win->data[k] * b->win->data[k];
      }
    norm=sqrt(norm);
    for (k=0; k< N; k++) 
      b->win->data[k] /= norm;

    RETURN(status);
  }
  
  static void fft_one_block(bcc_status* status, vector *out, const vector *in, const PSD_buffers *b)
  {
    int N,k;
    double s;

    CHECKSTATUSPTR(status);

    N=in->length;

    ASSERT(out,status,PSD_ENULL,PSD_MSGENULL);
    ASSERT(out->data,status,PSD_ENULL,PSD_MSGENULL);
    ASSERT(in,status,PSD_ENULL,PSD_MSGENULL);
    ASSERT(in->data,status,PSD_ENULL,PSD_MSGENULL);
    ASSERT(b,status,PSD_ENULL,PSD_MSGENULL);

    if (out->length!=b->in->length/2+1)
      {ABORT(status,PSD_EBLKZ,PSD_MSGEBLKZ);}

    /* make a local copy of the data */
    memcpy(b->in->data,in->data,N*sizeof(double));
    
    /* apply Hanning window */
    for (k=0 ; k<N; k++)
	b->in->data[k] *= b->win->data[k];

    /* take FFT */
    fftw_execute(b->plan);
	
    /* compute spectrum */
    for (k=0 ; k<out->length ; k++)
      {
	s = b->out[k][0]*b->out[k][0] + b->out[k][1]*b->out[k][1];
	out->data[k] = s;
      }

    RETURN(status);
  }
  
  void PSD_estimate(bcc_status* status, vector *psd, const time_series *in, const PSD_buffers* buff_plan, const int N)
  {

    int k, j, n;
    vector block_in;
    vector *spectrum=NULL;

    CHECKSTATUSPTR(status);

    ASSERT(in,status,PSD_ENULL,PSD_MSGENULL);
    ASSERT(in->vec,status,PSD_ENULL,PSD_MSGENULL);
    ASSERT(buff_plan,status,PSD_ENULL,PSD_MSGENULL);
    ASSERT(psd,status,PSD_ENULL,PSD_MSGENULL);
    ASSERT(N>0,status,PSD_EBLKZ,PSD_MSGEBLKZ);
    ASSERT(in->vec->length%N==0,status,PSD_EBLKZ,PSD_MSGEBLKZ);
    
    TRY(create_vector(status,&spectrum,psd->length),status);

    block_in.data=in->vec->data;
    block_in.length=N;

    TRY(reset_vector(status,psd),status);

    k=0; n=0;
    while(k<in->vec->length)
      {
	block_in.data=in->vec->data+k;
	TRY(fft_one_block(status,spectrum,&block_in,buff_plan),status);
	k+=N;

	for (j=0; j<psd->length; j++)
	  psd->data[j]+=spectrum->data[j];
	n++;
      }

    for (j=0; j<psd->length; j++)
      psd->data[j]/=n;

    RETURN(status);
  }
    
#ifdef  __cplusplus
}
#endif
