/* 
*  This file is part of BCC.
*
*  BCCsuite is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCCsuite is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "IO.h"

#ifdef HAVE_LIBHDF5
void IO_exist_hdf5(bcc_status* status, bcc_boolean *ans, const char* dir, const char* name)
{
  char fullpath[BCCStringLength];

  CHECKSTATUSPTR(status);

  ASSERT(dir,status,IO_ENULL,IO_MSGENULL);
  ASSERT(name,status,IO_ENULL,IO_MSGENULL);

  /* check if file exists */  
  sprintf(fullpath,"%s/%s",dir,name);
  
  *ans= ( access (fullpath, W_OK) == 0 ? BCC_TRUE:BCC_FALSE);
  
  RETURN(status);
}


void IO_create_hdf5(bcc_status* status, hid_t* f, const char* dir, const char* name)
{
  char fullpath[BCCStringLength];
  char infostr[BCCStringLength];

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(dir,status,IO_ENULL,IO_MSGENULL);
  ASSERT(name,status,IO_ENULL,IO_MSGENULL);

  sprintf(infostr,"Create HDF5 file %s/%s",dir,name);
  bcc_log(status,infostr);
  
  /* create a new file using default properties */
  sprintf(fullpath,"%s/%s",dir,name);
  *f = H5Fcreate(fullpath, H5F_ACC_TRUNC, H5P_DEFAULT, H5P_DEFAULT);

  if (*f < 0) 
    {ABORT(status, IO_EOPENF, IO_MSGEOPENF);}

  RETURN(status);
}

void IO_open_hdf5(bcc_status* status, hid_t* f, const char* dir, const char* name)
{
  char fullpath[BCCStringLength];
  char infostr[BCCStringLength];

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(dir,status,IO_ENULL,IO_MSGENULL);
  ASSERT(name,status,IO_ENULL,IO_MSGENULL);

  sprintf(infostr,"Open HDF5 file %s/%s",dir,name);
  bcc_log(status,infostr);

  /* open file using default properties */
  sprintf(fullpath,"%s/%s",dir,name);
  *f = H5Fopen(fullpath, H5F_ACC_RDWR, H5P_DEFAULT);
  
  if (*f < 0) 
    {ABORT(status, IO_EOPENF, IO_MSGEOPENF);}

  RETURN(status);
}

void IO_write_double_hdf5(bcc_status* status, const hid_t* f, const double* data, const char *label)
{   

  herr_t stat;
  hid_t dataset_id, dataspace_id;

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(data,status,IO_ENULL,IO_MSGENULL);
  ASSERT(label,status,IO_ENULL,IO_MSGENULL);


  /* create the data space for the dataset */
  dataspace_id = H5Screate(H5S_SCALAR);


  dataset_id = H5Dcreate(*f, label, H5T_IEEE_F64LE, dataspace_id, H5P_DEFAULT);

  /* write the dataset */
  stat = H5Dwrite(dataset_id, H5T_NATIVE_DOUBLE, H5S_ALL, H5S_ALL, H5P_DEFAULT, data);
  if (stat < 0)
    {ABORT(status, IO_EWRITE, IO_MSGEWRITE);}


/* close the dataset. */
  stat = H5Dclose(dataset_id);
  
  /* terminate access to the data space */
  stat = H5Sclose(dataspace_id);



  RETURN(status);
}



void IO_write_int_hdf5(bcc_status* status, const hid_t* f, const int* data, const char *label)
{   

  herr_t stat;
  hid_t dataset_id, dataspace_id;

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(data,status,IO_ENULL,IO_MSGENULL);
  ASSERT(label,status,IO_ENULL,IO_MSGENULL);


  /* create the data space for the dataset */
  dataspace_id = H5Screate(H5S_SCALAR);


  dataset_id = H5Dcreate(*f, label, H5T_NATIVE_INT, dataspace_id, H5P_DEFAULT);

  /* write the dataset */
  stat = H5Dwrite(dataset_id, H5T_NATIVE_INT, H5S_ALL, H5S_ALL, H5P_DEFAULT, data);
  if (stat < 0)
    {ABORT(status, IO_EWRITE, IO_MSGEWRITE);}


/* close the dataset. */
  stat = H5Dclose(dataset_id);
  
  /* terminate access to the data space */
  stat = H5Sclose(dataspace_id);



  RETURN(status);
}









void IO_write_vector_hdf5(bcc_status* status, const hid_t* f, const vector *data, const int n, const char *label)
{   

  herr_t stat;
  hid_t dataset_id, dataspace_id;
  hsize_t  dim;
  char infostr[BCCStringLength];

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(data,status,IO_ENULL,IO_MSGENULL);
  ASSERT(label,status,IO_ENULL,IO_MSGENULL);


  /* create the data space for the dataset */
  dim=n;
  dataspace_id = H5Screate_simple(1, &dim, NULL);

  /* test if the dataset exists */
  H5E_BEGIN_TRY
  dataset_id=H5Dopen(*f,label);
  H5E_END_TRY
  if (dataset_id >= 0)
    {
      sprintf(infostr,"Dataset %s already exists... replacing",label);
      bcc_log(status,infostr);
      H5Gunlink(*f, label);
    }

  dataset_id = H5Dcreate(*f, label, H5T_IEEE_F64LE, dataspace_id, H5P_DEFAULT);

  /* write the dataset */
  stat = H5Dwrite(dataset_id, H5T_NATIVE_DOUBLE, H5S_ALL, H5S_ALL, H5P_DEFAULT, data->data);
  if (stat < 0)
    {ABORT(status, IO_EWRITE, IO_MSGEWRITE);}

  /* close the dataset. */
  stat = H5Dclose(dataset_id);
  
  /* terminate access to the data space */
  stat = H5Sclose(dataspace_id);



  RETURN(status);
}


void IO_write_string_hdf5(bcc_status* status, const hid_t* f, const char *data, const char *label)
{   

  herr_t stat;
  hid_t dataset_id, dataspace_id, dataset_type_id;
  hsize_t  dim;

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(data,status,IO_ENULL,IO_MSGENULL);
  ASSERT(label,status,IO_ENULL,IO_MSGENULL);

  /* create the data space for the dataset */
  dim=strlen(data);
  dataspace_id = H5Screate(H5S_SCALAR);

  dataset_type_id = H5Tcopy(H5T_C_S1);
  H5Tset_size(dataset_type_id, BCCStringLength);
  dataset_id = H5Dcreate(*f, label, dataset_type_id, dataspace_id, H5P_DEFAULT);

  /* write the dataset */
  stat = H5Dwrite(dataset_id, dataset_type_id, H5S_ALL, H5S_ALL, H5P_DEFAULT, data);
  if (stat < 0)
    {ABORT(status, IO_EWRITE, IO_MSGEWRITE);}

  /* close the dataset. */
  stat = H5Dclose(dataset_id);
  






  /* terminate access to the data space */
  stat = H5Sclose(dataspace_id);





  RETURN(status);
}












void IO_write_buffer_hdf5(bcc_status* status, const hid_t* f, const buffer *in, const char *label, const char *info,double gps_time)
{

  herr_t stat;
  hid_t dataset_id, attribute_id, dataspace_id, attr_type_id;
  char fullpath[BCCStringLength];
  double attr;

  CHECKSTATUSPTR(status);
  
  ASSERT(f,status,IO_ENULL,IO_MSGENULL);
  ASSERT(in,status,IO_ENULL,IO_MSGENULL);
  ASSERT(label,status,IO_ENULL,IO_MSGENULL);
  ASSERT(info,status,IO_ENULL,IO_MSGENULL);

  /* write out data */
  TRY(IO_write_vector_hdf5(status,f,in->vec,in->used,label),status);
  
  /* open an existing dataset */
  sprintf(fullpath,"/%s",label);
  dataset_id = H5Dopen(*f, fullpath);

  /* attribute: info */

  /* create the data space for the attribute */
  dataspace_id = H5Screate(H5S_SCALAR);

  /* create the attribute */
  attr_type_id = H5Tcopy(H5T_C_S1);
  H5Tset_size(attr_type_id, BCCStringLength);
  attribute_id = H5Acreate(dataset_id, "info", attr_type_id, dataspace_id, H5P_DEFAULT);

  /* write the attribute */
  stat = H5Awrite(attribute_id, attr_type_id, info); 
  if (stat < 0)
    {ABORT(status, IO_EWRITE, IO_MSGEWRITE);}
  
  /* close the attribute */
  stat = H5Aclose(attribute_id);

  /* attribute: sampling frequency */
  
  /* create the data space for the attribute */
  dataspace_id = H5Screate(H5S_SCALAR);

  /* create the attribute */
  attribute_id = H5Acreate(dataset_id, "sampling frequency", H5T_IEEE_F64LE, dataspace_id, H5P_DEFAULT);

  /* write the attribute */
  attr=1.0/in->deltaT;
  stat = H5Awrite(attribute_id, H5T_NATIVE_DOUBLE, &attr);
  if (stat < 0)
    {ABORT(status, IO_EWRITE, IO_MSGEWRITE);}
  
  /* close the attribute */
  stat = H5Aclose(attribute_id);

  /* close the dataspace */
  stat = H5Sclose(dataspace_id);
  
  /* attribute: GPS */
  
  /* create the data space for the attribute */
  dataspace_id = H5Screate(H5S_SCALAR);

  /* create the attribute */
  attribute_id = H5Acreate(dataset_id, "GPS", H5T_IEEE_F64LE, dataspace_id, H5P_DEFAULT);

  /* write the attribute */
  attr=gps_time;
  stat = H5Awrite(attribute_id, H5T_NATIVE_DOUBLE, &attr);
  if (stat < 0)
    {ABORT(status, IO_EWRITE, IO_MSGEWRITE);}
  
  /* close the attribute */
  stat = H5Aclose(attribute_id);

  /* close the dataspace */
  stat = H5Sclose(dataspace_id);

  /* close to the dataset */
  stat = H5Dclose(dataset_id);

  RETURN(status);
}

void IO_close_hdf5(bcc_status* status, hid_t* f)
{
  herr_t stat;
  char infostr[BCCStringLength];

  CHECKSTATUSPTR(status);

  ASSERT(f,status,IO_ENULL,IO_MSGENULL);

  sprintf(infostr,"Close HDF5 file");
  bcc_log(status,infostr);

  /* Close the file. */
  stat = H5Fclose(*f);
  if (stat < 0)
    {ABORT(status, IO_ECLOSE, IO_MSGECLOSE);}

  RETURN(status);
}


void IO_destroy_hdf5(bcc_status* status, const char* dir, const char* name)
{
 
  char fullpath[BCCStringLength];
  char infostr[BCCInfoLength];

  RETURN(status);

  CHECKSTATUSPTR(status);

  ASSERT(dir,status,IO_ENULL,IO_MSGENULL);
  ASSERT(name,status,IO_ENULL,IO_MSGENULL);

  /* destroy already existing file */  
  sprintf(fullpath,"%s/%s",dir,name);
  remove(fullpath);
  sprintf(infostr,"Remove already existing HDF5 file %s",fullpath);
  bcc_log(status,infostr);
 
 
}


#endif // HAVE_LIBHDF5
