/* 
*  This file is part of BCC.
*
*  BCCsuite is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCCsuite is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#ifndef _BCC_IO_H
#define _BCC_IO_H

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "factories.h"

#include "config.h"
#ifdef HAVE_LIBHDF5
#include "hdf5.h"
#endif // HAVE_LIBHDF5
#ifdef HAVE_LIBFRAME
#include "FrameL.h"
#endif // HAVE_LIBFRAME
#ifdef HAVE_LIBFRV
#include "FrvBuf.h"
#endif // HAVE_LIBFRV

#ifdef  __cplusplus
extern "C" {
#endif

  /* data structures */

  typedef struct tagio_params{
    bcc_boolean read_from_frame;  /* switch input from frame or ASCII format */
    char in [BCCMaxDetectorNumber][BCCStringLength];  /* name of the input file */
    int n_detectors;
    char out_dir [BCCStringLength]; /* name of the output directory */
    char list_file [BCCStringLength]; /* name of the file listing the created hdf5 files */
    char channel [BCCMaxDetectorNumber][BCCStringLength]; /* name of the analyzed channel */
    char name_detectors [BCCMaxDetectorNumber][BCCStringLength]; /* name of the analyzed channel */
    double GPS_start;
    double GPS_duration;
    double GPS_end;
    double sample_rate[BCCMaxDetectorNumber]; /* sampling rate */
  } io_params;

  /* state description */

  enum {IO_EOF=-1, IO_FAIL=0, IO_OK=1};

  /* prototypes */

  void IO_open_read(bcc_status*, void**, const char *, const bcc_boolean);
  void IO_close_read(bcc_status*, void**, const bcc_boolean);
  void IO_read(bcc_status*, int *, time_series *, const void *, const char*, const double, const int, const bcc_boolean);
  void IO_info(bcc_status*, double *, const void *, const char*, const double, const bcc_boolean);
  void IO_create_dir(bcc_status*, const char *);

#ifdef HAVE_LIBHDF5
  void IO_exist_hdf5(bcc_status*, bcc_boolean *, const char*, const char*);
  void IO_create_hdf5(bcc_status*, hid_t*, const char*, const char*);
  void IO_open_hdf5(bcc_status*, hid_t*, const char*, const char*);
  void IO_write_double_hdf5(bcc_status*, const hid_t*, const double*, const char*);
  void IO_write_int_hdf5(bcc_status*, const hid_t*, const int*, const char*);
  void IO_write_vector_hdf5(bcc_status*, const hid_t*, const vector *, const int, const char *);
  void IO_write_string_hdf5(bcc_status*, const hid_t*, const char*, const char *);
  void IO_write_buffer_hdf5(bcc_status*, const hid_t*, const buffer *, const char *, const char *,double);
  void IO_close_hdf5(bcc_status*, hid_t*);
  void IO_destroy_hdf5(bcc_status*, const char*, const char*);
#endif // HAVE_LIBHDF5

#ifdef HAVE_LIBFRAME
#ifdef HAVE_LIBFRV
  void IO_open_frame(bcc_status*, FrFile**, const char*);
  void IO_close_frame(bcc_status*, FrFile **);
  void IO_read_frame(bcc_status*, int *, time_series *, FrFile *, const char*, const double, const int);
  void IO_info_frame(bcc_status*, double *, FrFile *, const char*, const double);

#endif // HAVE_LIBFRV
#endif // HAVE_LIBFRAME

  void IO_open_ascii(bcc_status*, FILE **, const char*);
  void IO_close_ascii(bcc_status*, FILE **);
  void IO_read_ascii(bcc_status*, int *, time_series *, const FILE *, const char*, const double, const int);
  void IO_info_ascii(bcc_status*, double *, const FILE *, const char*, const double);

  /* error management */

#define IO_ENULL    1
#define IO_ENNULL   2
#define IO_EMALLOC  3
#define IO_EDIR     4
#define IO_EOPENF   5
#define IO_ECLOSE   6
#define IO_EREADF   7
#define IO_EWRITE   8
#define IO_EMCHNL   9
#define IO_EMFRM    10
#define IO_ENOGPS   11
#define IO_ENSMPL   12
#define IO_EFRLIB   13
#define IO_EFRVLIB  14

#define IO_MSGENULL    "Null pointer"
#define IO_MSGENNULL   "Non null pointer"
#define IO_MSGEMALLOC  "Malloc failure"
#define IO_MSGEDIR     "Error creating dir"
#define IO_MSGEOPENF   "Error opening file"
#define IO_MSGECLOSE   "Error closing file"
#define IO_MSGEREADF   "Read error"
#define IO_MSGEWRITE   "Write error"
#define IO_MSGEMCHNL   "Could not find channel"
#define IO_MSGEMFRM    "Could not find frame"
#define IO_MSGENOGPS   "Invalid GPS time"
#define IO_MSGENSMPL   "Invalid number of samples"
#define IO_MSGEFRLIB   "No support for FrameLib"
#define IO_MSGEFRVLIB  "No support for FrVectLib"

#ifdef  __cplusplus
}
#endif

#endif /* _BCC_IO_H */
