/* 
*  This file is part of BCCsuite.
*
*  BCCsuite is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCCsuite is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "support.h"
#include "IO.h"
#include "init.h"
#include "PSD.h"
#include "pipeline.h"

int bccDebugLevel = BCCMSGLVL3;

/* Usage format string. */
#define USAGE "Usage: %s \n"\
              "   [--config file]     configuration file\n"

int main( int argc, char *argv[] )
{
  int inarg=1,k,n;
  static bcc_status status;
  char *config_name, *program_name;
  char infostr[BCCInfoLength];
  static io_params io;
  static resampling_params *r;
  static filtering_params  w;
  static pipeline_params   p;
  static time_series *raw=NULL, *resampled=NULL, *whitened=NULL;
  static buffer *proc=NULL, *left_over=NULL;
  static vector *psd=NULL, *wfilter=NULL;;
  static int io_flag=0;
  static void* in_fileid;
  static hid_t out_fileid;
  static char out_filename [BCCStringLength], label[BCCStringLength], info[BCCStringLength];
  static resampling_buffers resampling_bufs;
  static PSD_buffers PSD_bufs;
  static filtering_buffers filtering_bufs;
  static double time=0.0, step=0.0, fs=0.0, norm=0.0;
  FILE *lifi_file;
  char fullpath[BCCStringLength];
  double proc_gps_time;


  if (argc <= 1)
    {
      bcc_print_error( USAGE, *argv );
      exit(EXIT_FAILURE);
    }

  /* parse option arguments */
  while ( inarg < argc ) 
    {
      if ( !strcmp( argv[inarg], "--config" ) ) 
	{
	  if ( argc > inarg + 1 ) 
	    {
	      inarg++;
	      config_name = argv[inarg++];
	    }
	  else
	    {
	      bcc_print_error( USAGE, *argv );
	      exit(EXIT_FAILURE);
	    }
	}
      /* Check for unrecognized options. */
      else
	{
	  bcc_print_error( USAGE, *argv );
	  exit(EXIT_FAILURE);
	}
    }
  
  /* end parse option arguments */

  INITSTATUSPTR(&status);

  SUB(bcc_version(&status,BCC_TRUE),&status);

  /* initialization */
  if (!(program_name = rindex(argv[0], '/')))
    program_name= argv[0];
  else
    program_name++;


  r=(resampling_params *) malloc(BCCMaxDetectorNumber*sizeof(resampling_params));
  SUB(parse_configuration(&status,&io,&r,&w,&p,config_name),&status);
  SUB(check_and_adjust_params(&status,&io,&r,&w,&p),&status);


  /* check availability of input data */
  for (n=0; n<io.n_detectors; n++)
    {
      SUB(IO_open_read(&status, &in_fileid, io.in[n], io.read_from_frame),&status);

      SUB(IO_info(&status,&fs,in_fileid,io.channel[n],io.GPS_start,io.read_from_frame),&status);


      if (n==0)
	step=p.raw_buffer_length[n]*fs;
/*       else */
/* 	if (p.raw_buffer_length[n]*fs !=step) */
/* 	  ABORT(); */

      SUB(IO_close_read(&status,&in_fileid,io.read_from_frame),&status);      
    }


  /* memory allocation */
  SUB(create_time_series(&status,&resampled,p.rsp_buffer_length,0.0,0.0),&status);
  SUB(create_time_series(&status,&whitened,p.rsp_buffer_length,0.0,0.0),&status);
  SUB(create_vector(&status,&psd,w.n_fft/2+1),&status);
  SUB(create_vector(&status,&wfilter,w.n_fft/2+1),&status);
  SUB(create_buffer(&status,&left_over,p.segment_length,0.0,0.0),&status);
  SUB(create_buffer(&status,&proc,p.proc_buffer_length,0.0,0.0),&status);
  
  SUB(PSD_init_buffer(&status,&PSD_bufs,w.n_fft,w.block_length),&status);
  SUB(filtering_init_buffer(&status,&filtering_bufs,w.block_length,w.n_fft,w.flattop_edge_length),&status);
  
  /* init output dir */
  SUB(IO_create_dir(&status,io.out_dir),&status);
  
  for (n=0; n<io.n_detectors; n++)
    {
      
      /* create buffer for raw data */
      SUB(create_time_series(&status,&raw,p.raw_buffer_length[n],0.0,0.0),&status);
      
      /* init resampling buffer */
      SUB(resample_init_buffer(&status,&resampling_bufs,p.raw_buffer_length[n],&r[n]),&status);
      
      /* open input file */
      SUB(IO_open_read(&status, &in_fileid, io.in[n], io.read_from_frame),&status);
      
      /* do some cleaning */
      SUB(resample_clear_buffer(&status,&resampling_bufs),&status);
      SUB(filtering_clear_buffer(&status,&filtering_bufs),&status);
      SUB(clear_buffer(&status,left_over),&status);
      SUB(clear_buffer(&status,proc),&status);
      
      /* main loop */  
      time=io.GPS_start;
      sprintf(infostr,"detector %d GPS_start=%ld GPS_duration=%g GPS_end=%ld by step=%g",n+1,(long int) io.GPS_start,io.GPS_duration,(long int) io.GPS_end,step);
      bcc_log(&status,infostr);
      do
	{
	  
	  /* get data */
	  SUB(IO_read(&status,&io_flag,raw,in_fileid,io.channel[n],time,p.raw_buffer_length[n],io.read_from_frame),&status);
	  
	  raw->deltaT=fs;
	  
	  if (io_flag==IO_EOF)
	    break;
	  
	  if (io_flag==IO_FAIL)
	    {
	      SUB(resample_clear_buffer(&status,&resampling_bufs),&status);
	      SUB(filtering_clear_buffer(&status,&filtering_bufs),&status);
	      SUB(clear_buffer(&status,left_over),&status);
	      time+=step;
	      continue;
	    }

	  
	  /* resample */
	  SUB(resample(&status,resampled,&resampling_bufs,raw,&(r[n])),&status);

	  /* estimate PSD */
	  SUB(PSD_estimate(&status,psd,resampled,&PSD_bufs,w.block_length),&status);
	  
	  /* compute the (double) whitening filter */
	  TRY(reset_vector(&status,wfilter),&status);
	  norm=0.0;
	  for (k=max_i(0,(int) (w.lower_cutoff_f*w.n_fft)); k<min_i(wfilter->length,(int) (w.upper_cutoff_f*w.n_fft)) ; k++)
	    {
	      if (psd->data[k]>0.0)
		{
		  wfilter->data[k]=1.0/psd->data[k];
		  norm+=wfilter->data[k];
		}
	    }
	  norm=sqrt(2*w.n_fft*norm);

	  for (k=max_i(0,(int) (w.lower_cutoff_f*w.n_fft)); k<min_i(wfilter->length,(int) (w.upper_cutoff_f*w.n_fft)) ; k++)
	    wfilter->data[k]/=norm;
	    //wfilter->data[k]=0.0; //1.0/(double) p->n_fft;;

	  /* whiten data */
	  SUB(filtering(&status,whitened,&filtering_bufs,resampled,wfilter,&w),&status);

	  /* reshape output buffer size */
	  SUB(reshape_time_series(&status,proc,left_over,whitened,p.segment_length,p.overlap_length),&status);


	  /* write output data */
	  sprintf(out_filename,"%d.h5",(int) time);
	  SUB(IO_exist_hdf5(&status,(bcc_boolean*) &io_flag,io.out_dir,out_filename),&status);

	  if (io_flag==BCC_FALSE)
	    SUB(IO_create_hdf5(&status,&out_fileid,io.out_dir,out_filename),&status);
	  else
	    {
	      if (n==0){
		SUB(IO_destroy_hdf5(&status,io.out_dir,out_filename),&status);
		SUB(IO_create_hdf5(&status,&out_fileid,io.out_dir,out_filename),&status);
	      }
	      else
		SUB(IO_open_hdf5(&status,&out_fileid,io.out_dir,out_filename),&status);
	    }
	  if (time==io.GPS_start)
	      proc_gps_time = time;
	  else
	    proc_gps_time = time + proc->epoch;

	  /*sprintf(label,"data%d",n+1);
	  sprintf(info,"%s",io.channel[n]);
	  SUB(IO_write_buffer_hdf5(&status,&out_fileid,proc,label,info,proc_gps_time),&status);*/

	  sprintf(label,"psd%d",n+1);
	  sprintf(info,"%s",io.channel[n]);
	  SUB(IO_write_vector_hdf5(&status,&out_fileid,psd,psd->length,label),&status);

	  /*sprintf(label,"raw%d",n+1);
	  sprintf(info,"%s",io.channel[n]);
	  SUB(IO_write_vector_hdf5(&status,&out_fileid,raw->vec,raw->vec->length,label),&status);*/

	  /*sprintf(label,"rsp%d",n+1);
	  sprintf(info,"%s",io.channel[n]);
	  SUB(IO_write_vector_hdf5(&status,&out_fileid,resampled->vec,resampled->vec->length,label),&status);*/

	  /*sprintf(label,"fil%d",n+1);
	  sprintf(info,"%s",io.channel[n]);
	  SUB(IO_write_vector_hdf5(&status,&out_fileid,wfilter,wfilter->length,label),&status);*/

	  sprintf(label,"whi%d",n+1);
	  sprintf(info,"%s",io.channel[n]);
	  SUB(IO_write_vector_hdf5(&status,&out_fileid,whitened->vec,whitened->vec->length,label),&status);

	  sprintf(label,"deltaT%d",n+1);
	  sprintf(info,"%s",io.channel[n]);
	  SUB(IO_write_double_hdf5(&status,&out_fileid,&(resampled->deltaT),label),&status);

	  sprintf(label,"gps_time%d",n+1);
	  sprintf(info,"%s",io.channel[n]);
	  SUB(IO_write_double_hdf5(&status,&out_fileid,&time,label),&status);

	  sprintf(label,"name_detector%d",n+1);
	  sprintf(info,"%s",io.channel[n]);
	  SUB(IO_write_string_hdf5(&status,&out_fileid,io.name_detectors[n],label),&status);



	  SUB(IO_close_hdf5(&status,&out_fileid),&status);

	  time+=step;
	}
      while(time<io.GPS_end);
 
      /* close input files */
      SUB(IO_close_read(&status,&in_fileid,io.read_from_frame),&status);      
      SUB(destroy_time_series(&status,&raw),&status);
      
    }

  /* write list_file */
  sprintf(infostr,"Write list file %s",io.list_file);
  bcc_log(&status,infostr);
  time=io.GPS_start;
  lifi_file=fopen(io.list_file,"w");
  sprintf(out_filename,"N_DETECTORS=%d",io.n_detectors); 
  fprintf(lifi_file,"%s\n",out_filename); /* write number of detectors*/
  sprintf(out_filename,"OUT_DIR=%s",io.out_dir); 
  fprintf(lifi_file,"%s\n",out_filename); /* write output file address*/
  sprintf(out_filename,"BLOCK_SIZE=%d",p.segment_length); 
  fprintf(lifi_file,"%s\n",out_filename); /* write data block size*/
  do
    {
      sprintf(out_filename,"%d.h5",(int) time);
      fprintf(lifi_file,"%s\n",out_filename);
      time+=step;
    }
  while(time<io.GPS_end);
  fclose(lifi_file);
  

  /* free mem */
  free(r); r=NULL;
  SUB(destroy_time_series(&status,&resampled),&status);
  SUB(destroy_time_series(&status,&whitened),&status);
  SUB(destroy_buffer(&status,&left_over),&status);
  SUB(destroy_buffer(&status,&proc),&status);

  return(EXIT_SUCCESS);
}
