/* 
*  This file is part of BCCsuite.
*
*  BCCsuite is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  BCCsuite is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with BCC; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*
*  Copyright (C) 2006 Eric Chassande-Mottin, CNRS
*
*/
 
#include "init.h"

static int gcd(int n, int m)
{
  // computes the greatest common divisor of n and m
  // this is the Euclide's algorithm
  int t;
 
  while (m != 0)
    {
      t = n % m;   
      n = m;   
      m = t;
    }
 
 return(n);
}

static int largest_multiple(int n, int m)
{
  // check that the numbers are multiple and output the largest 
  int t;

  if (n>m)
    {
      t=m; // switch
      m=n;
      n=t;
    }

  if (n>0)
    {
      if (m%n==0)
	return (m);
      else
	return (-1);
    }

  return (m);
}

void parse_configuration(bcc_status* status, io_params *f, resampling_params **r, filtering_params *w, pipeline_params *q, const char* config_file)
{
  char *buffer=NULL;
  char tmp_str[BCCStringLength], infostr[BCCInfoLength];
  bcc_boolean success;
  int ncall=0, nfail=0, ndet=0, n;
  

  CHECKSTATUSPTR(status);

  ASSERT(f,status,INIT_ENULL,INIT_MSGENULL);
  ASSERT(r,status,INIT_ENULL,INIT_MSGENULL);
  ASSERT(*r,status,INIT_ENULL,INIT_MSGENULL);
  ASSERT(w,status,INIT_ENULL,INIT_MSGENULL);
  ASSERT(q,status,INIT_ENULL,INIT_MSGENULL);
  ASSERT(config_file,status,INIT_ENULL,INIT_MSGENULL);


  TRY(copy_file_to_buf(status,&buffer,config_file),status);

  n=0;
  do
    {
      sprintf(tmp_str,"IO_IN%d",n+1);
      TRY(parse(status,&success,&f->in[n],buffer,tmp_str,BCCString),status);
      ncall++; nfail+=(success==BCC_FALSE?1:0);
      n++; 
    }
  while(success);
  ndet=n-1;

  for (n=0; n<ndet; n++)
    {
      sprintf(tmp_str,"IO_CHANNEL%d",n+1);
      TRY(parse(status,&success,&f->channel[n],buffer,tmp_str,BCCString),status);
      ncall++; nfail+=(success==BCC_FALSE?1:0);
      sprintf(tmp_str,"IO_DETECTOR_NAME%d",n+1);
      TRY(parse(status,&success,&f->name_detectors[n],buffer,tmp_str,BCCString),status);
      ncall++; nfail+=(success==BCC_FALSE?1:0);
    }

  sprintf(infostr,"Configure for %d detector(s)",ndet);
  bcc_log(status,infostr);
  f->n_detectors=ndet;
  
  TRY(parse(status,&success,&f->read_from_frame,buffer,"IO_READ_FROM_FRAME",BCCBoolean),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&f->out_dir,buffer,"IO_OUT_DIR", BCCString),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&f->list_file,buffer,"IO_LIST_FILE", BCCString),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&f->GPS_start,buffer,"IO_GPS_START", BCCReal),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&f->GPS_duration,buffer,"IO_GPS_DURATION", BCCReal),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  f->GPS_end=f->GPS_start+f->GPS_duration;
  
  TRY(parse(status,&success,&q->max_raw_buffer_length,buffer,"PIPELINE_MAX_RAW_BUFFER_LENGTH", BCCReal),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&q->segment_length,buffer,"PIPELINE_SEGMENT_LENGTH",BCCInteger),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&q->overlap_ratio,buffer,"PIPELINE_SEGMENT_OVERLAP",BCCReal),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);

  for (n=0; n<ndet; n++)
    {
      sprintf(tmp_str,"RESAMPLING_DECIMATE%d",n+1);
      TRY(parse(status,&success,&(r[0]+n)->decimate,buffer,tmp_str,BCCInteger),status);
      ncall++; nfail+=(success==BCC_FALSE?1:0);
      sprintf(tmp_str,"RESAMPLING_INTERPOLATE%d",n+1);
      TRY(parse(status,&success,&(r[0]+n)->interpolate,buffer,tmp_str,BCCInteger),status);
      ncall++; nfail+=(success==BCC_FALSE?1:0);
      sprintf(tmp_str,"RESAMPLING_LOG10_REJECTION_OUT%d",n+1);
      TRY(parse(status,&success,&(r[0]+n)->log10_reject,buffer,tmp_str,BCCReal),status);
      ncall++; nfail+=(success==BCC_FALSE?1:0);
      sprintf(tmp_str,"RESAMPLING_ROLLOFF_WIDTH%d",n+1);
      TRY(parse(status,&success,&(r[0]+n)->roll_off_width,buffer,tmp_str,BCCReal),status);
      ncall++; nfail+=(success==BCC_FALSE?1:0);
      sprintf(tmp_str,"RESAMPLING_LOWER_CUTOFF_F%d",n+1);
      TRY(parse(status,&success,&(r[0]+n)->lower_cutoff_f,buffer,tmp_str,BCCReal),status);
      ncall++; nfail+=(success==BCC_FALSE?1:0);
      sprintf(tmp_str,"RESAMPLING_UPPER_CUTOFF_F%d",n+1);
      TRY(parse(status,&success,&(r[0]+n)->upper_cutoff_f,buffer,tmp_str,BCCReal),status);
      ncall++; nfail+=(success==BCC_FALSE?1:0);
    }

  TRY(parse(status,&success,&w->block_length,buffer,"FILTERING_BLOCK_LENGTH",BCCInteger),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&w->n_fft,buffer,"FILTERING_N_FFT", BCCInteger),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&w->overlap_length,buffer,"FILTERING_OVERLAP_LENGTH", BCCInteger),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&w->flattop_edge_length,buffer,"FILTERING_FLATTOP_EDGE_LENGTH", BCCInteger),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&w->lower_cutoff_f,buffer,"FILTERING_LOWER_CUTOFF_F",BCCReal),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&w->upper_cutoff_f,buffer,"FILTERING_UPPER_CUTOFF_F",BCCReal),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);
  TRY(parse(status,&success,&w->use_surrogate_data,buffer,"FILTERING_USE_SURROGATE_DATA",BCCBoolean),status);
  ncall++; nfail+=(success==BCC_FALSE?1:0);

  if (nfail==ncall)
    {ABORT(status,INIT_EPARS,INIT_MSGEPARS);}

  free(buffer);
  buffer=NULL;

  RETURN(status);
}

void check_and_adjust_params(bcc_status* status, io_params *f, resampling_params **r, filtering_params *w, pipeline_params *q)
{
  char infostr[BCCInfoLength];
  int t,n,nn,block_size,n_blocks;

  CHECKSTATUSPTR(status);

  ASSERT(f,status,INIT_ENULL,INIT_MSGENULL);
  ASSERT(r,status,INIT_ENULL,INIT_MSGENULL);
  ASSERT(*r,status,INIT_ENULL,INIT_MSGENULL);
  ASSERT(w,status,INIT_ENULL,INIT_MSGENULL);
  ASSERT(q,status,INIT_ENULL,INIT_MSGENULL);

  /* simplify resampling ratio */

  for (n=0; n<f->n_detectors; n++)
    {
      t=gcd((r[0]+n)->interpolate,(r[0]+n)->decimate);
      if (t>0)
	{
	  (r[0]+n)->interpolate/=t;
	  (r[0]+n)->decimate/=t;
	}
    }

  /* check detector names */
  for (n=0; n<f->n_detectors; n++)
    {
      if ( (strcmp(f->name_detectors[n], "H1")!= 0) & (strcmp(f->name_detectors[n], "H2") != 0) & (strcmp(f->name_detectors[n], "L1") != 0)  & (strcmp(f->name_detectors[n], "V1") != 0) & (strcmp(f->name_detectors[n], "G1") != 0) & (strcmp(f->name_detectors[n], "T1") != 0) )
	{
	  ABORT(status,INIT_EDETNAME,INIT_MSGEDETNAME);
	}
    }


  /* adjust size of the buffers holding data flow */

  q->overlap_length=(int) (q->segment_length*q->overlap_ratio);
  block_size=w->block_length;
  for (n=0; n<f->n_detectors; n++)
    block_size=largest_multiple(block_size,(r[0]+n)->interpolate);

  n_blocks=(int) ((r[0]->interpolate*q->max_raw_buffer_length)/((double) (r[0]->decimate*block_size)));
  for (n=1; n<f->n_detectors; n++)
    {
      nn=(int) (((r[0]+n)->interpolate*q->max_raw_buffer_length)/((double) ((r[0]+n)->decimate*block_size)));
      n_blocks = min_i(nn,n_blocks);
    }

  if (n_blocks>0)
    {
      q->rsp_buffer_length=n_blocks*block_size;
      q->proc_buffer_length=q->rsp_buffer_length+q->segment_length;
      for (n=0; n<f->n_detectors; n++)
	q->raw_buffer_length[n]=(q->rsp_buffer_length*(r[0]+n)->decimate)/(r[0]+n)->interpolate;
    }
  else
    {ABORT(status,INIT_EBUFZ,INIT_MSGEBUFZ);}

  if (q->segment_length > q->overlap_length)
    q->n_segments_per_buffer=(q->proc_buffer_length-q->overlap_length)/(q->segment_length-q->overlap_length);

  for (n=0; n<f->n_detectors; n++)
    {
      sprintf(infostr,"Buffer of raw data,       length=%d for detector %d",q->raw_buffer_length[n],n+1);
      bcc_log(status,infostr);
    }
  sprintf(infostr,"Buffer of resampled data, length=%d",q->rsp_buffer_length);
  bcc_log(status,infostr);
  sprintf(infostr,"Buffer of processed data, length=%d",q->proc_buffer_length);
  bcc_log(status,infostr);
  sprintf(infostr,"PSD estimation uses averages over %d block(s)",(int) q->rsp_buffer_length/w->block_length);
  bcc_log(status,infostr);

  RETURN(status);
}
