/* AVLDB - AVL DataBase library, transactions

   Copyright (C) 2002,2003 Petr Silhavy <silhavy@mef.cz>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.

*/

#define  __ADB 1
#include "adb.h"

static adb_tnx_id_t last_tnx_id ;

int
Adb_tnx_file_id_cmp(const void *a, const void *b, void *param UNUSED )
{
  adb_tnx_file_segs_t *fa = (adb_tnx_file_segs_t *)a , *fb = (adb_tnx_file_segs_t *)b ;
  assert( fa->magic == ADB_MAGIC_FTS );
  assert( fb->magic == ADB_MAGIC_FTS );

  return ADB_LCMP(fa->file, fb->file) ; /* ( a > b ) - ( a < b) */
}

int
Adb_tnx_offset_cmp(const void *a, const void *b, void *param UNUSED )
{
  adb_bh_t *ata = (adb_bh_t *)a , *bta = (adb_bh_t *)b ;
  return ADB_LCMP(ata->offset, bta->offset) ; /* ( a > b ) - ( a < b) */
}

static int
Adb_tnx_sb_cmp(const void *a, const void *b, void *param UNUSED )
{
  const adb_tnx_xip_t *ap = a , *bp = b ;
  int r = (ap->sb > bp->sb ) - (ap->sb < bp->sb );

  fprintf(stderr,"%s(%p,%p) = %d\n", __func__, ap->sb, bp->sb, r );
  return r ;  
}


/* Interface */
adb_tnx_t *
adb_tnx_new(/* adb_tnx_state_t state */ ) /* state NULL/HEAVY */
{
  adb_tnx_t *tnx = Malloc(sizeof(*tnx));

/*   tnx->state = state ; */

  tnx->id = ++last_tnx_id ;
  tnx->magic = ADB_MAGIC_TNX ;

  tnx->files = Malloc(sizeof (*tnx->files));
  tavl_create( tnx->files );

  tnx->xips = Malloc(sizeof(*tnx->xips));
  tavl_create( tnx->xips );

  return tnx ;
}

adb_tnx_file_segs_t *
Adb_find_file_in_tnx(adb_tnx_t *tnx, adb_file_t *file)
{
  adb_tnx_file_segs_t tmp ;
  tmp.magic = ADB_MAGIC_FTS ;
  tmp.file = file ;
  return tavl_find( tnx->files, Adb_tnx_file_id_cmp, &tmp );
}


adb_bh_t *
Adb_tnx_add(adb_tnx_t *tnx, adb_file_t *file, adb_tnx_state_t state, unsigned char *addr , size_t size, off_t offset)
{
  adb_bh_t *new = Malloc(sizeof(*new));
  void *ret ;
  adb_tnx_file_segs_t *fts ;

  new->magic = ADB_MAGIC_BH ;
  new->size = size ;
  new->addr = (unsigned int)addr ;
  new->file = file ;
/*   new->malloc = adb_meta_malloc0 ; */
/*   new->free = adb_meta_free ; */
  new->offset = offset ;

  if ( (fts = Adb_find_file_in_tnx(tnx, file)) == NULL ) /* no file now */
    {
      fts = Malloc(sizeof(*fts));
      fts->file = file ;
      fts->magic = ADB_MAGIC_FTS ;
      fts->atoms = Malloc(sizeof(*fts->atoms));
      tavl_create(fts->atoms);
      tavl_probe( tnx->files, Adb_tnx_file_id_cmp, &fts->node, fts );
    }

  ret = tavl_insert( fts->atoms, Adb_tnx_offset_cmp, &new->node, new ); 

  if ( ret )
    {
      adb_bh_t *ta ;
      if ( ( ta = Adb_bh_find(tnx, file, offset ) ) != NULL )
	{
	  ADB_ERROR("Allocated new BH for file \"%s\" @%Ld, but found in TNX\n",
		    file->meta->name, offset );
	  free(new);
	  return ta ;
	}
      else
	{
	  ADB_ERROR("Allocated new BH for file \"%s\", *** tavl_insert FAILED ***\n",
		    file->meta->name);
	  abort();
	}
    }

  return new ;
}

void
Adb_tnx_clean(adb_tnx_t *tnx, adb_file_t *file)
{
  /* cleanup - avoid swap trashing */
  adb_tnx_file_segs_t *fts ;
  int level , i ;
  tavl_traverser_t tr ;
  adb_bh_t *ta ;
  static int c ;

  if ( ( ++c % 64 ))
    return ;
  
  if ( (fts = Adb_find_file_in_tnx(tnx, file)) == NULL ) /* no file now */
    return ;
  
  if ( fts->atoms->tavl_count > Adb_file_blocks_max )
    {
      for ( level = 0 ; fts->atoms->tavl_count > Adb_file_blocks_max ; ++level )
	{
	  for ( i = 0 , ta = tavl_t_first(&tr, fts->atoms) ; 
		(ta = tavl_t_next(&tr));
		)
	    {
	      if ( ta->m->level == level )
		{
		  Adb_seg_unload(tnx, ta, TRUE);
		  ++i ;
		}
	    }
	  fprintf(stderr,"--- Unloaded %d blocks at level %d ---\n", i, level);
	}
    }
}
adb_bh_t *
Adb_tnx_find_parent_bh(adb_tnx_t *tnx,  adb_bh_t *child, adb_file_t *file)
{
  adb_tnx_file_segs_t *fts ;
  adb_bh_t *parent , tmp ;

  if ( child->m->parent )
    {
      tmp.offset = child->m->parent ;

      if ( (fts = Adb_find_file_in_tnx(tnx, file)) == NULL ) /* no file now */
	{
	  fts = Malloc(sizeof(*fts));
	  fts->file = file ;
	  fts->atoms = Malloc(sizeof(*fts->atoms));
	  fts->magic = ADB_MAGIC_FTS ;
	  tavl_create(fts->atoms);
	  tavl_probe( tnx->files, Adb_tnx_file_id_cmp, &fts->node, fts );
	}

      parent = tavl_find( fts->atoms,  Adb_tnx_offset_cmp, &tmp );

      if ( ! parent ) /* not mapped now */
	parent = Adb_seg_load(tnx, file, child->m->parent, ADB_TNX_NULL );

      assert( parent->offset == child->m->parent );
      return parent ;
    }
  return NULL ;
}


static void 
Adb_tnx_move_ai(adb_db_t *db, int ai /* fd */)
{
  adb_tnx_hdr_t hdr ;
  int r ;
  off_t here ;
  here = lseek(ai, 0, SEEK_SET);
  assert(!here);

  Adb_lock_claim_sb_with(db, ADB__LOCK_AI);

  Adb_meta_rw(db);

  for ( ; ; )
    {
      off_t hdr_off ;
      
      r = read(ai, &hdr, sizeof(hdr));
      if ( r != sizeof(hdr))
	break ;
      assert(hdr.magic == ADB_MAGIC_TNX_AI);
      
      if ( hdr.type == ADB_TNX_TYPE_BLOCK )
	{
	  r = read(ai, zbuf, hdr.size);
	  if ( r != hdr.size )
	    break ;
	}

      /* validation */
      switch ( hdr.type )
	{
	case ADB_TNX_TYPE_BLOCK:	  
	case ADB_TNX_TYPE_FILE_TYPE:
	case ADB_TNX_TYPE_FILE_START:
	case ADB_TNX_TYPE_BLOCK_ADD:
	case ADB_TNX_TYPE_BLOCK_DEL:
	  break ;
	default:
	  ADB_WARNING("Unexpected type %d\n",hdr.type);
	  ADB_BUG("AI meltdown");
	  break ;
	}

      hdr_off = lseek(ai, 0, SEEK_CUR);
      r = write(db->ai_fd, &hdr, sizeof(hdr) );
      assert( r == sizeof(hdr));
      r = write(db->ai_fd, zbuf, hdr.size);
      assert( r == hdr.size );
      fprintf(stderr,"%s: type: %d AI: #%lld-%lld \n", __func__, 
	      hdr.type, hdr_off, lseek(ai, 0, SEEK_CUR) );
    }

  Adb_lock_free_sb_with(db, ADB__LOCK_AI);
  Adb_meta_ro(db);

}

void
adb_tnx_apply_ai(adb_db_t *db, char *ai_file )
{
  int ai , r ;
  long long here ;
  adb_tnx_hdr_t hdr ;
  int fd ;

  if ( ai_file == NULL )
    {
      if ( ( ai = open(db->sb->ai_file, O_RDWR )) == EOF )
	ADB_ERROR("Can't open AI file `%s'\n", db->sb->ai_file);
    }
  else
    {
      if ( ( ai = open(ai_file, O_RDWR )) == EOF )
	ADB_ERROR("Can't open AI file `%s'\n", ai_file);
    }
  
  
  here = lseek(ai, 0, SEEK_SET);
  
  Adb_meta_rw(db);

  while (1)
    {
      r = read(ai, &hdr, sizeof(hdr));
      
      if ( r != sizeof(hdr))
	break ;
      
      assert(hdr.magic == ADB_MAGIC_TNX_AI);      

      fprintf(stderr,"%s: AI-HDR type: %d orig size: %d offset: %lld\n", 
	      __func__, hdr.type, hdr.orig_size, hdr.offset );

      switch ( hdr.type )
	{
	case ADB_TNX_TYPE_BLOCK:
	case ADB_TNX_TYPE_BLOCK_META:
	  {
	    void *block /* , *zblock */ ;
	    unsigned long dest_len ;
	    struct stat sb ;

	    fd = db->fds[ hdr.nfd ] ;
      
	    fprintf(stderr,"%s: DB-BLOCK size: %d nfd: %d fd: %d\n", __func__,
		    hdr.orig_size, hdr.nfd, fd);
	    
	    r = fstat(fd, &sb); assert( !r);
	    if ( sb.st_size < hdr.orig_size + hdr.offset )
	      {
		r = ftruncate(fd, hdr.orig_size + hdr.offset); assert( ! r);
		fprintf(stderr, "%s: DB resize %lld => %lld\n", __func__, 
			sb.st_size, hdr.orig_size + hdr.offset );
	      }
	    
	    block = mmap(NULL, hdr.orig_size, PROT_READ|PROT_WRITE,
			 MAP_SHARED, fd, hdr.offset);
	    if ( block == (void *)EOF)
	      ADB_BUG("mmap fail");
	    
	    fprintf(stderr,"%s: AI-BLOCK size: %d nfd: %d fd: %d #%lld\n", __func__,
		    hdr.size, hdr.nfd, fd, hdr.offset );
	  
	    r = read(ai, zbuf, hdr.size);
	    assert(r == hdr.size);

	    dest_len = hdr.orig_size ;
	    r = uncompress ((Bytef *)block, &dest_len, (Bytef *)zbuf, hdr.size);

	    assert(r == Z_OK);
	    assert(dest_len == hdr.orig_size);

	    r = msync(block, hdr.orig_size, MS_SYNC);
	    assert(!r);

	    munmap(block, hdr.orig_size);
/* 	    munmap(zblock, hdr.size); */
	  }
	  break ;
	case ADB_TNX_TYPE_FILE_TYPE:
	  {
	    adb_meta_file_t *meta = ( void *)hdr.__meta + db->addr ;
	    meta->data_start_type = hdr.offset ;
	  }
	  break ;
	case ADB_TNX_TYPE_FILE_START:
	  {
	    adb_meta_file_t *meta = ( void *)hdr.__meta + db->addr ;
	    Adb_meta_rw(db);
	    meta->data_start = hdr.offset ;
	  }
	  break ;
	case ADB_TNX_TYPE_BLOCK_ADD:
	  fd = db->fds[ hdr.nfd ] ;
	  Adb_lock_claim_sb_with(db, ADB__LOCK_EXTEND);

	  r = lseek( fd, hdr.orig_size, SEEK_END);  assert( r != EOF);

	  Adb_lock_free_sb_with(db, ADB__LOCK_EXTEND);
	  break ;
	case ADB_TNX_TYPE_BLOCK_DEL:
	  break ;
	  /*
	   * Block redo can be confused by this stuff !!!!!
	   */
#if 0
	  {
	    adb_rec_t *rec ;
	    rec = adb_insert_by_name(tnx, db->free_list,
			      "offset", hdr.offset,
			      "nfd", hdr.nfd,
			      "size", hdr.orig_size,
			      NULL );
	    assert(rec);
	  }
	  break ;
#endif /* 0 */
	default:
	  ADB_WARNING("Unexpected AI header type %d\n", hdr.type);
	  ADB_BUG("Anarchy in the AI");
	  break ;
	}
    }

  Adb_ai_stop(db);
  Adb_bi_stop(db);
  Adb_meta_ro(db);
}

static void 
Adb_tnx_apply_bi(adb_db_t *db, adb_tnx_t *tnx /* for free-list */, int bi /* fd */)
{
  int tmp ;
  char *tmp_bi_file = NULL ;
  off_t off , end ;
  adb_tnx_hdr_t hdr ;
  int r ;
  off_t here ;
  here = lseek(bi, 0, SEEK_CUR);
  
  Adb_meta_rw(db);

  asprintf(&tmp_bi_file, "%s/%d.tmp-bi", db->sb->bi_dir,/*  db->name, */  getpid() );
  if ( ( tmp = open(tmp_bi_file, O_CREAT|O_RDWR|O_EXCL|O_TRUNC, 0600)) == EOF )
    ADB_ERROR("Can't open %s\n",tmp_bi_file);
  /* read forward */

  lseek(bi, 0, SEEK_SET);
  for ( ; ; )
    {
      off_t hdr_off ;
      
      hdr_off = lseek(bi, 0, SEEK_CUR);

      r = read(bi, &hdr, sizeof(hdr));
      if ( r != sizeof(hdr))
	break ;
      assert(hdr.magic == ADB_MAGIC_TNX_BI);
      
      if ( hdr.type == ADB_TNX_TYPE_BLOCK )
	{
	  r = read(bi, zbuf, hdr.size);
	  if ( r != hdr.size )
	    break ;
	}

      /* validation */
      switch ( hdr.type )
	{
	case ADB_TNX_TYPE_BLOCK:	  
	case ADB_TNX_TYPE_FILE_TYPE:
	case ADB_TNX_TYPE_FILE_START:
	case ADB_TNX_TYPE_BLOCK_ADD:
	case ADB_TNX_TYPE_BLOCK_DEL:
	  break ;
	default:
	  ADB_WARNING("Unexpected type %d\n",hdr.type);
	  ADB_BUG("BI meltdown");
	  break ;
	}
      r = write(tmp, &hdr_off, sizeof(hdr_off));
      assert( r == sizeof(hdr_off));
      
      fprintf(stderr,"%s-test: type: %d BI: #%lld-%lld tmp: #%lld\n", __func__, 
	      hdr.type, hdr_off, lseek(bi, 0, SEEK_CUR) ,lseek(tmp, 0, SEEK_CUR));
    }

  end = lseek(tmp, 0, SEEK_END);
  
  assert(! (end % sizeof(end)));

  while ( end )
    {
      end -= sizeof(end);

      fprintf(stderr,"%s: tmp offset %lld\n", __func__, end);
      pread(tmp, &off, sizeof(off), end);
      fprintf(stderr,"%s: BI offset %lld\n", __func__, off);
      pread(bi, &hdr, sizeof(hdr), off);

      assert(hdr.magic == ADB_MAGIC_TNX_BI);      

      switch ( hdr.type )
	{
	case ADB_TNX_TYPE_BLOCK:
	  {
	    void *block /* , *zblock */ ;
	    int fd ;
	    unsigned long dest_len ;

	    fd = db->fds[ hdr.nfd ] ;
	    
	    fprintf(stderr,"%s: DB-BLOCK size: %d nfd: %d fd: %d\n", __func__,
		    hdr.orig_size, hdr.nfd, fd);
	    
	    block = mmap(NULL, hdr.orig_size, PROT_READ|PROT_WRITE,
			 MAP_SHARED, fd, hdr.offset);
	    if ( block == (void *)EOF)
	      ADB_BUG("mmap fail");
	    
	    fprintf(stderr,"%s: BI-BLOCK size: %d nfd: %d fd: %d #%lld\n", __func__,
		    hdr.size, hdr.nfd, fd, off + sizeof(hdr));
	  
/* 	    zblock = mmap(NULL, hdr.size, PROT_READ, MAP_PRIVATE, bi, off + sizeof(hdr)); */
/* 	    if ( zblock == (void *)EOF) */
/* 	      ADB_BUG("mmap fail"); */
	    r = pread(bi, zbuf, hdr.size, off + sizeof(hdr));
	    assert(r == hdr.size);

	    dest_len = hdr.orig_size ;
	    r = uncompress ((Bytef *)block, &dest_len, (Bytef *)zbuf, hdr.size);

	    assert(r == Z_OK);
	    assert(dest_len == hdr.orig_size);

	    r = msync(block, hdr.orig_size, MS_SYNC);
	    assert(!r);

	    munmap(block, hdr.orig_size);
/* 	    munmap(zblock, hdr.size); */
	  }
	  break ;
	case ADB_TNX_TYPE_FILE_TYPE:
	  {
	    adb_meta_file_t *meta = ( void *)hdr.__meta + db->addr ;
	    meta->data_start_type = hdr.offset ;
	  }
	  break ;
	case ADB_TNX_TYPE_FILE_START:
	  {
	    adb_meta_file_t *meta = ( void *)hdr.__meta + db->addr ;
	    Adb_meta_rw(db);
	    meta->data_start = hdr.offset ;
	  }
	  break ;
	case ADB_TNX_TYPE_BLOCK_ADD:
	  {
	    adb_rec_t *rec ;
	    rec = adb_insert_by_name(tnx, db->free_list,
			      "offset", hdr.offset,
			      "nfd", hdr.nfd,
			      "size", hdr.orig_size,
			      NULL );
	    assert(rec);
	  }
	  break ;
	case ADB_TNX_TYPE_BLOCK_DEL:
	  break ;
	default:
	  ADB_WARNING("Unexpected BI header type %d\n", hdr.type);
	  ADB_BUG("Anarchy in the BI");
	  break ;
	}
    }
  Adb_meta_ro(db);

  close(tmp);
  unlink(tmp_bi_file);
}




typedef enum
  {
    ADB_TNX_END_DROP = 0,
    ADB_TNX_END_SYNC = 1,
  } _adb_tnx_end_t ;

void
Adb_tnx_end(adb_tnx_t **_tnx, _adb_tnx_end_t sync )
{
  tavl_traverser_t tr ;
  adb_tnx_file_segs_t *fts ;
  adb_tnx_xip_t *xip ; 
  adb_tnx_t *tnx = *_tnx ;

  assert( tnx->magic == ADB_MAGIC_TNX );

  tavl_t_init( &tr, tnx->files);

  if ( sync )
    puts("Sync: ");
  else
    puts("Drop: ");

  while ( ( fts = tavl_t_next( &tr )) != NULL )
    {
      tavl_traverser_t atr ;
      adb_bh_t *bh ;

      tavl_t_init( &atr, fts->atoms);
      while ( ( bh = tavl_t_next( &atr )) != NULL )
	{
	  Adb_seg_unload(tnx, bh, sync);
	}
      fts->file->xip = NULL ; 
    }
  puts("...Done\n");

  if ( ! sync ) /* Abort */
    {
      struct tavl_traverser tr ;
      adb_tnx_t *undo = adb_tnx_new();
      
      puts("Undo: ");
      for ( xip = tavl_t_first(&tr, tnx->xips ) ; 
	    xip ;
	    xip = tavl_t_next(&tr) )
	{
	  Adb_tnx_apply_bi(xip->db, undo /* for free-list */, xip->bi_fd);
	  close(xip->bi_fd);
	  unlink(xip->bi_name);
	  free(xip->bi_name);
	}
      adb_tnx_end(&undo); /* commit free list */
    }

  puts("BI close : ");
  /* destruction of bips tree */
  for ( xip = tavl_t_first(&tr, tnx->xips ) ; 
	xip ; 
	xip = tavl_t_first(&tr, tnx->xips ) )
    {
      adb_tnx_xip_t *b ;
      b = tavl_delete(tnx->xips, Adb_tnx_sb_cmp, xip );
      assert(b);

      close(b->bi_fd);
      unlink(b->bi_name);

      if ( b->ai_name )       /* move AI */
	{
	  assert(b->ai_fd != EOF);

	  if ( sync )
	    Adb_tnx_move_ai(b->db, b->ai_fd);

	  close(b->ai_fd);
	  unlink(b->ai_name);
	}
      free(b);
    }

  free( tnx->files );
  free( tnx->xips );

  free ( *_tnx );

  *_tnx = NULL ;

  if ( ! sync )
    puts("...Done\n");
}

/* Interface */

void
adb_tnx_end(adb_tnx_t **tnx)
{
  Adb_tnx_end(tnx, ADB_TNX_END_SYNC );
}

void
Adb_tnx_check(adb_tnx_t *tnx)
{
  tavl_traverser_t atoms_tr , fts_tr ;
  adb_tnx_file_segs_t *fts ;
  adb_bh_t *ta ;

  for ( fts = tavl_t_first(&fts_tr, tnx->files ) ; 
	fts ; 
	fts = tavl_t_next(&fts_tr))
    {
      assert( fts->magic == ADB_MAGIC_FTS );
      for ( ta = tavl_t_first( &atoms_tr, fts->atoms) ;
	    ta ;
	    ta = tavl_t_next(&atoms_tr))
	{
	  assert(ta->magic == ADB_MAGIC_BH );
	}
    }
  
}

void
Adb_tnx_bi_open(adb_tnx_t *tnx, adb_file_t *file)
{
  adb_tnx_xip_t b = { .magic = ADB_MAGIC_TNX_BIP, .sb = file->db->sb } , *xip ;

  void *ret ;

  if ( ( xip = tavl_find(tnx->xips, Adb_tnx_sb_cmp, &b )) != NULL )
    {
      file->xip = xip ;
      return ;
    }
  xip = Malloc(sizeof(*xip));
  xip->magic = ADB_MAGIC_TNX_BIP ;
  xip->sb = file->db->sb ;
  xip->db = file->db ;
  xip->bi_name = NULL ;
  xip->ai_name = NULL ;

/*   asprintf(&bip->bi_name, "%s/%s-%d.bi", file->db->sb->bi_dir, file->db->name, getpid()); */
  asprintf(&xip->bi_name, "%s/%d-%d.bi", file->db->sb->bi_dir, getpid(), tnx->id);


  fprintf(stderr,"%s BI file for tnx %p %d : %s\n", __func__, tnx, tnx->id,
	  xip->bi_name);
  
  if ( ( xip->bi_fd = open(xip->bi_name, O_CREAT|O_TRUNC|O_RDWR|O_EXCL, 0600)) == EOF )
    ADB_ERROR("Can't open bi : %s\n", xip->bi_name);

  if ( file->db->sb->mode & ADB_MODE_AI )
    {
      asprintf(&xip->ai_name, "%s/%d-%d.ai", file->db->sb->ai_dir, getpid(), tnx->id);
      fprintf(stderr,"%s AI tnx-file for tnx %p %d : %s\n", __func__, tnx, tnx->id,
	  xip->ai_name);

      if ((xip->ai_fd = open(xip->ai_name, O_CREAT|O_TRUNC|O_RDWR|O_EXCL, 0600)) == EOF)
	ADB_ERROR("Can't open tnx-ai : %s\n", xip->ai_name);
    }
  else
    {
      xip->ai_name = NULL ;
      xip->ai_fd = EOF ;
    }

  ret = tavl_insert(tnx->xips, Adb_tnx_sb_cmp, &xip->node, xip );
  assert(!ret);

  
  file->xip = xip ;
}

void
Adb_tnx_set_data_start(adb_file_t *file, off_t *start, off_t new_val)
{
  /* AI/BI HERE !!! */
/*   if ( file->pseudo ) */
/*     { */
/*       *start = new_val ; */
/*       return ; */
/*     } */

  if ( *start != new_val )
    {
      Adb_seg_bi_type(ADB_TNX_TYPE_FILE_START, file, 0LL);

      Adb_meta_rw(file->db);

      *start = new_val ;

      Adb_meta_ro(file->db);

      Adb_seg_ai_type(ADB_TNX_TYPE_FILE_START, file, 0LL);
    }
}

void
Adb_tnx_set_data_start_type(adb_file_t *file, adb_magic_t *type, 
			      adb_magic_t new_type)
{
  /* AI/BI HERE !!! */

/*   if ( file->pseudo ) */
/*     { */
/*       *type = new_type ; */
/*       return ; */
/*     } */

  if ( *type != new_type )
    {
      Adb_seg_bi_type(ADB_TNX_TYPE_FILE_TYPE, file, 0LL);
      Adb_meta_rw(file->db);
      
      *type = new_type ;
      
      Adb_meta_ro(file->db);
		   
      /* AI HERE !!! */
      Adb_seg_ai_type(ADB_TNX_TYPE_FILE_TYPE, file, 0LL);
    }
}



void
adb_tnx_abort(adb_tnx_t **tnx)
{
  Adb_tnx_end(tnx, ADB_TNX_END_DROP);
}

void
Adb_tnx_mass_undo(adb_db_t *db)
{
  int i ;
  glob_t result ;
  char *pattern = NULL ;
  int fd ;
  char *bi = NULL ; /* to make gcc happy */
  adb_tnx_t *undo ;
  int ret ;

  asprintf(&pattern, "%s/*-*.bi", db->sb->bi_dir );
  if ( ( ret = glob ( pattern, 0, NULL, &result )))
    {
      ADB_WARNING("glob failed retval %d (pattern `%s'), no stale BIs ?", ret, pattern);
      return ;
    }

  /* test run */
  for ( i = 0 ; i < result.gl_pathc ; ++i )
    {
      bi = result.gl_pathv[i] ;
      if ( access(bi, R_OK ))
	ADB_ERROR("Can't read `%s', NO database cleanup",bi);
    }

  undo = adb_tnx_new();

  for ( i = 0 ; i < result.gl_pathc ; ++i )
    {
      if ( ( fd = open(bi, O_RDONLY)) == EOF )
	ADB_ERROR("Can't open `%s' (how I got here ?)",bi);
      fprintf(stderr,"Undoing BI file %s\n",bi);

      Adb_tnx_apply_bi(db, undo /* for free-list */, fd);      
      close(fd);
    }
   globfree(&result);
   
   adb_tnx_end(&undo);
}
