/* AVLDB - AVL DataBase library, intervals of trees AKA ranges

   Copyright (C) 2002,2003 Petr Silhavy <silhavy@mef.cz>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.

*/

#define  __ADB 1
#include "adb.h"

/* adb_tex */
#define TEX_DEBUG 0

adb_rec_t *
adb_range_to_rec(adb_range_t *range)
{
  adb_rec_t *rec = (adb_rec_t *)( range + 1 ) ;
  assert( range->magic == ADB_MAGIC_RANGE );
  /*  assert( rec->magic == ADB_MAGIC_RECORD ); memcpy!!!! */
  return rec ;
}

adb_rec_t *Last ;

void
Adb__rec_range_show(adb_bh_t *bh, void *data)
{
  adb_rec_t *rec = data ;

  switch ( rec->magic )
    {
    case ADB_MAGIC_RECORD:
      Adb_data_show(rec, bh->file );
      break ;
    case ADB_MAGIC_RANGE:
      Adb_show_one_range( (adb_range_t *) data, bh, bh->file );
      break ;
    default:
      assert("Bug" == 0 );
      break ;
    }
}

int
Adb__cmp_string(const void *ap , const void *bp)
{
  const char *a = ap ;
  const char *b = bp ;
  
  return strcmp(a,b);
}

int
Adb__match_string(const void *ap , const void *lp, const void *rp)
{
  const char *a = ap ;
  const char *l = lp ;
  const char *r = rp ;
  int al = strcmp(a, l );
  int ar = strcmp(a, r );
  if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
    fprintf(stderr, "%s: al = %d ar = %d, a = %s, l = %s, r = %s\n%s: return %d\n", 
	    __func__, al, ar, a, l, r, __func__ , al > 0 && ar < 0);
  return ( al > 0 && ar < 0 ) ;
}

int
Adb__cmp_vstring(const void *ap, const void *bp)
{
  const adb_vstring_t *avs = ap , *bvs = bp ;
  const char *a = (void *)avs + avs->offset ;
  const char *b = (void *)bvs + bvs->offset ;

  return strcmp(a,b);
}

int
Adb__match_vstring(const void *ap , const void *lp, const void *rp)
{
  const adb_vstring_t *avs = ap , *lvs = lp, *rvs = rp ;
  const char *a = (void *)avs + avs->offset ;
  const char *l = (void *)lvs + lvs->offset ;
  const char *r = (void *)rvs + rvs->offset ;

  int al = strcmp(a, l );
  int ar = strcmp(a, r );

  assert( avs->magic == ADB_MAGIC_VSTRING );
  assert( lvs->magic == ADB_MAGIC_VSTRING );
  assert( rvs->magic == ADB_MAGIC_VSTRING );

  if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
    fprintf(stderr, "%s: al = %d ar = %d, a = %s, l = %s, r = %s\n%s: return %d\n", 
	    __func__, al, ar, a, l, r, __func__ , al > 0 && ar < 0);

  return ( al > 0 && ar < 0 ) ;
}

int
Adb__cmp_wchar(const void *ap, const void *bp)
{
  const adb_wchar_t *awc = ap , *bwc = bp ;
  const wchar_t *a = (void *)awc + awc->offset ;
  const wchar_t *b = (void *)bwc + bwc->offset ;

  return wcscoll(a,b);
}

int
Adb__match_wchar(const void *ap , const void *lp, const void *rp)
{
  const adb_wchar_t *awc = ap , *lwc = lp, *rwc = rp ;
  const wchar_t *a = (void *)awc + awc->offset ;
  const wchar_t *l = (void *)lwc + lwc->offset ;
  const wchar_t *r = (void *)rwc + rwc->offset ;

  int al = wcscoll(a, l );
  int ar = wcscoll(a, r );

  if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
    fprintf(stderr, "%s: al = %d ar = %d, a = %ls, l = %ls, r = %ls\n%s: return %d\n", 
	    __func__, al, ar, a, l, r, __func__ , al > 0 && ar < 0);

  return ( al > 0 && ar < 0 ) ;
}


#include "cmp.h"

/* CMP_TYPE(unsigned int, uint) */
/* CMP_TYPE(int, int ) */
/* CMP_TYPE(long long, ll) */
/* CMP_TYPE(unsigned long long, ull) */
/* CMP_TYPE(long double, ld) */
/* CMP_TYPE(ptrdiff_t, ptr) */

     
int (*Adb__cmp[])(const void *, const void *) = 
{ 
  [ADB_FIELD_INT] = Adb__cmp_int, 
  [ADB_FIELD_LONG_LONG] = Adb__cmp_ll,
  [ADB_FIELD_DOUBLE] = Adb__cmp_ld,
  [ADB_FIELD_UNSIGNED_INT] = Adb__cmp_uint,
  [ADB_FIELD_UNSIGNED_LONG_LONG] = Adb__cmp_ull,
  [ADB_FIELD_PTR_DIFF] = Adb__cmp_ptr,
  [ADB_FIELD_STRING] = Adb__cmp_string,
  [ADB_FIELD_VSTRING] = Adb__cmp_vstring,
  [ADB_FIELD_WCHAR] = Adb__cmp_wchar
} ;

int (*Adb__match[])(const void *, const void *, const void *) = 
{ 
  [ADB_FIELD_INT] = Adb__match_int, 
  [ADB_FIELD_LONG_LONG] = Adb__match_ll,
  [ADB_FIELD_DOUBLE] = Adb__match_ld,
  [ADB_FIELD_UNSIGNED_INT] = Adb__match_uint,
  [ADB_FIELD_UNSIGNED_LONG_LONG] = Adb__match_ull,
  [ADB_FIELD_PTR_DIFF] = Adb__match_ptr,
  [ADB_FIELD_STRING] = Adb__match_string,
  [ADB_FIELD_VSTRING] = Adb__match_vstring,
  [ADB_FIELD_WCHAR] = Adb__match_wchar,
} ;

/*      The return value is expected to be like that returned by `strcmp' */
/*      in the standard C library: negative if A < B, zero if A = B, */
/*      positive if A > B.  PARAM is an arbitrary value defined by the */
/*      user when the tree was created. */

int
Adb_range_cmp(const void *_a, const void *_b, const void *_param, int uuu UNUSED)
{
  adb_rec_t *rec_a = ( adb_rec_t *) _a ;
/*   adb_rec_t *rec_b = ( adb_rec_t *) _b ; */
  adb_bh_t *bh = ( adb_bh_t *) _param ;
  adb_file_t *file = ( adb_file_t *) bh->file ;

  adb_field_t *fld ;
  int i ;
  int ret = 0 ;
  /* range vs range */
  adb_range_t *a_range = (adb_range_t *) _a ;
  adb_range_t *b_range = (adb_range_t *) _b ;
  const void *A, *L, *R ;
  adb_tavl_traverser_t tr ;

  assert( file->magic == ADB_MAGIC_FILE );

  /************************************************************************/
  /*                     Record vs Interval AKA Range                     */
  /************************************************************************/

  if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
    fprintf(stderr, "%s: ========= level %d ==============\n",
	    __func__ , bh->m->level);

  if ( rec_a->magic == ADB_MAGIC_RECORD ) /* record vs range */
    {
      adb_range_t *range = (adb_range_t *) _b ;
      adb_range_t *right , *left ;
      int right_eq = 0 , left_eq = 0 ;

      assert( range->magic == ADB_MAGIC_RANGE );

      if ( range->flag & ADB_RANGE_LEFT_INF )
	{
	  if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
	    {
	      fprintf(stderr,"%s: Rec vs : ", __func__); Adb__rec_range_show(bh, rec_a );
	      fprintf(stderr,"%s: Range flag: -INF\n", __func__ );
	    }
	  if ( range->node.tavl_tag[1] == TAVL_CHILD ) /* has right child */
	    {
	      tr.tavl_node = &range->node ;
	      tr.tavl_table = &bh->m->tree ;

	      right = adb_tavl_t_next( bh, &tr );

	      assert( right->magic == ADB_MAGIC_RANGE );
	      /* -INF range can't have left child ( until tree is damaged ) */
	      assert( range->node.tavl_tag[0] != TAVL_CHILD );

	      if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
		fprintf(stderr,"%s: Strange: -INF has right child\n", __func__ );

	      ret = Adb_data_cmp( rec_a, adb_range_to_rec(right), bh , 0);
	      if ( ret > 0 ) 
		{
		  if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
		    fprintf(stderr,"%s: Strange: Move to -INF's right child, ret = %d\n",
			    __func__, ret);
		  return ret ;
		}
	      else
		{
		  if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
		    fprintf(stderr,
			    "%s: Strange: -INF has right child, %d = __data_cmp\n",
			    __func__, ret);
		  return 0 ;
		}
	    }
	  else
	    return 0 ; /* if THIS range was checked record belongs HERE - ranges are
		      * ( as I hope ) balanced too
		      */
	}
      if ( range->flag & ADB_RANGE_RIGHT_INF )
	{
	  if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
	    {
	      fprintf(stderr,"%s: Rec vs : ",__func__); Adb__rec_range_show(bh, rec_a );
	      fprintf(stderr,"%s: Range flag: +INF\n",__func__);
	    }
	  if ( range->node.tavl_tag[0] == TAVL_CHILD ) /* has left child */
	    {
	       tr.tavl_node = &range->node ;
	       tr.tavl_table = &bh->m->tree ;

	       left = adb_tavl_t_prev( bh, &tr );

	      /* +INF range can't have right child ( as I hope :) */
	      assert( range->node.tavl_tag[1] != TAVL_CHILD );
	      if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
		fprintf(stderr, "%s: Strange: +INF has left child\n",__func__);

	      ret = Adb_data_cmp( rec_a, adb_range_to_rec(range), bh , 0); 
	      /* range , not left                       ^^^^^ 
	       * record belongs to left child if is lesser than right edge 
	       * => range 
	       */
	      if ( ret < 0 ) 
		{
		  if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
		    fprintf(stderr,"%s: Strange: Move to +INF's left child, ret = %d\n",
			    __func__, ret);

		  return ret ;
		}
	      else
		{
		  if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
		    fprintf(stderr,"%s: Strange: +INF has left child, %d = __data_cmp\n",
			    __func__, ret);

		  return 0 ;
		}
	    }
	  else
	    return 0 ; /* if THIS range was checked record belongs HERE - ranges are
			* ( as I hope ) balanced too
			*/
	}

      tr.tavl_node = &range->node ;
      tr.tavl_table = &bh->m->tree ;

      right = adb_tavl_t_next( bh, &tr );
      if ( Adb_debug & ADB_DEBUG_RANGE_CMP )
	{
	  fprintf(stderr, "%s: Right found: ",__func__);
	  Adb__rec_range_show(bh, adb_range_to_rec(right) );
	}
      /* right should exist - no flag seen */
      /* so : NO BUGWARE  */
      /*       if ( ! right ) */
      /* 	return 0 ; */

      assert( right->magic == ADB_MAGIC_RANGE );

      if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
	{
	  fprintf(stderr, "%s: Rec A : ", __func__); Adb__rec_range_show(bh, rec_a);
	  fprintf(stderr, "%s: Range L: ", __func__ ); 
	  Adb__rec_range_show(bh, adb_range_to_rec(range) );
	  fprintf(stderr, "%s: Range R: ", __func__);
	  Adb__rec_range_show(bh, adb_range_to_rec(right));
	}
      /*      The return value is expected to be like that returned by `strcmp' */
      /*      in the standard C library: negative if A < B, zero if A = B, */
      /*      positive if A > B.  PARAM is an arbitrary value defined by the */
      /*      user when the tree was created. */

      for ( i = 0 ; i < file->meta->idx_size ; ++i )
	{
	  fld = (void *)file->idx[i] + file->db->addr ;
	  if ( fld && fld->magic == ADB_MAGIC_FIELD ) /* avoid "holes" in idx */
	    {
	      A = _a  + fld->offset ;
	      L = (void *) (adb_range_to_rec(range)) + fld->offset ;
	      R = (void *) (adb_range_to_rec(right)) + fld->offset ;

	      assert( ! (right_eq && left_eq) );

	      switch (  fld->type )
		{
		case ADB_FIELD_UNSIGNED_INT:
		case ADB_FIELD_INT:
		case ADB_FIELD_PTR_DIFF:
		case ADB_FIELD_LONG_LONG:
		case ADB_FIELD_UNSIGNED_LONG_LONG:
		case ADB_FIELD_DOUBLE:
		case ADB_FIELD_STRING:
		case ADB_FIELD_VSTRING:
		case ADB_FIELD_WCHAR:
		  {
		    int (*cmp)(const void *, const void *) = Adb__cmp[fld->type] ;
		    /* 	int (*match)(const void *, const void *, const void *) =  */
		    /* 			  Adb__match[fld->type] ; */

			
		    if (  Adb_debug & ADB_DEBUG_RANGE_CMP )  
		      fprintf(stderr, "%s: comparison by field %s\n", 
			      __func__,fld->name);

		    if ( ! right_eq && ! left_eq )
		      {
			switch ( cmp(A, L) )
			  {
			  case -1: 
			    if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
			      fprintf(stderr, "%s: Result: LT field %s\n", 
				      __func__,fld->name);
			    return -1 ; break ;
			  case 1:
			    switch ( cmp(A, R))
			      {
			      case 1:
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr, "%s: Result: GT field %s\n", 
					  __func__,fld->name);
				return 1 ; break ;
			      case 0:
				++right_eq ;
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr, 
					  "%s: Result: CONTINUE-right-EQ field %s\n",
					  __func__,fld->name);
				continue ; break ;
			      case -1:
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr, "%s: Result: MATCH field %s\n", 
					  __func__,fld->name);
				return 0 ;	break ;
			      default: 
				abort();
				break ;
			      }
			    break ;
			  case 0: /* A == L */
			    switch ( cmp(A, R))
			      {
			      case 0:
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr, 
					  "%s: Result: CONTINUE All-EQ field %s\n", 
					  __func__,fld->name);
				continue ; break ;
			      case -1:
				++left_eq ;
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr, 
					  "%s: Result: CONTINUE-left-eq field %s\n", 
					  __func__,fld->name);
				continue ; break ;
			      case 1: /* A > R && A == L && L < R */
				/* IMHO impossible */
			      default:
				abort();
			      }
			    break ;
			  default:
			    abort();
			  }
#if 0 /* ***************************************************** */
			if ( match(A, L, R) ) 
			  {
			    if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
			      fprintf(stderr, "%s: Result: MATCH field %s\n", 
				      __func__,fld->name);
			    return 0 ;
			  }
			else if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
			  fprintf(stderr, "%s: Result: NOT MATCH field %s\n", 
				  __func__,fld->name);
			if ( ! cmp(A, R) ) /* A == R */
			  {
			    if ( ! cmp(L, R) )
			      {
				assert( ! cmp(A, L) );
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr, 
					  "%s: Result: CONTINUE-all-EQ field %s\n", 
					  __func__,fld->name);
				continue ;
			      }
			    else
			      ++right_eq ;
			    if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
			      fprintf(stderr, 
				      "%s: Result: CONTINUE-right-EQ field %s\n", 
				      __func__,fld->name);
			    continue ;
			  }
			if ( ! cmp(A, L) )
			  {
			    if ( ! cmp(L, R) )
			      {
				assert( ! cmp(A, R) );
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr, 
					  "%s: Result: CONTINUE-all-EQ field %s\n", 
					  __func__,fld->name);
				continue ;
			      }
			    else
			      ++left_eq ;
			    if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
			      fprintf(stderr, 
				      "%s: Result: CONTINUE-left-eq field %s\n", 
				      __func__,fld->name);
			    continue ;
			  }
#endif /* 0 */ /* ***************************************************** */
		      }
		    else /* after first left/right equals */
		      {
			if ( right_eq )
			  {
			    switch ( cmp(A, R) ) 
			      {
			      case 0:
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr, "%s: Result: CONTINUE field %s\n", 
					  __func__,fld->name);
				continue ;
				break ;
			      case -1: /* match */
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr,
					  "%s: Result: MATCH-after-REQ field %s\n",
					  __func__,fld->name);
				return 0 ;
				break ;
			      case 1:
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr,
					  "%s: Result: GT-after-REQ field %s\n",
					  __func__,fld->name);
				return 1 ;
				break ;
			      default:
				abort();
				break ;
			      }
			  }
			else /* left_eq */
			  {
			    assert(left_eq); 
				
			    switch ( cmp(A, L ) )
			      {
			      case 0:
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr, "%s: Result: CONTINUE field %s\n", 
					  __func__,fld->name);
				continue ;
				break ;
			      case 1: /* match */
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr,
					  "%s: Result: MATCH-after-LEQ field %s\n",
					  __func__,fld->name);
				return 0 ;
				break ;
			      case -1:
				if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
				  fprintf(stderr,
					  "%s: Result: LT-after-LEQ field %s\n",
					  __func__,fld->name);
				return -1 ;
				break ;
			      default:
				abort();
				break ;
			      }
			  }
		      }
		  }
		default:
		  assert("FIXME" == 0);
		  break ;
		}
	    }
	}
      if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
	fprintf(stderr,
		"%s: Result: EQ all fields checked, last %s\n",
		 __func__,fld->name);
       return 0 ;
    }
  else
    { 
      /* ***************** range vs range ********************** */
      
      assert( a_range->magic == ADB_MAGIC_RANGE );
      assert( b_range->magic == ADB_MAGIC_RANGE );

      /* right child of -INF range is NOT PROBLEM HERE, since range vs range
       * sitiation occurs only during inserting etc. range to range tree 
       */
      if ( a_range->flag && a_range->flag == b_range->flag )
	{
	  if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
	    {
	      fputs("A...->flag && A...->flag == B...->flag ",stderr);
	      Adb__rec_range_show(bh, adb_range_to_rec(a_range) );
	    }
	  return 0 ;
	}
      if ( a_range->flag & ADB_RANGE_LEFT_INF ) /* A < B */
	{
	 if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
	    {
	      fputs("A...->flag & -INF ",stderr);
	      Adb__rec_range_show(bh, adb_range_to_rec(a_range));
	    } 
	 return -1 ; /* everything is greater than a_range */
	}
      if ( b_range->flag & ADB_RANGE_LEFT_INF ) /* A > B */
	{
	  if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
	    {
	      fputs("B...->flag & -INF ",stderr);
	      Adb__rec_range_show(bh, adb_range_to_rec(b_range));
	    }
	return 1 ; /* A is greater */
	}
      if ( a_range->flag & ADB_RANGE_RIGHT_INF ) /* A > B */
	{
	  if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
	    {
	      fputs("A...->flag & +INF ",stderr);
	      Adb__rec_range_show(bh, adb_range_to_rec(a_range) );
	    } 
	  return 1 ; /* nothing is greater than a_range */
	}
      if ( b_range->flag & ADB_RANGE_RIGHT_INF ) /* A < B */
	{
	  if (  Adb_debug & ADB_DEBUG_RANGE_CMP )
	    {
	      fputs("B...->flag & +INF ",stderr);
	      Adb__rec_range_show(bh, adb_range_to_rec(b_range) );
	    }
	  return -1 ; /* A is lesser */
	}
      if (  Adb_debug & ADB_DEBUG_RANGE_CMP  ) 
	{
	  fprintf(stderr, "Range A L: ");
	  Adb__rec_range_show(bh, adb_range_to_rec(a_range) );
	  fprintf(stderr, "Range B L: "); 
	  Adb__rec_range_show(bh, adb_range_to_rec(b_range) );
	}

      return Adb_data_cmp( adb_range_to_rec(a_range), 
			   adb_range_to_rec(b_range), bh, 0);
    }
  return 0;
}

void
Adb_verify_tree_content(adb_tnx_t *tnx, adb_file_t *file, adb_bh_t *bh, adb_rec_t **last)
{
  adb_tavl_traverser_t tr ;
  adb_rec_t *rec ;

  int ret , i ;

  adb_tavl_t_init(bh, &tr );

  for ( i = 0 ; ( rec = adb_tavl_t_next(bh, &tr)) ; ++i )
    {
      if ( *last )
	{
	  ret = Adb_data_cmp(*last, rec , bh, 0);
	  if ( ret > 0 ) /* ( ret != -1 ) */
	    adb_tex(tnx, file, "BUG");
	  assert(ret < 0 )/* ( ret == -1 ) */;
	}
      *last = rec ;
    }

}

void
Adb_range_verify(adb_tnx_t *tnx, adb_bh_t *rbh, adb_file_t *file, adb_rec_t **last)
{
  adb_tavl_traverser_t tr , first ;
  adb_range_t *range , *r ;
  adb_rec_t *rec ;
  adb_bh_t *bh ;
  int ret ;

  adb_tavl_t_init(rbh, &tr );
  while ( ( range = (adb_range_t *)adb_tavl_t_next( rbh, &tr )) != NULL )
    {
      assert( range->magic == ADB_MAGIC_RANGE );
      bh = Adb_seg_load( tnx, file, range->offset, ADB_TNX_NULL);
      assert( bh->m->parent == rbh->offset );
      assert( bh->offset == range->offset );
      Adb_check_bh( bh, __FUNCTION__, FALSE );
      adb_tavl_t_init(bh, &first );
      
      switch ( bh->m->content )
	{
	case ADB_MAGIC_RANGE:
	  Adb_range_verify(tnx, bh, file , last );
	  r = adb_tavl_t_first(bh, &first );
	  ret = Adb_range_cmp( r, range, rbh, 0);
	  if ( ret > 0 )
	    {
	      adb_tex(tnx, file, "bug" );
	      abort();
	    }
	  break ;
	case ADB_MAGIC_TREE:
	  if ( range->flag == ADB_RANGE_FLAG_NULL )
	    { /* checking +-INF doesn't make sense */
	      rec = adb_tavl_t_first(bh, &first );
	      ret = Adb_range_cmp( rec, range, rbh, 0);
	      if ( ret > 0 )
		{
		  adb_tex( tnx, file, "bug" );
		  abort();
		}
	    }
	  Adb_verify_tree_content(tnx, file, bh, last);
	  break ;
	default:
	  assert("BUG" == 0 );
	  break ;
	}
    }
}

void
Adb_verify(adb_tnx_t *tnx, adb_file_t *file)
{
  adb_bh_t *top = Adb_bh_find(tnx, file, file->meta->data_start );
  Last = NULL ; 
  if ( top->m->content == ADB_MAGIC_RANGE ) /* don't care if there is one level only */
    Adb_range_verify( tnx, top, file , &Last );
}

static char *___RANGE_FLAGS[] = { "NULL", "-INF", "+INF", NULL } ;

static void
Adb__range_show(adb_bh_t *bh, void *a,  adb_file_t *file UNUSED , char *func )
{
  adb_range_t *r = ( adb_range_t *) a ;
  assert( r->magic == ADB_MAGIC_RANGE );

  fprintf(stderr,"%s - %d - <%s>", func, r->id, ___RANGE_FLAGS[r->flag] );
/*   assert( r->item && r->item < (void **)ADB_SEG_SIZE); */
  Adb__rec_range_show( bh, adb_range_to_rec(r));

}

void
Adb_range_show(adb_bh_t *rbh, adb_file_t *file,char *func)
{
  adb_tavl_traverser_t tr ;
  void *data ;
  adb_tavl_t_init(rbh, &tr );
  
  while ( ( data = adb_tavl_t_next( rbh, &tr )) != NULL )
    Adb__range_show ( rbh, data, file , func );
}

adb_range_t *
Adb_range_new(adb_tnx_t *tnx, adb_bh_t *bh, off_t data_start, adb_magic_t type, int vsize)
{
  adb_range_t *range ;

  if ( ! ( bh->state * ADB_TNX_DIRTY ))
    Adb_seg_dirty(tnx, bh);

  range = Adb_malloc( bh->file, /* sizeof(adb_range_t) + */ vsize ,
					   bh->m , __FUNCTION__);
  if ( range == NULL )
    {
      Adb_cs_range( tnx, bh, bh->file );
      longjmp(jb, 1);
    }

  range->magic = ADB_MAGIC_RANGE ;
  if ( data_start )
    range->offset = data_start ; 
/*   range->left_edge = range + 1 ; */
/*   range->left_edge -= bh->addr ; */
  range->id = bh->file->db->last_range_id++ ;
  range->flag = ADB_RANGE_FLAG_NULL ;
  range->data_start_type = type ;
  fprintf(stderr, "New range id %d #%Ld\n",range->id, data_start);

  return range ;
}


/* void */
/* Adb_fill_with(adb_rec_t *rec, adb_file_t *file, int is_min ) */
/* { */

/* } */


void
Adb_show_one_range(adb_range_t *range, adb_bh_t *bh, adb_file_t *file UNUSED )
{
  adb_range_t *next ;
  adb_tavl_traverser_t tr ;
  int save = Adb_debug ;

  fprintf(stderr,"R%d <left edge> level %d :", range->id , bh->m->level );
  switch ( range->flag )
    {
    case ADB_RANGE_LEFT_INF:
      fputs("<-INF> ", stderr);
      break ;
    case ADB_RANGE_RIGHT_INF:
      fputs("<+INF> ", stderr);
      break ;
    default:
      break ;
    }
  Adb__rec_range_show( bh, adb_range_to_rec(range) );
  Adb_debug = 0 ;

  adb_tavl_t_find(bh, &tr, range );
  next = adb_tavl_t_next( bh, &tr );

  Adb_debug = save ;

  if ( ! next ) return ;
  assert( (void *)next >= (void *)bh->m + 8192 );
  assert( (void *)next < (void *)bh->m + 8192 + 0x100000 );

  fputs(" ... ",stderr);
  switch ( next->flag )
    {
    case ADB_RANGE_LEFT_INF:
      fputs("<-INF> ", stderr);
      break ;
    case ADB_RANGE_RIGHT_INF:
      fputs("<+INF> ", stderr);
      break ;
    default:
      break ;
    }
  Adb__rec_range_show( bh, adb_range_to_rec(next) );
}

#if 0
void
_adb_tree_dump(adb_tavl_node_t *p, tavl_comparison_func cmp, unsigned int sb , adb_file_t *file)
{
  adb_magic_t *magic ;
  adb_range_t *range ;
  fprintf(stderr,"Node %p <%p %p> ",p, 
	  (void *)p->tavl_link[0] + sb , 
	  (void *)p->tavl_link[1] + sb );
  if ( p->tavl_data )
    {
      magic = (adb_magic_t *)((void *)p->tavl_data + sb) ;
      switch ( *magic )
	{
	case ADB_MAGIC_RANGE:
	  range = (adb_range_t *) magic ;
	  Adb_data_show( file, (void *)adb_range_to_rec(range));
	  break  ;
	default:
	  break ;
	}
    }
  if ( p->tavl_link[0] && (void *)p->tavl_link[0] + sb != p )
    _adb_tree_dump((void *)p->tavl_link[0] + sb , cmp, sb, file );  
  if ( p->tavl_link[1] && (void *)p->tavl_link[1] + sb != p )
    _adb_tree_dump((void *)p->tavl_link[1] + sb , cmp, sb, file );  
}

void
adb_tree_dump(adb_tavl_table_t *_tree, tavl_comparison_func cmp, unsigned int sb , adb_file_t *file)
{
  adb_tavl_table_t *tree = (void *)_tree + sb ;
  adb_tavl_node_t *p;
  
  assert( file->magic == ADB_MAGIC_FILE );
  p = (void *)tree->tavl_root + sb;
  if ( tree->tavl_count )
    _adb_tree_dump( p, cmp, sb, file );
}
#endif /* 0 */

/* USE !!! */
#if 0
inline void
Adb_range_set_right_edge(adb_bh_t *bh, adb_range_t *range)
{
  adb_range_t *next ;
  adb_tavl_traverser_t tr ;
  
  tr.tavl_table = &bh->m->tree ;
  tr.tavl_node = &range->node ;
  next = (adb_range_t *) adb_tavl_t_next( bh, &tr );

  assert( next->magic == ADB_MAGIC_RANGE );  

  range->right_edge = next->left_edge ; /* WOW! */

}
#endif /* 0 */

/*
 *                         *** Strategy ***
 *
 * A.0 tree segment S0 (arg bh) full - no range tree <Adb_create_range_0>
 *
 *                S0	       	        S1             S0             S2
 *	  first ....... last  ===>     	^ 	       ^ \	      ^
 *					|	       |  \	      |
 *					|	       |   \data move |
 *				       	|      	       |    \  	      |
 *		     left edge:   undef ...    	  first ...  +-->last ...
 *		     flag:     	   -INF	R1     	       R0             R2 +INF
 *                   records:           0    \        max-1        /  1
 *                                             \       |       	 /
 *                                               \  range-tree /
 *                   new: S1 S2 R0 R1 R2 range-tree
 */

/* range_1 -INF , range_0 , range_2 +INF */

#define FIRST 0
#define MIDDLE 1
#define LAST 2

/* makes interval tree above tree blocks */

adb_bh_t *
Adb_create_range_0(adb_tnx_t *tnx, adb_bh_t *bh , adb_file_t *file)
{
  adb_bh_t *rbh , *new_bh[3];
  adb_rec_t *rec[3]  ;
  size_t size[3] /* range */ , rsize[3] /* record */ ;
  adb_range_t *range[3] ; /* new ranges -INF, NULL, +INF , Space: rbh */
  adb_rec_t *new_data[3] ;

  adb_tavl_traverser_t tr ;

  const void *ret ;
  adb_magic_t new_seg_magic = ADB_MAGIC_TREE ;
  void *(*op[3])() = { adb_tavl_t_first, adb_tavl_t_next, adb_tavl_t_last } ;
  int i ;
  int new_range_level = 1 ; /* rta level */

  static int usage ;
  fprintf(stderr, "%s: usage %d file %s\n", __func__, usage++, file->meta->name);

#if TEX_DEBUG == 1
  adb_tex(tnx, file , "create-0-before" );
#endif /* 0 */
#if 0
  tavl_walk( bh, Adb_data_show, file );
#endif


  assert(bh->tree->tavl_count > 3 );

  Adb_check_bh(bh , __FUNCTION__, FALSE );

  new_bh[MIDDLE] = bh ;

  /* new top level range tree */
  rbh = Adb_seg_add(tnx, file, &file->meta->data_start, ADB_MAGIC_RANGE, new_range_level, -1LL);
  Adb_tnx_set_data_start_type(file, &file->meta->data_start_type, ADB_MAGIC_RANGE) ;

  /* adb_avlt_create - this stuff does _adb_seg_init */


  adb_tavl_t_init(bh, &tr );

  /* 
   * New top level range's problems
   *
   * Initialize 1st,2nd,3rd data/length, allocate new ranges 
   * 
   */
  for ( i = 0 ; i < 3 ; ++i )
    {
      rec[i] = (adb_rec_t *)op[i] ( bh, &tr ); /* nth item */
      assert(rec[i]);
      size[i] = Adb_vsize_get( file, rec[i] ) + file->meta->all_data_size + 
	sizeof (adb_range_t) ;
      rsize[i] = size[i] - sizeof (struct ADB_RANGE) /* adb_range_t */;

      /* range->offset filled below */
      range[i] = Adb_range_new( tnx, rbh, 0LL , new_seg_magic, size[i] ); /* !FAIL */

      memcpy(adb_range_to_rec(range[i]), rec[i], rsize[i] );
      
      ret = adb_tavl_insert( rbh, &range[i]->node, range[i]); assert(!ret);

      assert( range[i] );
      switch (i)
	{
	case FIRST: range[i]->flag = ADB_RANGE_LEFT_INF ; break ;
	case MIDDLE: range[i]->flag = ADB_RANGE_FLAG_NULL ; break ;
	case LAST: range[i]->flag = ADB_RANGE_RIGHT_INF ; break ;
	default: assert("Bug" == 0); break ;
	}
    }

  /* S1 */
  /* now to bother with rbh->m->level - 1 stuff e.g. copy/move one/three records to
     each new range/tree block */

  for ( i = 0 ; i <= LAST ; ++i )
    {
      switch ( i )
	{
	case FIRST:
	case LAST:
	  new_bh[i] = Adb_seg_add( tnx, file, &range[i]->offset,
				    new_seg_magic, bh->m->level, rbh->offset);
	  adb_tavl_t_init(bh, &tr);
	  new_data[i] = Adb_malloc( file, rsize[i], new_bh[i]->m, __FUNCTION__);
	  memcpy( new_data[i], rec[i], rsize[i]  );
	  
	  ret = adb_tavl_insert( new_bh[i], &new_data[i]->node, new_data[i] ); 
	  assert(!ret);
#if 0
	  _adb_link_update(tnx, file, bh, rec[i], new_bh[i], new_data[i] );
#endif
	  ret = adb_tavl_delete( bh /* old BH */, rec[i] ); assert(ret);
#if 0
	  Adb_free( (void *)rec[i],  rsize[i] , bh->m, __FUNCTION__);
#endif /* 0 */
	  break ;
	case MIDDLE:
	  range[i]->offset = bh->offset ;
	  bh->m->parent = rbh->offset ;
	  new_bh[i] = bh ;
	  break ;
	default:
	  assert("Bug" == 0 );
	  break ;
	}
    }

  for ( i = 0 ; i <= LAST ; ++i )
    if ( i != MIDDLE )
      Adb_link_update(tnx, file, bh, rec[i], new_bh[i], new_data[i] );

  for ( i = 0 ; i < 3 ; ++i )
    {
      Adb_show_one_range( range[i] , rbh, file );
      Adb_check_bh(new_bh[i] , __FUNCTION__, EOF ); /* BOOOOM if rbh->m->level > 1 */
    }

  for ( i = 0 ; i <= LAST ; ++i )
    if ( i == FIRST || i == LAST )
      Adb_free( file, (void *)rec[i],  rsize[i] , bh->m, __FUNCTION__);

  /* 2nd range item */

  Adb_range_show( rbh, file , __PRETTY_FUNCTION__ );

   Adb_check_bh(rbh , __FUNCTION__, FALSE );
   for ( i = 0 ; i < 3 ; ++i )
     Adb_check_bh(new_bh[i] ,  __FUNCTION__, FALSE );

   if ( Adb_debug & ADB_DEBUG_INDEX )
     Adb_verify( tnx, file );
#if TEX_DEBUG == 1
   adb_tex(tnx, file , "create-0-after" );
#endif
   return rbh ;
}

/* #error "+INF rbh-1 block's range have flags in _reverse order_ !!!" */

adb_bh_t *
Adb_create_range_n(adb_tnx_t *tnx, adb_bh_t *bh , adb_file_t *file)
{
  adb_bh_t *rbh , *new_bh[3];
  off_t old_data_start = file->meta->data_start ;
  adb_range_t *rg[3] , *new_range ;
  size_t size[3] ;
  adb_range_t *range[3] ; /* new ranges -INF, NULL, +INF , Space: rbh */

  adb_tavl_traverser_t tr ;

  const void *ret ;
  adb_magic_t new_seg_magic = ADB_MAGIC_RANGE ;
  void *(*op[3])() = { adb_tavl_t_first, adb_tavl_t_next, adb_tavl_t_last } ;
  int i , j ;
  int new_range_level = bh->m->level + 1 ;
  adb_bh_t *child ;
#if TEX_DEBUG == 1
  adb_tex(tnx, file , "create-n-before" );
#endif /* 0 */
#if 0
  tavl_walk( bh, Adb_data_show, file );
  Adb_check_bh(bh , NULL );
#endif

  new_bh[MIDDLE] = bh ;

  assert( bh->m->content == ADB_MAGIC_RANGE );
  assert( bh->m->level >= 1 );

  if ( bh->m->tree.tavl_count < 9 ) /* 3 nodes left in original range,
				       3 nodes in moved to left -1 range,
				       3 nodes in moved to right -1 range,
				       3 nodes to point to the stuff above,
				       In the Land of Mordor where the Shadows lie. */
    {
      ADB_ERROR("\nCan't create nth range tree level because top level range tree\n\
contains only %d nodes, while the lower possible limit is 9 .\n\n\
This definitely isn't bug. This is a *feature*. Have a nice day!\n\n\
P.S. Hint: redesign your database, average record shouldn't be longer\n\
than ~110KB excluding BLOBs.",
		bh->m->tree.tavl_count);
      exit(2);
    }

  /* new to level range tree */
  rbh = Adb_seg_add( tnx, file, &file->meta->data_start, 
			  ADB_MAGIC_RANGE, new_range_level, -1LL );
  Adb_tnx_set_data_start_type(file, &file->meta->data_start_type, ADB_MAGIC_RANGE);

  /* adb_avlt_create - this stuff does Adb_seg_init */


  adb_tavl_t_init(bh, &tr );

  /* 
   * New top level range's problems
   *
   * Initialize 1st,2nd,3rd data/length, allocate new ranges 
   * 
   */
  for ( i = 0 ; i < 3 ; ++i )
    {
      /* range tree of range tree */
      switch ( i )
	{
	case FIRST:
	  /* seek to 3rd item */
	  adb_tavl_t_first( bh, &tr); adb_tavl_t_next( bh, &tr );
	  rg[i] = (adb_range_t *) adb_tavl_t_next( bh, &tr );
	  break ;
	case MIDDLE:
	  rg[i] = (adb_range_t *) adb_tavl_t_next( bh, &tr );
	  break ;
	case LAST:
	  adb_tavl_t_init(bh, &tr );
	  /* seek to 3rd item */
	  adb_tavl_t_last( bh, &tr); adb_tavl_t_prev( bh, &tr );
	  rg[i] = adb_tavl_t_prev( bh, &tr );
	  break ;
	default:
	  assert("Bug" == 0 );
	  break ;
	}
      size[i] = Adb_vsize_get( file,  adb_range_to_rec(rg[i])) +
	file->meta->all_data_size + sizeof(adb_range_t );
      range[i] = Adb_range_new( tnx, rbh, 0LL , ADB_MAGIC_RANGE, size[i] ); /*!FAIL*/

      memcpy(adb_range_to_rec(range[i]), 
	     adb_range_to_rec(rg[i]) , size[i] - sizeof(adb_range_t));

      switch ( i )
	{
	case FIRST: range[i]->flag = ADB_RANGE_LEFT_INF ; break ;
	case LAST: range[i]->flag = ADB_RANGE_RIGHT_INF ; break ;
	case MIDDLE: 
	  range[i]->flag = ADB_RANGE_FLAG_NULL ; 
	  range[i]->offset = old_data_start ;
	  bh->m->parent = rbh->offset ;
	  fprintf(stderr,"(%s) Old segment's parent #%Ld\n",__FUNCTION__, bh->m->parent);
	  break ;
	default: abort(); break ;
	}
      
      ret = adb_tavl_insert( rbh, &range[i]->node, range[i]); assert(!ret);
  
      assert( range[i] );
    }

  /* S1 */
  /* now to bother with rbh->m->level - 1 stuff e.g. move three records to
     each new range block */

  for ( i = 0 ; i <= LAST ; ++i )
    {
      switch ( i )
	{
	case FIRST:
	case LAST:
	  new_bh[i] = Adb_seg_add( tnx, file, &range[i]->offset,
				    new_seg_magic , bh->m->level , rbh->offset );
	  adb_tavl_t_init(bh, &tr);
	  for ( j = 0 ; j < 3 ; ++j )
	    {
	      adb_range_t *org ; /* old range */
	      size_t osize ;
	      org = (adb_range_t *) op[i] ( bh, &tr );
	      osize = Adb_vsize_get( file,  adb_range_to_rec(org)) +
		file->meta->all_data_size + sizeof(adb_range_t );
	      
	      new_range = Adb_range_new(tnx, new_bh[i], 
					  org->offset, 
					  org->data_start_type , osize );
	      if ( i == FIRST )
		switch (j)
		  {
		  case FIRST: new_range->flag = ADB_RANGE_LEFT_INF ; break ;
		  case MIDDLE: new_range->flag = ADB_RANGE_FLAG_NULL ; break ;
		  case LAST: new_range->flag = ADB_RANGE_RIGHT_INF ; break ;
		  default: assert("Bug" == 0 ); break ;
		  }
	      else /* i == LAST */
		{
		  assert( i == LAST );
		  switch(j)
		    {
		    case FIRST: new_range->flag = ADB_RANGE_RIGHT_INF ; break ;
		    case MIDDLE: new_range->flag = ADB_RANGE_FLAG_NULL ; break ;
		    case LAST: new_range->flag = ADB_RANGE_LEFT_INF ; break ;
		    default: assert("Bug" == 0 ); break ; 
		    }
		}
	      memcpy( adb_range_to_rec(new_range),
		      adb_range_to_rec(org), osize - sizeof(adb_range_t));
	      ret = adb_tavl_insert(new_bh[i], &new_range->node, new_range);
	      assert(!ret);
	      ret = adb_tavl_delete( bh, org ); assert(ret);
	      Adb_free( file, (void *)org, osize , bh->m, __FUNCTION__); 
	      
	      child = Adb_seg_load( tnx, file, new_range->offset, 
				     ADB_TNX_DIRTY);
	      child->m->parent = new_bh[i]->offset ;
	    }
	  assert(new_bh[i]->m->tree.tavl_count == 3 );
	  break ;
	case MIDDLE:
	  range[i]->offset = bh->offset ;
	  new_bh[i] = bh ;
	  new_range = range[i] ;
	  break ;
	default:
	  assert("Bug" == 0 );
	  break ;
	}
    }

  /* set flags for rbh - 1 ranges */
#if 0
  for ( i = 0 ; i < 3 ; ++i )
    {
      adb_range_t *r ;
      adb_tavl_t_init(new_bh[i], &tr);
      r = adb_tavl_t_first( new_bh[i], &tr );
      r->flag = ADB_RANGE_LEFT_INF ;
      r = adb_tavl_t_last( new_bh[i], &tr );
      r->flag = ADB_RANGE_RIGHT_INF ;
    }
#else
  i = MIDDLE ;
  {
    adb_range_t *r ;
    adb_tavl_t_init(new_bh[i], &tr);
    r = adb_tavl_t_first( new_bh[i], &tr );
    r->flag = ADB_RANGE_LEFT_INF ;
    r = adb_tavl_t_last( new_bh[i], &tr );
    r->flag = ADB_RANGE_RIGHT_INF ; 
  }
#endif /* 0 */
  
  for ( i = 0 ; i < 3 ; ++i )
    {
      Adb_show_one_range( range[i] , rbh, file );
      Adb_check_bh(new_bh[i] , __FUNCTION__, EOF ); /* BOOOOM if rbh->m->level > 1 */
    }
  /* 2nd range item */

  Adb_range_show( rbh, file , __PRETTY_FUNCTION__ );

/*   adb_tex(tnx, file , "create-n-after" ); */
  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify( tnx, file );
#if TEX_DEBUG == 1
  adb_tex(tnx, file , "create-n-after" );
#endif
  return rbh ;
}

void
Adb_create_range(adb_tnx_t *tnx, adb_bh_t *bh , adb_file_t *file)
{

  if ( bh->m->content == ADB_MAGIC_TREE )   /* allocate level 1 range tree space */
    {
      Adb_create_range_0(tnx, bh, file);
    }
  else
    {
      Adb_create_range_n(tnx, bh, file);
    }
}

static int Adb__count , Adb__walk_count ;

void
Adb__tree_count(void *data, void *param)
{
  adb_rec_t *r  = (adb_rec_t *) data ;
  adb_bh_t *bh = ( adb_bh_t *) param ;

  assert (r->magic == ADB_MAGIC_RECORD || r->magic == ADB_MAGIC_RANGE );

  ++Adb__walk_count;
  if ( Adb__walk_count > Adb__count )
    {
      fputs("= + err ==> ",stderr);
      Adb_data_show(r, bh->file );
    }
}



void
Adb_check_bh(adb_bh_t *bh, char *func , int dump)
{
  unsigned int each  ;
  adb_tavl_traverser_t tr ;
  adb_rec_t *rec ;
  adb_range_t *range ;

  assert(bh->magic == ADB_MAGIC_BH );

  adb_tavl_t_init(bh, &tr);
  if ( dump == EOF )
    {
      fprintf(stderr,"------------ (%s) %p #%Ld -------------\n 1st: ",func,
	      bh->m, bh->offset );
      rec =  (adb_rec_t *)adb_tavl_t_first(bh, &tr );
      Adb__rec_range_show(bh, rec);
    }

  adb_tavl_t_init(bh, &tr);
  for (  each = 0 ; ( rec = (adb_rec_t *)adb_tavl_t_next(bh, &tr )) ; ++each )
    {
      assert( (void *)rec >= (void *)bh->m +   0x2000 );
      assert( (void *)rec < (void *) bh->m + 0x102000 );
      
      switch ( dump )
	{
	case TRUE:
	  Adb__rec_range_show(bh, rec);
	  break ;
	default:
	  break ;
	}
      if ( rec->magic == ADB_MAGIC_RANGE )
	{
	  range = (adb_range_t *)rec ;
	  rec = adb_range_to_rec(range);
	}
      else
	{
	  assert( rec->magic == ADB_MAGIC_RECORD );
	}
      Adb_vsize_get(bh->file, rec );
      if ( ! bh->m->level )
	Adb_link_test(bh, rec );
    }

  if ( dump == EOF )
    {
      fprintf(stderr,"last: ");
      rec =  (adb_rec_t *)adb_tavl_t_last(bh, &tr );
      Adb__rec_range_show(bh, rec);
    }
  
  assert ( each >= bh->m->tree.tavl_count );
  fprintf(stderr,"[%s] %p valid\n", func, bh->m );
}

adb_range_t *
Adb_range_find_parent_range( adb_tnx_t *tnx, adb_bh_t *child, adb_file_t *file)
{
  adb_bh_t *parent ;
  adb_range_t  *parent_range ;
  adb_tavl_traverser_t tr ;

  
  parent = Adb_tnx_find_parent_bh(tnx, child, file );
  assert( parent->offset == child->m->parent );


  adb_tavl_t_init(parent, &tr);
  while ( ( parent_range = adb_tavl_t_next( parent, &tr )) != NULL )
    if ( parent_range->offset == child->offset )
      goto L_ok;
  
  
  fprintf(stderr,"(%s) parent #%Ld child #%Ld @ %p\n",__FUNCTION__, parent->offset,
	  child->offset, child->m );
  fprintf(stderr,"(%s) Found range : ",__FUNCTION__);
/*   Adb_show_one_range( parent_range, parent , file); */
/*   if ( parent_range->offset != child->offset ) */
  Adb_check_bh(child, __FUNCTION__, TRUE );
  abort();

 L_ok:
  assert( parent_range->magic == ADB_MAGIC_RANGE );
  assert( parent_range->offset == child->offset );
  return parent_range ;
}
/*
 *                         *** Strategy ***
 *
 * A.1 tree segment S0 full - range tree exists && S0 has no INF flag e.g. middle of
 *     range tree
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
/* *rbh - range's space */

/* totally inefficient  */
#define ZAPBUG(rec) assert((unsigned int)rec->node.tavl_link[0] < ADB_SEG_SIZE );  assert((unsigned int)rec->node.tavl_link[1] < ADB_SEG_SIZE )

static adb_range_t *
Adb__range_reincarnate(adb_tnx_t *tnx, adb_bh_t *rbh, adb_bh_t *old_bh,
			 adb_range_t *range )
{
  size_t old_size , new_size ;
  adb_rec_t *tmp_rec = NULL ;
  adb_range_t *new_range , *tmp_range = NULL ;
  adb_tavl_traverser_t tr ;
  adb_magic_t new_seg_magic ;
  void *ret ;
  const void *to_free ;

  assert( ! range->flag );

  adb_tavl_t_init(old_bh, &tr );

  old_size = Adb_vsize_get(old_bh->file, adb_range_to_rec(range) ) 
	+ old_bh->file->meta->all_data_size + sizeof(*range) ;
  switch ( old_bh->m->content )
    {
    case ADB_MAGIC_TREE:
      tmp_rec = adb_tavl_t_first( old_bh, &tr );
      new_size = Adb_vsize_get(old_bh->file, tmp_rec ) 
	+ old_bh->file->meta->all_data_size + sizeof(*range) ;
      new_seg_magic = ADB_MAGIC_TREE ;
      break ;
    case ADB_MAGIC_RANGE:
      tmp_range = adb_tavl_t_first( old_bh, &tr ); /* skip inf */
      new_size = Adb_vsize_get(old_bh->file, adb_range_to_rec(tmp_range) ) 
	+ old_bh->file->meta->all_data_size + sizeof(*range) ;
      new_seg_magic = ADB_MAGIC_RANGE ;
      break ;
    default:
      abort();
    }
  new_range = Adb_range_new(tnx, rbh, range->offset, new_seg_magic, 
			      new_size );
  switch ( old_bh->m->content )
    {
    case ADB_MAGIC_TREE:
      memcpy( adb_range_to_rec(new_range), tmp_rec, new_size - sizeof(*range));
      break ;
    case ADB_MAGIC_RANGE:
      memcpy( adb_range_to_rec(new_range), 
	      adb_range_to_rec(tmp_range), new_size - sizeof(*range));
      break ;
    default: 
      abort();
    }
  to_free = adb_tavl_delete(rbh, range);  assert(to_free);
  Adb_free( rbh->file, (void *)to_free, old_size , rbh->m, __FUNCTION__);
  
  ret = adb_tavl_insert( rbh, &new_range->node, new_range); assert(!ret);

  return new_range ;
}

static void
Adb__record_move(adb_tnx_t *tnx, adb_file_t *file, adb_bh_t *new_bh, 
		   adb_bh_t *old_bh, adb_rec_t *tmp_rec , size_t vsize)
{
  adb_rec_t *old_data = tmp_rec  , *new_data ;
  const void *to_free ;

  new_data = Adb_malloc( file, vsize , new_bh->m , __FUNCTION__);
  memcpy( new_data , old_data, vsize );

  if ( adb_tavl_insert( new_bh, &new_data->node, new_data ) )  /* stange ... */
    {
      fprintf(stderr,"Insert failed \a"); 
      assert("Bug" == 0 );
    }
  /* don't know if someone is pointing to me so : */
  Adb_link_update(tnx, file, old_bh, tmp_rec, new_bh, new_data );

  to_free = adb_tavl_delete( old_bh, old_data );  assert ( to_free );

  Adb_free(file, (void *)to_free, vsize , old_bh->m, __FUNCTION__);
}

/* create or split range */
void
Adb_cs_range(adb_tnx_t *tnx, adb_bh_t *fail_bh, adb_file_t *file )    
{
  
  if ( fail_bh->m->parent == -1LL ) /* top range full */
    return Adb_create_range( tnx, fail_bh, file );
  else 
    {
      adb_range_t *range = Adb_range_find_parent_range(tnx, fail_bh, file );
      if ( range->flag )
	Adb_seg_split_inf( tnx, fail_bh, file );
      else
	Adb_seg_split( tnx, fail_bh, file );
    }
}

adb_bh_t *
Adb_seg_split_tree(adb_tnx_t *tnx, adb_bh_t *old_bh /* where malloc failed */,
		     adb_file_t *file)
{
  adb_bh_t *new_bh , *rbh = Adb_tnx_find_parent_bh(tnx, old_bh , file);
  unsigned int count = old_bh->m->tree.tavl_count ;
  unsigned int i, j , limit = count / 2 ;
  adb_range_t *range = NULL /* parent */ ;
  adb_range_t *new_r_range ;
  adb_rec_t *tmp_rec , *stop_rec ;

  void *ret ;
  adb_tavl_traverser_t tr , last_live ;
  size_t new_r_size , vsize ;
  adb_magic_t new_seg_magic = old_bh->m->level ? ADB_MAGIC_RANGE : ADB_MAGIC_TREE ;
#if TEX_DEBUG == 1
  adb_tex(tnx, file , "split-tree-before" );
#endif /* 0 */

#if 0
  adb_tavl_walk( old_bh, Adb_data_show, file );
  sleep(3);
#endif
  Adb_check_bh(old_bh, __FUNCTION__ , FALSE );
  Adb_check_bh(rbh, __FUNCTION__ , FALSE );

  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);


  /* those records left in old bh */
  adb_tavl_t_init( old_bh, &tr );
  for ( tmp_rec = NULL , i = 0 ; i < limit ; ++i )
    {
      tmp_rec = adb_tavl_t_next( old_bh , &tr );
      assert( tmp_rec->magic == ADB_MAGIC_RECORD );
    }

  /* traverser is set to last record left in old block => next should return
     1st record of new block */
  adb_tavl_t_copy( old_bh, &last_live, &tr ); 
  /* content left in old block seen so look for left edge of new block */
  tmp_rec = adb_tavl_t_next( old_bh , &tr );
  stop_rec = tmp_rec ;
  new_r_size = Adb_vsize_get(file, tmp_rec) + file->meta->all_data_size +
	sizeof(adb_range_t);

  range = Adb_range_find_parent_range(tnx, old_bh, file);

  new_r_range = Adb_range_new( tnx, rbh, 0LL, new_seg_magic, 
				 new_r_size );
  memcpy( adb_range_to_rec( new_r_range) , tmp_rec, new_r_size - sizeof(adb_range_t));
  /*
   * I don't care about new_range != NULL because Adb_range_new does longjmp
   * after Adb_cs_range(), when malloc failed in ^^^
   */
  /* no longjmp, real work is about to begin */
  new_bh = Adb_seg_add( tnx, file, &new_r_range->offset ,
			 ADB_MAGIC_TREE, old_bh->m->level, rbh->offset );
      
  fprintf(stderr,"left range (old) <left edge> :"); 
  Adb_data_show( adb_range_to_rec(range), file);
  fprintf(stderr,"%d Left in old segment\n",i);
  /* 1st (old) done */

  /* 2nd range (new) */
  /* new tree */

/*   Adb_debug |= ADB_DEBUG_DATA_CMP; */
  for ( j = 0 ; ; ++i , ++j )
    {
      tmp_rec = adb_tavl_t_last( old_bh, &tr );
      assert( tmp_rec );
#if 1
      {
	adb_tavl_table_t *tree = &old_bh->m->tree ;
	adb_tavl_node_t *node = (void *)tree->tavl_root + old_bh->addr ;
	fprintf(stderr, "Next to move %p ",tmp_rec); Adb_data_show(tmp_rec, file );
	fprintf(stderr, "Root of segment %p ", node ); Adb_data_show(DATA(node), file);
      
	Adb_check_bh(old_bh, __FUNCTION__ , FALSE );
	Adb_check_bh(new_bh, __FUNCTION__ , FALSE );

      }
#endif
      assert ( tmp_rec->magic == ADB_MAGIC_RECORD );
      vsize = Adb_vsize_get(file, tmp_rec ) + file->meta->all_data_size ;
      /* freeing here too */
      Adb__record_move( tnx, file, new_bh, old_bh, tmp_rec , vsize );

      if ( tmp_rec == stop_rec )
	break ;
    }
      
#if 0
  __debug__range_cmp = 1 ;
#endif
#if 0 /* ???? */
  ret = adb_tavl_delete(rbh, range);  assert(ret);
  fputs("deleting range (old) ", stderr);

  Adb_data_show(adb_range_to_rec(range), file);

  ret = adb_tavl_insert( rbh, &range->node, range ); assert( ! ret) ;
#endif /* 0 */
  ret = adb_tavl_insert( rbh, &new_r_range->node, new_r_range ); assert( ! ret) ;

  Adb_check_bh(rbh, __FUNCTION__ , TRUE );

#if 1
  Adb_check_bh(old_bh, __FUNCTION__, EOF);
  Adb_check_bh(new_bh, __FUNCTION__, EOF);
  assert( count == ( old_bh->m->tree.tavl_count + new_bh->m->tree.tavl_count ));
/*   assert( old_bh->m->tree.tavl_count == limit ); */
#endif
  fprintf(stderr,"right range <left edge> :"); 
  Adb_data_show( adb_range_to_rec(new_r_range), file);
/*   fprintf(stderr,"right range <right edge> :"); Adb_data_show(range->right_edge, file); */

  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);  

#if TEX_DEBUG == 1
  adb_tex(tnx, file , "split-tree-after" );
#endif /* 0 */
  return rbh ;
}

adb_bh_t *
Adb_seg_split_range(adb_tnx_t *tnx, adb_bh_t *old_bh /* where malloc failed */,
			   /* adb_range_t *range, */ adb_file_t *file)
{
  adb_bh_t *new_bh , *rbh = Adb_tnx_find_parent_bh(tnx, old_bh , file);
  unsigned int count = old_bh->m->tree.tavl_count ;
  unsigned int i, j , limit = count / 2 ;
  adb_range_t *tmp_range = NULL  , *range = NULL /* parent */ ;
  adb_range_t *new_r_range ;
  const void *to_free ;

  const void *ret ;
  adb_tavl_traverser_t tr , last_live ;
  size_t new_r_size , vsize ;
  adb_magic_t new_seg_magic = old_bh->m->level ? ADB_MAGIC_RANGE : ADB_MAGIC_TREE ;
  adb_range_t *r ;

  assert( old_bh->m->level >= 1 );

  assert( old_bh->m->tree.tavl_count >= 6 );

  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);

#if TEX_DEBUG == 1
  adb_tex(tnx, file , "split-range-before" );
#endif /* 0 */

#if 0
  adb_tavl_walk( old_bh, Adb_data_show, file );
  Adb_check_bh(old_bh, __FUNCTION__ );
  sleep(3);
#endif

  adb_tavl_t_init( old_bh, &tr );
  for ( tmp_range = NULL , i = 0 ; i < limit ; ++i )
    {
      tmp_range = adb_tavl_t_next( old_bh , &tr );
      assert( tmp_range->magic == ADB_MAGIC_RANGE );
    }
  
  /* traverser is set to last record left in old block => next should return
     1st record of new block */
  adb_tavl_t_copy( old_bh, &last_live, &tr ); 
  /* content left in old block seen so look for left edge of new block */
  tmp_range = adb_tavl_t_next( old_bh , &tr );

  new_r_size = file->meta->all_data_size + sizeof(adb_range_t ) +
    Adb_vsize_get( file, adb_range_to_rec(tmp_range));

  range = Adb_range_find_parent_range(tnx, old_bh, file);
  
  /*
   * I don't care about new_range != NULL because Adb_range_new does longjmp
   * after Adb_cs_range() when malloc failed 
   */
  new_r_range = Adb_range_new( tnx, rbh, 0LL, ADB_MAGIC_RANGE , 
				 new_r_size );
  memcpy( adb_range_to_rec(new_r_range), 
	  adb_range_to_rec(tmp_range),
	  new_r_size - sizeof(adb_range_t ));

  /* no longjmp, so real work is about to begin */
  new_bh = Adb_seg_add( tnx, file, &new_r_range->offset,
			 new_seg_magic, old_bh->m->level, rbh->offset );
/*   new_r_range->offset = new_data_start ; */
      

  fprintf(stderr,"left range (old) <left edge> :"); 
  Adb_data_show( adb_range_to_rec(range), file);
  fprintf(stderr,"%d Left in old segment\n",i);

  /* 1st (old) done */

  /* 2nd range (new) */

/*   Adb_debug |= ADB_DEBUG_DATA_CMP; */
  for ( j = 0 ; ; ++i , ++j )
    {
      adb_range_t *old_range , *new_range = NULL ;
      adb_bh_t *child ;
      old_range = adb_tavl_t_last( old_bh, &tr );

#if 0
      {
	adb_tavl_table_t *tree = &old_bh->m->tree ;
	adb_tavl_node_t *node = (void *)tree->tavl_root + old_bh->addr ;
	fprintf(stderr, "Next to move %p ",tmp_rec); Adb_data_show(tmp_rec, file );
	fprintf(stderr, "Root of segment %p ", node ); 
      
	Adb_data_show( (void *)node->tavl_data + old_bh->addr, file);
      }
#endif
      vsize = Adb_vsize_get(file, adb_range_to_rec(old_range) ) 
	+ file->meta->all_data_size + sizeof(*old_range) ;
	  
      new_range = Adb_malloc(file, vsize, new_bh->m, __FUNCTION__ );
      memcpy( new_range, old_range, vsize );

      if ( adb_tavl_insert( new_bh, &new_range->node, new_range ) )  
	fprintf(stderr,"Insert failed \a"); /* stange ... */
      to_free = adb_tavl_delete( old_bh, old_range );
      assert ( to_free != NULL );
      Adb_free(file, (void *)to_free, vsize ,old_bh->m, __FUNCTION__);
	  
      /* set new parent's offset in child block */
      child =  Adb_seg_load( tnx, file, new_range->offset,
			      ADB_TNX_DIRTY);
      child->m->parent = new_bh->offset ;
      /* Adb_seg_unload( tnx, child, file ); */
      if ( old_range == tmp_range )
	break ;
    }
      
#if 0
  __debug__range_cmp = 1 ;
#endif
  Adb_check_bh(rbh, __FUNCTION__ , FALSE );

  ret = adb_tavl_delete(rbh, range);  assert(ret);
  fputs("deleting range (old) ", stderr);

  Adb_data_show(adb_range_to_rec(range), file);

  ret = adb_tavl_insert( rbh, &range->node, range ); assert( ! ret) ;
  ret = adb_tavl_insert( rbh, &new_r_range->node, new_r_range ); assert( ! ret) ;

  /* if content is RANGE */
  /* 
   * set flags 
   */
  adb_tavl_t_init(old_bh, &tr);
  r = adb_tavl_t_first(old_bh, &tr);
  r->flag = ADB_RANGE_LEFT_INF ;
  r = adb_tavl_t_last( old_bh, &tr );
  r->flag = ADB_RANGE_RIGHT_INF ;
  
  adb_tavl_t_init(new_bh, &tr);
  r = adb_tavl_t_first(new_bh, &tr);
  r->flag = ADB_RANGE_LEFT_INF ;
  r = adb_tavl_t_last( new_bh, &tr );
  r->flag = ADB_RANGE_RIGHT_INF ;

#if 1
  Adb_check_bh(old_bh, __FUNCTION__, EOF);
  Adb_check_bh(new_bh, __FUNCTION__, EOF);
  assert( count == ( old_bh->m->tree.tavl_count + new_bh->m->tree.tavl_count ));
/*   assert( old_ta->m->tree.tavl_count == limit ); */
#endif
  fprintf(stderr,"right range <left edge> :"); 
  Adb_data_show( adb_range_to_rec(new_r_range), file);
/*   fprintf(stderr,"right range <right edge> :"); Adb_data_show(range->right_edge, file); */

  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);

#if TEX_DEBUG == 1
  adb_tex(tnx, file , "split-range-after" );
#endif /* 0 */
  return rbh ;
}

adb_bh_t *
Adb_seg_split(adb_tnx_t *tnx, adb_bh_t *old_bh /* where malloc failed */,
			   /* adb_range_t *range, */ adb_file_t *file)
{
  adb_bh_t *rbh = Adb_tnx_find_parent_bh(tnx, old_bh , file); 

/*   adb_magic_t new_seg_magic = old_bh->m->level ? ADB_MAGIC_RANGE : ADB_MAGIC_TREE ; */


  if ( old_bh->m->level == 0 )
    Adb_seg_split_tree( tnx, old_bh, file);
  else
     Adb_seg_split_range( tnx, old_bh, file);

#ifdef BLOATWARE
#if TEX_DEBUG == 1
  adb_tex(tnx, file , "split-before" );
#endif /* 0 */

#if 0
  adb_tavl_walk( old_bh, Adb_data_show, file );
  Adb_check_bh(old_bh, __FUNCTION__ );
  sleep(3);
#endif
  
  adb_tavl_t_init( old_bh, &tr );
  tmp_rec = adb_tavl_t_first( old_bh, &tr );

  if ( tmp_rec->magic == ADB_MAGIC_RECORD )
    new_r_size = file->all_data_size + Adb_vsize_get( file, tmp_rec) + 
      sizeof(adb_range_t ) ;
  else
    {
      tmp_range = (adb_range_t *)tmp_rec ;
      new_r_size = file->all_data_size + sizeof(adb_range_t ) +
	Adb_vsize_get( file, adb_range_to_rec(tmp_range));
    }

  
  rbh = Adb_tnx_find_parent_bh(tnx, old_bh , file);
  

  range = Adb_range_find_parent_range(tnx, old_bh, file);

  new_r_range = Adb_range_new( tnx, rbh, 0LL, new_seg_magic, 
				 new_r_size );
  /*
   * I don't care about new_range != NULL because Adb_range_new does longjmp
   * after Adb_cs_range() when malloc failed 
   */
  assert( new_r_range );
  /* no longjmp, so real work is about to begin */
  new_bh = Adb_seg_add( tnx, file , adb_fd, ADB_SEG_SIZE, &new_r_range->offset ,
			 new_seg_magic , old_bh->m->level , rbh->offset );
/*   new_r_range->offset = new_data_start ; */
      

  /* really needed ??? */
#ifdef BUGWARE
  if ( tmp_rec->magic == ADB_MAGIC_RECORD )
    memcpy( adb_range_to_rec(range), tmp_rec, new_r_size );
  else
    memcpy( adb_range_to_rec(range), adb_range_to_rec(tmp_range), new_r_size );
#endif

  fprintf(stderr,"left range (old) <left edge> :"); 
  Adb_data_show( adb_range_to_rec(range), file);

  adb_tavl_t_init( old_bh, &tr );
  /* old range's tree block */
  for ( tmp_rec = NULL , i = 0 ; i < limit ; ++i )
    {
      tmp_rec = adb_tavl_t_next( old_bh , &tr );
      ZAPBUG(tmp_rec);
    }

  assert ( tmp_rec );
  /* 1st (old) done */

  /* 2nd range (new) */

  fprintf(stderr,"%d Left in old segment\n",i);
  /* new tree */

/*   Adb_debug |= ADB_DEBUG_DATA_CMP; */
  for ( adb_tavl_t_copy( old_bh, &last_live , &tr) , j = 0 ; 
	;
	++i , ++j , adb_tavl_t_copy( old_bh, &last_live , &tr)  )
    {
      if ( ! ( tmp_rec = adb_tavl_t_next( old_bh, &last_live )))
	break ;

#if 0
      {
	adb_tavl_table_t *tree = &old_bh->m->tree ;
	adb_tavl_node_t *node = (void *)tree->tavl_root + old_bh->addr ;
	fprintf(stderr, "Next to move %p ",tmp_rec); Adb_data_show(tmp_rec, file );
	fprintf(stderr, "Root of segment %p ", node ); 
      
	Adb_data_show( (void *)node->tavl_data + old_bh->addr, file);
      }
#endif
      if ( tmp_rec->magic == ADB_MAGIC_RECORD )
	{
	  vsize = Adb_vsize_get(file, tmp_rec ) + file->all_data_size ;
	  if ( ! j )
	    memcpy( adb_range_to_rec(new_r_range), tmp_rec , vsize );
	  Adb__record_move( new_bh, old_bh, tmp_rec , vsize );
	    
	}  
      else
	{
	  adb_range_t *old_range = (adb_range_t *) tmp_rec , *new_range = NULL ;
	  adb_bh_t *child ;
	  vsize = Adb_vsize_get(file, adb_range_to_rec(old_range) ) 
	    + file->all_data_size + sizeof(*old_range) ;
	  
	  new_range = Adb_malloc(file, vsize, new_bh->m, __FUNCTION__ );
	  memcpy( new_range, old_range, vsize );
	  if ( !j )
	    memcpy( adb_range_to_rec(new_r_range), adb_range_to_rec(new_range), vsize );

	  if ( adb_tavl_insert( new_bh, &new_range->node, new_range ) )  /* stange ... */
	    fprintf(stderr,"Insert failed \a"); 
#if 0
	  Adb_data_show( new_range , file );
#endif	  
	  to_free = adb_tavl_delete( old_bh, old_range );
#if 0
	  Adb_check_bh( old_bh, NULL );
#endif
	  assert ( to_free != NULL );
	  Adb_free( to_free, vsize ,old_bh->m);
	  
	  /* set new parent's offset in child block */
	  child =  Adb_seg_load( tnx, file, adb_fd, ADB_SEG_SIZE, new_range->offset, 
				  ADB_TNX_DIRTY);
	  child->m->parent = new_bh->offset ;
	  /* Adb_seg_unload( tnx, child, file ); */
	}
    }
      
#if 0
  __debug__range_cmp = 1 ;
#endif
  Adb_check_bh(rbh, __FUNCTION__ , FALSE );

  ret = adb_tavl_delete(rbh, range);  assert(ret);
  fputs("deleting range (old) ", stderr);

  Adb_data_show(adb_range_to_rec(range), file);

  ret = adb_tavl_insert( rbh, &range->node, range ); assert( ! ret) ;
  ret = adb_tavl_insert( rbh, &new_r_range->node, new_r_range ); assert( ! ret) ;

  if ( rbh->m->level > 1 ) /* if content is RANGE */
    {
      adb_range_t *r ;
      adb_tavl_traverser_t tr ;
      /* 
       * set flags 
       */
      adb_tavl_t_init(old_bh, &tr);
      r = adb_tavl_t_first(old_bh, &tr);
      r->flag = ADB_RANGE_LEFT_INF ;
      r = adb_tavl_t_last( old_bh, &tr );
      r->flag = ADB_RANGE_RIGHT_INF ;

      adb_tavl_t_init(new_bh, &tr);
      r = adb_tavl_t_first(new_bh, &tr);
      r->flag = ADB_RANGE_LEFT_INF ;
      r = adb_tavl_t_last( new_bh, &tr );
      r->flag = ADB_RANGE_RIGHT_INF ;
    }
#if 1
  Adb_check_bh(old_bh, __FUNCTION__, EOF);
  Adb_check_bh(new_bh, __FUNCTION__, EOF);
  assert( count == ( old_bh->m->tree.tavl_count + new_bh->m->tree.tavl_count ));
  assert( old_bh->m->tree.tavl_count == limit );
#endif
  fprintf(stderr,"right range <left edge> :"); 
  Adb_data_show( adb_range_to_rec(new_r_range), file);
/*   fprintf(stderr,"right range <right edge> :"); Adb_data_show(range->right_edge, file); */

  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify( tnx, file );
#if TEX_DEBUG == 1
  adb_tex(tnx, file , "split-after" );
#endif /* 0 */
#endif /* BLOATWARE */
  return rbh ;
}

/*
 * Spliting +-INF segment S0 [left... ]
 *
 *  +INF
 *
 *       R0    	             R0		         R1
 *     	 |    	       	     | 		         |
 *	 |    	    ==>	     |		         |
 *	 v    		     v		         v
 *	 S0 +INF	     S0                  S1 +INF
 *     left... 		  left ... inf-1     last S0's ...
 *
 * new: R1 S1
 * 
 *  -INF
 *
 *       R0    	             R1		         R0
 *     	 |    	       	     | 		         |
 *	 |    	    ==>	     |		         |
 *	 v    		     v		         v
 *	 S0 -INF	     S1 -INF             S0 
 *     undef... 	undef ...          first S0's ...
 *
 * clear R0 -INF , set in S1
 * new: R1 S1 
 */

/* *rbh - range's space */
static void
Adb_seg_split_inf_tree_left(adb_tnx_t *tnx, adb_bh_t *old_bh, adb_file_t *file, 
			      adb_range_t *range)
{
  adb_bh_t *new_bh , *rbh = Adb_tnx_find_parent_bh(tnx, old_bh , file);
/*   void *tmp ; */
  adb_rec_t *tmp_rec = NULL ; /* first/last record of old bh */
  adb_rec_t *new_data = NULL , *tmp_rr ; /* first/last range of old bh */
  adb_range_t *new_range ;
  adb_range_flag_t old_flag = range->flag ;
  adb_range_t *range_reincarnated = NULL  ;
  const void *ret ;
  adb_tavl_traverser_t tr ;
  size_t new_size , rr_size , old_range_size ;
  int new_magic = ADB_MAGIC_TREE ;

  assert(tnx->magic == ADB_MAGIC_TNX );
  assert(rbh->magic == ADB_MAGIC_BH);
  assert(range->magic == ADB_MAGIC_RANGE);
  assert( rbh->m->level == old_bh->m->level + 1 );
  assert( range->flag == ADB_RANGE_LEFT_INF );
  assert ( rbh->m->level == 1 );

  assert( old_bh->m->tree.tavl_count > 3 );

  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);

#if TEX_DEBUG == 1
  adb_tex(tnx, file , "inf-tree-before-left" );
#endif /* 0 */

  assert(old_flag == ADB_RANGE_LEFT_INF );

  fprintf(stderr,"(%s) former range <left edge> :", __FUNCTION__); 
  Adb_data_show( adb_range_to_rec(range) , file);

  tmp_rec = adb_tavl_t_first( old_bh, &tr );
  tmp_rr = adb_tavl_t_next( old_bh, &tr);
  rr_size = Adb_vsize_get(file, tmp_rr) + file->meta->all_data_size ;

  old_range_size = Adb_vsize_get(file, adb_range_to_rec(range)) + 
    file->meta->all_data_size + sizeof(adb_range_t);

  range_reincarnated = Adb_range_new( tnx, rbh, range->offset, ADB_MAGIC_TREE, 
					rr_size + sizeof(adb_range_t));
  range_reincarnated->offset = range->offset ;
  range_reincarnated->flag = ADB_RANGE_FLAG_NULL ; /* clear, having two INF records
						      seems to be strange idea */
  /* undo if next malloc failed */
  Adb_malloc_register(file, range_reincarnated , rbh, rr_size + sizeof(adb_range_t)); 
  memcpy( adb_range_to_rec( range_reincarnated),
	  tmp_rr,
	  rr_size );
  
  new_size = Adb_vsize_get(file, tmp_rec) + file->meta->all_data_size ;
  /*
   * I don't care about new_range != NULL because Adb_range_new does longjmp
   * when malloc in Adb_cs_range() failed 
   */
  new_range = Adb_range_new( tnx, rbh, 0LL, new_magic, new_size + sizeof(adb_range_t));
  
  /* critical part passed so : */
  Adb_malloc_register_forget();

  new_bh = Adb_seg_add( tnx, file, &new_range->offset,
			ADB_MAGIC_TREE, old_bh->m->level, rbh->offset );

  range->flag = ADB_RANGE_FLAG_NULL ;
  new_range->flag = old_flag ; /* +-INF */
  memcpy( adb_range_to_rec(new_range) , tmp_rec, new_size );
  ret = adb_tavl_insert( rbh, &new_range->node, new_range ); assert(!ret);
  /* done with rbh's space */

  new_data = Adb_malloc(file, new_size, new_bh->m, __FUNCTION__);
  memcpy(new_data, tmp_rec, new_size );

  /* be kind to back-reference */
   Adb_link_update(tnx, file, old_bh, tmp_rec, new_bh, new_data );

  /* this record'll live in new range */
  ret = adb_tavl_delete( old_bh , tmp_rec ); assert(ret) ;
  Adb_free(file, (void *)tmp_rec, new_size , old_bh->m, __FUNCTION__);
  ret = adb_tavl_insert( new_bh, &new_data->node, new_data ); assert(!ret);
  /* done with old_ta */

  /* -INF - old range changed */
  adb_tavl_t_init(old_bh, &tr );
  tmp_rec = adb_tavl_t_first( old_bh, &tr );
  assert( tmp_rec == tmp_rr );

  ret = adb_tavl_delete( rbh, range ); assert(ret); /* Zap old */
  Adb_free(file, (void *)range,  old_range_size, rbh->m, __FUNCTION__);
  
  ret = adb_tavl_insert( rbh, &range_reincarnated->node, range_reincarnated ); 
  assert(!ret);



  range = range_reincarnated ;

  fprintf(stderr,"(%s) old range <left edge> :", __FUNCTION__); 
  Adb_data_show( adb_range_to_rec(range) , file);

/*   Adb_range_show( rbh, file , __PRETTY_FUNCTION__); */

  fprintf(stderr,"(%s) new (INF) range <left edge> :",__FUNCTION__); 
  Adb_data_show( adb_range_to_rec(new_range), file);

  Adb_show_one_range( range , rbh, file );
  Adb_show_one_range( new_range, rbh,  file );
  /* 2nd range item */

#if 0
  Adb_range_show( rbh, file , __PRETTY_FUNCTION__);
#endif /* 0 */
  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);

#if TEX_DEBUG == 1
  adb_tex(tnx, file , "inf-tree-after-left" );
#endif
  return ;
}

static void
Adb_seg_split_inf_tree_right(adb_tnx_t *tnx, adb_bh_t *old_bh, adb_file_t *file,
			       adb_range_t *range)
{
  adb_bh_t *new_bh , *rbh = Adb_tnx_find_parent_bh(tnx, old_bh , file);
/*   void *tmp ; */
  adb_rec_t *tmp_rec = NULL ; /* first/last record of old bh */
  adb_rec_t *new_data = NULL ; /* first/last range of old bh */
  adb_range_t *new_range ;
  adb_range_flag_t old_flag = range->flag ;
  const void *ret ;
  adb_tavl_traverser_t tr ;
  size_t new_size  ;
  int new_magic = ADB_MAGIC_TREE ;

  assert(tnx->magic == ADB_MAGIC_TNX );
  assert(rbh->magic == ADB_MAGIC_BH);
  assert(range->magic == ADB_MAGIC_RANGE);
  assert( rbh->m->level == old_bh->m->level + 1 );
  assert(range->flag == ADB_RANGE_RIGHT_INF );

  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);

#if TEX_DEBUG == 1
  adb_tex(tnx, file , "inf-tree-before" );
#endif /* 0 */
  assert ( rbh->m->level == 1 );

  tmp_rec = adb_tavl_t_last( old_bh , &tr );
  new_size = Adb_vsize_get(file, tmp_rec) + file->meta->all_data_size ;
  /*
   * I don't care about new_range != NULL because Adb_range_new does longjmp
   * when malloc in Adb_cs_range() failed 
   */
  new_range = Adb_range_new( tnx, rbh, 0LL, new_magic, new_size + sizeof(adb_range_t));
  
  /* critical part passed so : */
  Adb_malloc_register_forget();

  new_bh = Adb_seg_add( tnx, file, &new_range->offset,
			 ADB_MAGIC_TREE, old_bh->m->level, rbh->offset );

  range->flag = ADB_RANGE_FLAG_NULL ;
  new_range->flag = old_flag ; /* +-INF */
  memcpy( adb_range_to_rec(new_range) , tmp_rec, new_size );
  ret = adb_tavl_insert( rbh, &new_range->node, new_range ); assert(!ret);
  /* done with rbh's space */

  new_data = Adb_malloc(file, new_size, new_bh->m, __FUNCTION__);
  memcpy(new_data, tmp_rec, new_size );

   Adb_link_update(tnx, file, old_bh, tmp_rec, new_bh, new_data );

  /* this record'll live in new range */
  ret = adb_tavl_delete( old_bh , tmp_rec ); assert(ret) ;
  Adb_free(file, (void *)tmp_rec, new_size , old_bh->m, __FUNCTION__);
  ret = adb_tavl_insert( new_bh, &new_data->node, new_data ); assert(!ret);
  /* done with old_bh */

  fprintf(stderr,"(%s) old range <left edge> :", __FUNCTION__); 
  Adb_data_show( adb_range_to_rec(range) , file);

/*   Adb_range_show( rbh, file , __PRETTY_FUNCTION__); */

  fprintf(stderr,"(%s) new (INF) range <left edge> :",__FUNCTION__); 
  Adb_data_show( adb_range_to_rec(new_range), file);
  Adb_show_one_range( range , rbh, file );
  Adb_show_one_range( new_range, rbh,  file );
  /* 2nd range item */

#if 0
  Adb_range_show( rbh, file , __PRETTY_FUNCTION__);
#endif /* 0 */

  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);

#if TEX_DEBUG == 1
  adb_tex(tnx, file , "inf-tree-after-right" );
#endif
  return ;
}

static void
Adb_seg_split_inf_tree(adb_tnx_t *tnx, adb_bh_t *old_bh, adb_file_t *file)
{
  adb_range_t *range = Adb_range_find_parent_range(tnx, old_bh, file);

  Adb_seg_dirty(tnx, old_bh);

  if ( range->flag == ADB_RANGE_LEFT_INF )
    {
      Adb_seg_split_inf_tree_left(tnx, old_bh, file, range );
    }
  else
    {
      assert( range->flag == ADB_RANGE_RIGHT_INF );
      Adb_seg_split_inf_tree_right(tnx, old_bh, file, range );
    }
  return ;
}

static void
Adb_seg_split_inf_range(adb_tnx_t *tnx, adb_bh_t *old_bh, adb_file_t *file)
{
  adb_bh_t *new_bh , *rbh = Adb_tnx_find_parent_bh(tnx, old_bh , file);
/*   void *tmp ; */
  adb_range_t *tmp_range ;
  adb_range_t *new_range /* rbh - 1 level */, *new_rbh_range /* rbh level */;
  adb_range_t *range = Adb_range_find_parent_range(tnx, old_bh, file);
  adb_range_flag_t old_flag = range->flag ;
  
  const void *ret ;
  adb_tavl_traverser_t tr ;
  size_t new_size ;
  int new_magic = ADB_MAGIC_RANGE ;

  size_t size ;
  int i ;
  int flags[3] ;
  void *(*op)() ;
  adb_bh_t *child ;


  assert(tnx->magic == ADB_MAGIC_TNX );
  assert(rbh->magic == ADB_MAGIC_BH);
  assert(range->magic == ADB_MAGIC_RANGE);
  assert( rbh->m->level == old_bh->m->level + 1 );
#if TEX_DEBUG == 1
  adb_tex(tnx, file , "inf-range-before" );
#endif /* 0 */

  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);


  adb_tavl_t_init(old_bh, &tr );
  if ( old_flag == ADB_RANGE_RIGHT_INF ) /* +INF */
    {
      /* we are at +INF so upper level bh should be copy of THIRD from behind */
      adb_tavl_t_last( old_bh , &tr ); 
      adb_tavl_t_prev( old_bh , &tr );
      tmp_range = adb_tavl_t_prev( old_bh , &tr );
      op = adb_tavl_t_last ;
      flags[0] = ADB_RANGE_RIGHT_INF ;
      flags[1] = ADB_RANGE_FLAG_NULL ;
      flags[2] = ADB_RANGE_LEFT_INF ;
    }
  else /* -INF */
    {
      assert(old_flag == ADB_RANGE_LEFT_INF );
      tmp_range =  adb_tavl_t_first( old_bh, &tr ); /* 1st */
      adb_tavl_t_next( old_bh, &tr ); /* 2nd */
      adb_tavl_t_next( old_bh, &tr ); /* 3rd */
#if 0
      tmp_rr = adb_tavl_t_next( old_bh, &tr ); /* 4th - new edge for old bh */
      rr_size = Adb_vsize_get(file, adb_range_to_rec(tmp_rr)) + file->all_data_size +
	sizeof(adb_range_t);
      range_reincarnated = Adb_range_new(tnx, rbh, range->offset, ADB_MAGIC_RANGE,
					   rr_size);
      memcpy( range_reincarnated, range , sizeof(*range));
      range_reincarnated->flag = ADB_RANGE_FLAG_NULL ; /* clear, having two INF records
							  seems to be strange idea */
      /* undo if next malloc failed */
      Adb_malloc_register(range_reincarnated, rbh, rr_size ); 
      memcpy( adb_range_to_rec( range_reincarnated), 
	      adb_range_to_rec(tmp_rr), rr_size - sizeof(adb_range_t ));
#endif /* 0 */
      op = adb_tavl_t_first ;
      flags[0] = ADB_RANGE_LEFT_INF ;
      flags[1] = ADB_RANGE_FLAG_NULL ;
      flags[2] = ADB_RANGE_RIGHT_INF ;
    }

  new_size = Adb_vsize_get(file, adb_range_to_rec(tmp_range)) + 
    file->meta->all_data_size + sizeof(adb_range_t);
  
  new_rbh_range = Adb_range_new( tnx, rbh, 0LL, new_magic, new_size);
  /* critical part passed so : */
  Adb_malloc_register_forget();

  /*
   * I don't care about new_range != NULL etc. because Adb_range_new does longjmp
   * when malloc in Adb_cs_range() failed 
   */
  new_bh = Adb_seg_add( tnx, file, &new_rbh_range->offset,
			 ADB_MAGIC_RANGE, old_bh->m->level, rbh->offset );

  range->flag = ADB_RANGE_FLAG_NULL ;
  new_rbh_range->flag = old_flag ; /* +-INF */
  memcpy( adb_range_to_rec(new_rbh_range) , 
	  adb_range_to_rec(tmp_range), 
	  new_size - sizeof(adb_range_t)); /* don't overwrite range part !!! */

  ret = adb_tavl_insert( rbh, &new_rbh_range->node, new_rbh_range ); assert(!ret);
  /* rbh's space done */

  /* spliting range block */

  for ( i = 0 ; i < 3 ; ++i )
    {
      adb_tavl_t_init(old_bh, &tr );
      tmp_range = op( old_bh, &tr );  
      assert(tmp_range->magic == ADB_MAGIC_RANGE );
      
      size = Adb_vsize_get(file, adb_range_to_rec(tmp_range)) + 
	file->meta->all_data_size + sizeof(adb_range_t);

      new_range = Adb_malloc( file, size, new_bh->m , __FUNCTION__ );
	  
      memcpy( new_range, tmp_range, size );
      ret = adb_tavl_delete( old_bh, tmp_range ) ; assert(ret) ;
      Adb_free(file, (void *)tmp_range, size, old_bh->m, __FUNCTION__);
      new_range->flag = flags[i] ;

      ret = adb_tavl_insert( new_bh, &new_range->node, new_range ); assert( !ret);
	  
      /* set new parent */
      child = Adb_seg_load( tnx, file, new_range->offset, ADB_TNX_DIRTY);
      child->m->parent = new_bh->offset ;
    }

  /* set flag on 1st/last range of old_bh */
  tmp_range = op( old_bh, &tr );
  assert( ! tmp_range->flag );
  tmp_range->flag = old_flag ;
  /* done with old_bh */

  /* old range change , makes sense only if old_flag == ADB_RANGE_LEFT_INF,
   * but who cares ? 
   */

/*   if ( old_flag == ADB_RANGE_LEFT_INF ) */
  range = Adb__range_reincarnate(tnx, rbh, old_bh, range );

#if 0
  if ( old_flag == ADB_RANGE_LEFT_INF )
    {
      /* just for sake of integrity testing overhead */
      adb_tavl_t_init(old_bh, &tr );
      tmp_range =  adb_tavl_t_first( old_bh, &tr );
      assert( tmp_range == tmp_rr );

      ret = adb_tavl_delete( rbh, range ); assert(ret); /* zap old range */
      ret = adb_tavl_insert( rbh, &range_reincarnated->node, range_reincarnated ); 
      assert(!ret);

      Adb_free( (void *)range, rr_size, rbh->m, __FUNCTION__);
      range = range_reincarnated ;
    }
#endif /* 0 */
  fprintf(stderr,"(%s) old range <left edge> :", __FUNCTION__); 
  Adb_data_show( adb_range_to_rec(range) , file);

  fprintf(stderr,"(%s) new (INF) range <left edge> :",__FUNCTION__); 
  Adb_data_show( adb_range_to_rec(new_range), file);

  Adb_show_one_range( range , rbh, file );
  Adb_show_one_range( new_range, rbh,  file );
  /* 2nd range item */

  Adb_check_bh( old_bh, __FUNCTION__, EOF );
  Adb_check_bh( new_bh, __FUNCTION__, EOF );
#if TEX_DEBUG == 1
  adb_tex(tnx, file , "inf-range-after" );
#endif /* 0 */
  if ( Adb_debug & ADB_DEBUG_INDEX )
    Adb_verify(tnx,file);

#if 0
  Adb_range_show( rbh, file , __PRETTY_FUNCTION__);
#endif /* 0 */
#if 0
  adb_tex(tnx, file , "inf-range-after" );
#endif /* 0 */
  return ;
}

adb_bh_t *
Adb_seg_split_inf(adb_tnx_t *tnx, adb_bh_t *old_bh, adb_file_t *file)
{
  adb_bh_t *rbh = Adb_tnx_find_parent_bh(tnx, old_bh , file);
/*   void *tmp ; */
  adb_range_t *range = Adb_range_find_parent_range(tnx, old_bh, file);


  assert(tnx->magic == ADB_MAGIC_TNX );
  assert(rbh->magic == ADB_MAGIC_BH);
  assert(range->magic == ADB_MAGIC_RANGE);
  assert( rbh->m->level == old_bh->m->level + 1 );

  fprintf(stderr,"=== %s === block at level %d full\n",__func__, old_bh->m->level);
  {
    char *buf = NULL ;
    asprintf(&buf, "%s (old_bh)", __func__ );
    Adb_check_bh( old_bh, buf, EOF );
    free(buf);
  }

  if ( rbh->m->level == 1 )
    Adb_seg_split_inf_tree(tnx, old_bh, file);
  else
    Adb_seg_split_inf_range(tnx, old_bh, file);

  return NULL ;

}

void
adb_range_zap_empty(adb_tnx_t *tnx, adb_file_t *file, adb_bh_t *bh)
{
  if ( bh->m->tree.tavl_count == 0 )
    {
      adb_bh_t *parent ;
      adb_range_t *parent_range, *parent_next = NULL ;
      adb_range_flag_t flag ;
      adb_tavl_traverser_t tr ;

      if ( Adb_debug )
	fprintf(stderr,"%s Zero tree @ %p\n", __FUNCTION__, bh->m);

      parent = Adb_tnx_find_parent_bh(tnx, bh, file );
      parent_range = Adb_range_find_parent_range(tnx, bh, file);
	  
      adb_tavl_t_find( parent, &tr, parent_range);
      flag = parent_range->flag ;

      switch ( parent_range->flag )
	{
	case ADB_RANGE_LEFT_INF:
	  parent_next = adb_tavl_t_next(parent, &tr);
	  break ;
	case ADB_RANGE_RIGHT_INF:
	  parent_next = adb_tavl_t_prev(parent, &tr);
	  break ;
	default:
	  break ;
	}
      Adb_seg_dirty(tnx, parent);
      adb_tavl_delete(parent, parent_range);
      /* no free() here, whole block is EMPTY */

      /* FIXME: remember freed block !!!! */
      if ( parent_next )
	parent_next->flag = flag ;

      adb_range_zap_empty(tnx, file, parent);
    }
}
