/* AVLDB - AVL DataBase library,   meta-scheme functions

   Copyright (C) 2002,2003 Petr Silhavy <silhavy@mef.cz>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.

*/

/* superblock
 * 
 * 0x30000000 - 0x30000fff superblock magic   - 0x1000
 * 0x30001000 - 0x30001fff meta  magic        - 0x1000
 * 0x30002000 - 0x30011fff meta bitmap       - 0x10000
 * 0x30012000 - 0x31011fff meta data        0x01000000
 * 0x31012000 - 0x31111fff data bitmap        0x100000  1TB ~ 0x20000 - 8TB - *unused*
 * 0x31112000 -
 *
 *
 *
 *
 *
 *
 *
 * MM - meta 
 * 0x00000000 - 0x00000fff base / magic
 * 0x00001000 - 0x000fffff map
 * 0x01000000 - 0x
 *
 *
 *
 * MM - seg
 * 0x00000000 - 0x00000fff  base / magic 
 * 0x00001000 - 0x00001fff  map 
 * 0x00002000 - 0x00102000  pool
 *
 *
 * 
 *
 *
 *
 *
 *
 *
 */

#define  __ADB 1
#include "adb.h"

#define BACK_LINK "$back_link"
#define FREE_LIST "$free_list"
#define LINK "$link"

adb_bh_t *
Adb_new_fake_bh(adb_db_t *db, int (*cmp)(), adb_tavl_table_t *tree )
{
 adb_bh_t *bh = Malloc0(sizeof(*bh));
 bh->magic = ADB_MAGIC_BH ;
 bh->db = db ;
 bh->file = NULL ;
 bh->addr = db->addr ;
 bh->cmp = cmp ;
 bh->tree = tree ;
 return bh ;
}

void
Adb_free_fake_bh(adb_bh_t *bh)
{
  free(bh);
}


/* unsigned char *adb_sb, *adb_mm ; */

void
adb_sweep(adb_db_t *db)
{
  fputs("Removing semaphores ...",stderr);
  fputs( semctl( db->semid, 0, IPC_RMID, 0) == EOF ? "FAIL": "Ok" , stderr);
  fputc('\n',stderr);

  fputs("Removing lock shared memory ...",stderr);
  fputs( shmctl( db->shmid, IPC_RMID, 0) == EOF ? "FAIL": "Ok" , stderr);
  fputc('\n',stderr);
#if 0
  fputs("Removing login shared memory ...",stderr);
  fputs( shmctl( db->ptsid, IPC_RMID, 0) == EOF ? "FAIL": "Ok" , stderr);
#endif /* 0 */
  fputc('\n',stderr);
}

static adb_db_t *sh_db ;
static char *sh_lock ;

static void
Adb_bye()
{
  fputs("Killed! Why?\n",stderr);
  adb_sweep(sh_db);

  /* TODO: Kill'em all !!! */

  if (unlink(sh_lock))
    ADB_ERROR("Can't unlink `%s' do it manually: `rm %s'\n", sh_lock, sh_lock );
  fflush(stdout);
  fflush(stderr);
  exit(0);
}


void
adb_stop(char *dbname)
{
  char *lock = Adb_name(dbname,"lock");
  int lfd ;
  int mode , pid ;
  if ( (lfd = open(lock, O_RDONLY)) == EOF )
    {
      ADB_WARNING("Can't open lock file : `%s'\n", lock );
      return ;
    }

  read(lfd, &mode, sizeof(mode));
  read(lfd, &pid, sizeof(pid));
  close(lfd);

  if (kill(pid, SIGTERM))
    ADB_WARNING("Can't kill pid %d",pid);
}

void
adb_close(adb_db_t *db )
{
  struct stat statbuf ;
#if 0
  adb_mm_t *m = db->mm ;
  byte_t *last_free ;
  byte_t *map = (void *) m + ADB_MAGIC_SIZE ;
#endif
/*   byte_t *pool = (void *) m + ADB_MAGIC_SIZE + m->map_size ; */
  int r ;

  int i ;
  for ( i = 0 ; i < ADB_EXT_MAX ; ++i )
    {
      if ( db->fds[i] )
	{
	  r = fstat ( db->fds[i] , &statbuf );
	  assert(r == 0);
	}
    }

  r = msync( db->sb , ADB_META_SIZE + ADB_META_SB_SIZE , MS_SYNC );
  assert(r != EOF );

  Adb_meta_rw(db);
  db->sb->last_range_id = db->last_range_id ; /* FIXME AI/BI */
  db->sb->last_link_id = db->last_link_id ;
  Adb_meta_ro(db);

#if 0
  if ( ! ( last_free = memchr(map, 0, m->map_size )) )
    last_free = map + m->map_size ;
  fprintf(stderr,"Meta used : %d%% \n", (( last_free - map ) * 100 / m->map_size ) );
  fprintf(stderr,"Files     : %d \n", db->files->tavl_count);
  
  system("cat /proc/`pidof adb-test`/maps");
  sleep(1);
#endif /* 0 */


}

adb_mm_t *
adb_meta_mm_init(unsigned char *base , size_t size ) /* void *base ??? */
{
  adb_mm_t *m = ( adb_mm_t *)base ; 
  int r ;

  m->magic = ADB_MAGIC_MM ;
  m->size = size ;
  m->map_size = ADB_META_MAP_SIZE; /* size / 256 - sizeof(*m) ; */
/*   m->map = base + ADB_META_MAGIC_SIZE ; */
/*   m->pool = base + ADB_META_SB_SIZE ; */
  memset( (void *)m + ADB_META_MAGIC_SIZE , 0, m->map_size );
  r =  msync( base ,ADB_META_SB_SIZE, MS_SYNC );
  assert(r != EOF);
  /* debug */
  memset( (void *)m + ADB_META_SB_SIZE, 0 , m->size) ;
  r = msync( (void *)m + ADB_META_SB_SIZE , m->size, MS_SYNC );
  assert( r != EOF);

  return m ;
}


void
adb_resign()
{
  kill(getpid(), SIGBUS );
}

int
Adb_file_cmp(const void *a, const void *b , 
	   const void *param __attribute__ ((__unused__)),
	   int uuu UNUSED )
{

  const adb_meta_file_t *file_a = a ;
  const adb_meta_file_t *file_b = b ;
/*   fprintf(stderr,"file_a->name %s file_b->name %s\n",file_a->name, file_b->name ); */

  return strcmp(file_a->name, file_b->name) ;
}
#if 0
void
adb_exit_wrap()
{
  adb_exit(adb_fd, adb_sb);
}
#endif

adb_db_t *
adb_creat(char *dbname, char *ext[] ,int multi )
{
  struct stat sb ;
  char *db ;
  adb_db_t *DB ;
  int i ;

  DB = Malloc0(sizeof(*DB));

  if ( ! stat ( dbname, &sb )  )
    {
      adb_db_t *old ;
      old = adb_open(dbname,TRUE);
      adb_close(old);
      adb_sweep(old);

      if ( unlink(dbname))
	ADB_ERROR("Can't unlink `%s'", dbname);
    }

  if ( ( DB->fds[0] = open(dbname, O_CREAT | O_RDWR , 0666 )) < 0)
    ADB_ERROR("Can't creat `%s",dbname);

  db = Adb_name_db(dbname);

  printf("Database: %s\n",db);

  DB->name = db ;
/*   DB->magic = ADB_MAGIC_DB ; */

#ifdef USE_FTRUNCATE
  if ( ftruncate(DB->fds[0], ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE) == EOF)
    ADB_BUG("Can't extend database");
#else
  if ( lseek( DB->fds[0], ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE, SEEK_END) == EOF)
    ADB_BUG("Can't extend database");
#endif 

  DB->sb = (adb_sb_t *) mmap(ADB_SB_BASE ,
			     ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE ,
			     PROT_READ | PROT_WRITE ,
			     /* MAP_FIXED |  */MAP_SHARED, 
			     DB->fds[0] ,0 );
  if ( DB->sb == (void *)EOF ) ADB_BUG("Can't mmap");
  DB->mm = adb_meta_mm_init( ((unsigned char *)DB->sb) + ADB_SB_SIZE , ADB_META_SIZE );
  if ( DB->mm == (void *)EOF ) ADB_BUG("Can't initialize MM");

  DB->sb->magic = ADB_MAGIC_SB ;
  DB->sb->meta = DB->mm ;
/*   DB->sb->__nick = _adb_name_db_nick( DB->name ); */
  DB->addr = (unsigned int) DB->sb ;
  DB->files = &DB->sb->__files ;

  {
    char *np ;
    char *n = Adb_name_db_nick( DB->name , &np );
    DB->nick =  Adb_malloc(NULL, strlen(n) + 1, DB->mm, __func__ );
    DB->sb->__nick = (void *)DB->nick - DB->addr ;

    strcpy(DB->nick, n);

    free(np); /* malloced */
  }

/*   DB->ta = __adb_new_fake_ta( DB, DB->addr, __file_cmp, &DB->ta); */

  memset(DB->sb->ai_file, 0, sizeof(DB->sb->ai_file));
  memset(DB->sb->bi_dir, 0, sizeof(DB->sb->bi_dir));

  if ( multi )
    adb_lock_init( DB, dbname );

  for ( i = 0 ; ext && ext[i] && i < ADB_EXT_MAX ; ++i )
    {
      char *e ;
      DB->fds[i+1] = open(ext[i], O_CREAT | O_RDWR, 0666);
      if ( DB->fds[i+1] == EOF )
	ADB_ERROR("Can't open database extend `%s'",ext[i]);
      e = Adb_malloc(NULL, strlen(ext[i])+1, DB->mm ,__FUNCTION__ );
      strncpy(e, ext[i], ADB_NAME_MAX );
      DB->sb->ext[i] = (void *)e - DB->addr ;
    }
  DB->nfd = i + 1 ; /* # of open fd */

  DB->files_bh = Adb_new_fake_bh( DB,  Adb_file_cmp, DB->files ); 
  DB->sb->last_id = 0 ;
  adb_tavl_create(DB->files_bh->tree);

  /* internal files */
  {
    adb_meta_file_t *m ;

    m = adb_create_file_by_name(DB,FREE_LIST,
				"offset", ADB_FIELD_LONG_LONG, 0, TRUE,
				/* nth fd of database fds */
				"nfd", ADB_FIELD_INT, 1, TRUE ,
				"size", ADB_FIELD_INT, 0, FALSE,
				NULL );
    DB->free_list = adb_file_open(DB, FREE_LIST);

    m = adb_create_file_by_name(DB, BACK_LINK, 
				"__mfile", ADB_FIELD_PTR_DIFF, 0, TRUE,
				"dst_bh_offset", ADB_FIELD_LONG_LONG, 1, TRUE,
				"dst_rec_offset",ADB_FIELD_UNSIGNED_INT, 2, TRUE,
				"id", ADB_FIELD_UNSIGNED_LONG_LONG, 3, TRUE,
				NULL);
    DB->back_link = adb_file_open(DB, BACK_LINK );

    m = adb_create_file_by_name(DB, LINK,
				/* 
				 * index part 
				 */
				"id", ADB_FIELD_UNSIGNED_LONG_LONG, 0, TRUE, 
				"__mfile", ADB_FIELD_PTR_DIFF, 1, TRUE,
				/* datablock offset */
				"dst_bh_offset", ADB_FIELD_LONG_LONG, 2, TRUE ,
				/* record offset */
				"dst_rec_offset", ADB_FIELD_UNSIGNED_INT, 3, TRUE,
				NULL );
    DB->link = adb_file_open(DB, LINK );
  }  

  Adb_meta_ro(DB);

  return DB ;
}

void
Adb_mkdir(char *dir, int mode)
{
  struct stat sb ;
  if ( stat(dir, &sb))
    {
      if ( mkdir(dir, mode) )
	ADB_ERROR("Can't create directory `%s'\n",dir);
    }
  else
    if ( ! S_ISDIR(sb.st_mode ))
      ADB_ERROR("Not a directory `%s'\n",dir);
}

void
adb_bi_start(adb_db_t *db, char *bi_dir)
{
  if ( ! bi_dir )
    ADB_ERROR("NULL BI directory");

  Adb_meta_rw(db);

  strncpy(db->sb->bi_dir, bi_dir , ADB_NAME_MAX );
  db->sb->mode |= ADB_MODE_BI ;

  Adb_mkdir(bi_dir, 0777);

  printf("BI directory : %s\n",bi_dir);
  if ( db->lock_sb )
    db->lock_sb->mode |= ADB_MODE_BI ;

  Adb_meta_ro(db);
}


void
adb_ai_start(adb_db_t *db, char *ai_file, char *ai_tnx_dir)
{
  if ( ! ai_file )
    ADB_ERROR("NULL AI file");

  Adb_meta_rw(db);
  strncpy(db->sb->ai_file, ai_file , ADB_NAME_MAX );
  strncpy(db->sb->ai_dir, ai_tnx_dir, ADB_NAME_MAX );

  Adb_mkdir( ai_tnx_dir, 0777);

  db->sb->mode |= ADB_MODE_AI ;
  printf("AI file      : %s\n", ai_file);
  printf("AI directory : %s\n", ai_tnx_dir);

  if ( db->lock_sb )
    db->lock_sb->mode |= ADB_MODE_AI ;

  if ( ( db->ai_fd = open(db->sb->ai_file, O_RDWR|O_CREAT|O_APPEND|O_SYNC, 0666)) < 0 )
    ADB_ERROR("Can't open AI file '%s`", db->sb->ai_file);

  Adb_meta_ro(db);
}


void
Adb_ai_stop(adb_db_t *db)
{
  db->sb->mode &= ~ADB_MODE_AI ;

  if ( db->lock_sb )
    db->lock_sb->mode &= ~ADB_MODE_AI ;
}

void
adb_ai_stop(adb_db_t *db)
{
  Adb_meta_rw(db);
  Adb_ai_stop(db);
  Adb_meta_ro(db);
}

void
Adb_bi_stop(adb_db_t *db)
{
  db->sb->mode &= ~ADB_MODE_BI ;

  if ( db->lock_sb )
    db->lock_sb->mode &= ~ADB_MODE_BI ;
}

void
adb_bi_stop(adb_db_t *db)
{
  Adb_meta_rw(db);
  Adb_bi_stop(db);
  Adb_meta_ro(db);
}

void
Adb_meta_rw(adb_db_t *db)
{
  munmap( db->sb , ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE );
  db->sb = (adb_sb_t *) mmap( db->sb ,
			      ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE ,
			      PROT_READ | PROT_WRITE  ,
			      MAP_FIXED | MAP_SHARED, 
			      db->fds[0] ,0 );
 if ( db->sb == (void *)EOF)
   ADB_BUG("mmap");
#ifdef HAVE_MADVISE
 madvise(db->sb, ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE , MADV_RANDOM);
#endif /* HAVE_MADVISE */
}

void
Adb_meta_ro(adb_db_t *db)
{
  msync( db->sb, ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE, MS_SYNC );
  munmap( db->sb , ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE );
  db->sb = (adb_sb_t *) mmap( db->sb ,
			      ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE ,
			      PROT_READ /* | PROT_WRITE  */ ,
			      MAP_FIXED | MAP_SHARED, 
			      db->fds[0] ,0 );
 if ( db->sb == (void *)EOF)
   ADB_BUG("mmap");
}

void
adb_meta_open(adb_db_t *db)
{
  adb_tnx_hdr_t *hdr = (void *)zbuf ;
  unsigned long dest_len = sizeof(zbuf) - sizeof(*hdr);
  
  unsigned char *z = (unsigned char *) zbuf + sizeof(*hdr), *zb = NULL , *zp = zbuf;
  
  char *bi = NULL ;

  int r ;

  Adb_lock_claim_sb_with(db, ADB__LOCK_META);
  Adb_meta_rw(db);

  r = compress2 (z, &dest_len, (char *)db->sb, 
		 ADB_ALL_META_SIZE /* => ta->size */ , 1);

  if ( r != Z_OK )
    {
      zb = malloc(ADB_ALL_META_SIZE);
      z = (unsigned char *) zbuf + sizeof(*hdr);
      hdr = (void *)zb ;
      dest_len = ADB_ALL_META_SIZE - sizeof(*hdr);
      zp = zb ;
      r = compress2 (z, &dest_len, (char *)db->sb, 
		 ADB_ALL_META_SIZE /* => ta->size */ , 1);
      assert(r == Z_OK);
    }

  if ( Adb_debug & ADB_DEBUG_VM )
    {
      fprintf(stderr,"%s dest_len: %lu compress ratio %.2f\n",__func__, dest_len,
	       dest_len * 100.0 / (float) ADB_ALL_META_SIZE /* => ta->size */ );
    }

  asprintf(&bi,"%s.meta.bi", db->name);
  if ( (db->meta_bi_fd = open(bi, O_RDWR | O_CREAT | O_TRUNC | O_SYNC, 0600 )) == EOF )
    ADB_ERROR("Can't open meta-BI `%s'\n",bi);

  db->meta_bi_name = bi ;

  hdr->magic = ADB_MAGIC_XI_HDR ;
  hdr->offset = 0 ;
  hdr->size = dest_len ;
  hdr->nfd = 0 ;

  write(db->meta_bi_fd, zp, sizeof(*hdr) + dest_len );

  if ( zb )
    free(zb);
}

void
adb_meta_commit(adb_db_t *db)
{
  adb_tnx_hdr_t *hdr = (void *)zbuf ;
  int r ;
  unsigned char *z = (unsigned char *) zbuf + sizeof(*hdr), *zb = NULL ;
  unsigned long dest_len = sizeof(zbuf) - sizeof(*hdr);
  unsigned char *zp = zbuf ;
  
  if ( db->meta_bi_fd && db->meta_bi_name)
    {
      close(db->meta_bi_fd);
      r = unlink(db->meta_bi_name); assert(r != EOF);
      free(db->meta_bi_name);

      Adb_lock_claim_sb_with(db, ADB__LOCK_AI);
      
      r = compress2 (z, &dest_len, (char *)db->sb, 
		     ADB_ALL_META_SIZE /* => ta->size */ , 1);
      
      if ( r != Z_OK )
	{
	  zb = malloc(ADB_ALL_META_SIZE);
	  z = (unsigned char *) zbuf + sizeof(*hdr);
	  hdr = (void *)zb ;
	  dest_len = ADB_ALL_META_SIZE - sizeof(*hdr);
	  zp = zb ;
	  
	  r = compress2 (z, &dest_len, (char *)db->sb, 
			 ADB_ALL_META_SIZE /* => ta->size */ , 1);
	  assert(r == Z_OK);
	}
      
      hdr->magic = ADB_MAGIC_TNX_AI ;
      hdr->type = ADB_TNX_TYPE_BLOCK_META ;
      hdr->offset = 0LL ;
      hdr->nfd = 0 ;
      hdr->orig_size = ADB_ALL_META_SIZE ;
      hdr->size = dest_len ;
      
      if ( db->sb->mode & ADB_MODE_AI )
	write(db->ai_fd, zp, sizeof(*hdr) + dest_len );
      
      Adb_lock_free_sb_with(db, ADB__LOCK_AI);
      
      Adb_meta_ro(db);
      Adb_lock_free_sb_with(db, ADB__LOCK_META);
      
      if ( zb )
	free(zb);
    }
}

void
adb_meta_abort(adb_db_t *db)
{
  off_t off , curr;
  void *bi ;
  int ret ;
  unsigned char *z ;
  adb_xi_hdr_t *hdr ;
  unsigned long dest_len = ADB_ALL_META_SIZE ;

  curr = lseek(db->meta_bi_fd, 0, SEEK_CUR);
  off = lseek(db->meta_bi_fd, 0, SEEK_SET);
  assert( off != EOF );
  
  bi = mmap( NULL ,
	     curr ,
	     PROT_READ ,
	     /* MAP_FIXED | */ MAP_PRIVATE, 
	     db->fds[0] ,0 );

  hdr = (void *)bi ;
  assert(hdr->magic == ADB_MAGIC_XI_HDR );
  
  z = (char *)bi + sizeof(*hdr);

  ret = uncompress ((char *)db->sb, &dest_len, z, curr - sizeof(*hdr));
  assert(ret == Z_OK);
   
  ret = munmap(bi, ADB_ALL_META_SIZE); assert(ret != EOF);

  close(db->meta_bi_fd);
  ret = unlink(db->meta_bi_name); assert(ret != EOF);
  free(db->meta_bi_name);
  Adb_meta_ro(db);

  Adb_lock_free_sb_with(db, ADB__LOCK_META);
}

adb_db_t *
adb_open(char *dbname, int multi)
{
  char *db ;
  adb_db_t *DB ;
  int i ;

  adb_page_size = (size_t) sysconf (_SC_PAGESIZE);

/*   signal(SIGINT, adb_exit_wrap); */

  printf("Page size: %d\n",adb_page_size);
  printf("Sizeof off_t %d\n",sizeof(off_t));

  db = Adb_name_db(dbname);

  printf("Database: %s\n",db);

  DB = Malloc0(sizeof(*DB));
  DB->name = db ;
/*   DB->magic = ADB_MAGIC_DB ; */

  if (( DB->fds[0] = open(dbname, O_RDWR , 0666 )) < 0 )
    ADB_ERROR("Can't open `%s",dbname);
  DB->sb = (adb_sb_t *) mmap( ADB_SB_BASE ,
			      ADB_SB_SIZE + ADB_META_SIZE + ADB_META_SB_SIZE ,
			      PROT_READ | PROT_WRITE ,
			      /* MAP_FIXED | */ MAP_SHARED, 
			      DB->fds[0] ,0 );
  if ( DB->sb == (void *)EOF) ADB_BUG("mmap");
/*   DB->mm = DB->sb->meta ; */
  DB->mm = (void *)DB->sb + ADB_SB_SIZE ;

  DB->addr = (unsigned int) DB->sb ;
  DB->files = &DB->sb->__files ;

  DB->nick = (void *)DB->sb->__nick + DB->addr ;

/*   DB->files->addr = DB->addr ; */

  DB->last_range_id = DB->sb->last_range_id ;
  DB->last_link_id = DB->sb->last_link_id ;

  if ( multi )
    adb_lock_init( DB, dbname );

  if ( ADB_MODE(DB) & ADB_MODE_AI )
    {
      printf("AI file %s\n", DB->sb->ai_file );
      if ( ( DB->ai_fd = 
	     open( DB->sb->ai_file, O_RDWR|O_CREAT|O_APPEND|O_SYNC, 0666)) < 0 )
	ADB_ERROR("Can't open AI file '%s`", DB->sb->ai_file);
    }
  else puts("AI mode Off\n");

  if ( ADB_MODE(DB) & ADB_MODE_BI )
    {
      struct stat sb ;
      stat(DB->sb->bi_dir, &sb);

      if ( !S_ISDIR (sb.st_mode))
	{
	  unlink(DB->sb->bi_dir);
	  mkdir(DB->sb->bi_dir, 0666);
	}

      printf("BI directory : %s\n",DB->sb->bi_dir);

    }
  else puts("BI mode Off\n");

  for ( i = 0 ; DB->sb->ext[i] && i < ADB_EXT_MAX ; ++i )
    {
      char *e = (void *)DB->sb->ext[i] + DB->addr ;
      DB->fds[i+1] = open(e, O_RDWR, 0666);	
      if ( DB->fds[i+1] == EOF )
	ADB_ERROR("Can't open database extend `%s'",e);
    }

  DB->files = &DB->sb->__files ;
  DB->files_bh = Adb_new_fake_bh( DB, Adb_file_cmp, DB->files );

  DB->back_link = adb_file_open(DB, BACK_LINK );
  DB->free_list = adb_file_open(DB, FREE_LIST); 
  DB->link = adb_file_open(DB, LINK); 
  
  Adb_meta_ro(DB);

  fflush(stdout);

  return DB ;
}

adb_meta_file_t *
adb_create_file(adb_db_t *db, char *name, int n_fields)
{
  adb_meta_file_t *mfile ;
  int n = n_fields ? n_fields : 32 ;
  adb_file_t *order , *idx ;

  ADB_ERROK();

  mfile = Adb_malloc(NULL, sizeof(*mfile), db->mm ,__FUNCTION__ );
  memset(mfile,0,sizeof(*mfile));

  mfile->magic = ADB_MAGIC_META_FILE ;
  strncpy(mfile->name, name, ADB_NAME_MAX - 1 ) ;

  if ( adb_tavl_insert( db->files_bh , &mfile->node, mfile ) )
    {
      g_warning("*** file `%s' exists ***", name );
      Adb_free( NULL, (unsigned char *)mfile , sizeof(*mfile), db->mm, __FUNCTION__);
      return NULL ;
    }

  mfile->id = db->sb->last_id++ ;
  mfile->block_size = ADB_SEG_SIZE ; /* default */

  order = Adb_malloc( NULL, sizeof( adb_field_t *) * n, db->mm, __FUNCTION__);
  mfile->__order = ( void *)order - db->addr ;
  idx = Adb_malloc( NULL, sizeof( adb_field_t *) * n, db->mm ,__FUNCTION__);
  mfile->__idx = (void *)idx - db->addr ;
#if 0
  mfile->indexes_bh = __adb_new_fake_bh( db, __file_cmp, &mfile->indexes);
  mfile->fields_bh = __adb_new_fake_bh( db, __field_cmp, &mfile->fields);
#endif /* 0 */

  adb_tavl_create(&mfile->fields);

  adb_tavl_create(&mfile->indexes);

  mfile->idx_alloc = n ;
  mfile->idx_size = n ;
  mfile->order_size = n ;
/*   file->secondary = FALSE ; */
  mfile->__parent = NULL ;

  mfile->all_data_size = sizeof(adb_rec_t );

  return mfile ;
}

/* 
 *  va args:
 *
 * char *field_name, adb_field_type_t type, int len , ins is_idx 
 * and so on ...
 */

adb_meta_file_t *
adb_create_file_by_name(adb_db_t *db, char *name, ... )
{
  va_list ap ;

  int idx = 0 ;
  int len ;
  int is_idx ;
  adb_field_type_t type ;
  char *field ;
  int n ;
  adb_meta_file_t *file ;

  ADB_ERROK();

  va_start(ap, name ); 

  for ( n = 0 ; ( field = va_arg(ap , char * ) ) ; ++n )
    {
      type = va_arg(ap, int );
      len = va_arg(ap, int );
      is_idx = va_arg(ap , int );
    }
  file = adb_create_file(db, name, n ); /* file_open is NOT hidden here */
  
  va_start(ap, name ); 

  while ( ( field = va_arg(ap , char * ) ) )
    {
      type = va_arg(ap, int );
      len = va_arg(ap, int );
      is_idx = va_arg(ap , int );
      adb_create_field(db, file, field, type, len, is_idx > 0 ? idx++ : EOF );
    }

  return file ;
}

/*
 * va_args :
 *
 * char *field_name
 *
 */

adb_meta_file_t *
adb_create_index_by_name(adb_db_t *db, adb_meta_file_t *mfile, int nth_ext, char *name, ... )
{
  va_list ap ;
  adb_meta_file_t *si ;
  adb_field_t *fld ;
  char *fld_name ;
  adb_field_t **siidx ;
  int idx ;
  int n ;
  void *r ;
  adb_bh_t fbh = { .magic = ADB_MAGIC_BH, 
		   .cmp = Adb_field_cmp, 
		   .addr = db->addr, 
		   .tree = &mfile->fields } ;

  adb_bh_t ibh = { .magic = ADB_MAGIC_BH, 
		   .cmp = Adb_file_cmp, 
		   .addr = db->addr, 
		   .tree = &mfile->indexes } ;

  ADB_ERROK();

  va_start(ap, name );

  /* test loop */
  for ( n = 0 ; ( fld_name = va_arg(ap, char *)) ; ++n )
    {
      adb_field_t f ;
      strncpy(f.name, fld_name, ADB_NAME_MAX - 1 );
      if ( ! ( fld = adb_tavl_find( &fbh, &f )))
	{
	  ADB_WARNING("Can't find field `%s' in file `%s' database `%s'",
		      fld_name, mfile->name, db->name );
	  return NULL ;
	}
    }
  
  if ( !(si = Adb_malloc( NULL, sizeof(*si) , db->mm , __FUNCTION__ )))
    ADB_ERROR("No free space in meta-scheme `%s'",db->name);
  memcpy( si , mfile, sizeof(*si));

  si->idx_size = si->idx_alloc = n ;

  si->all_data_size = mfile->all_data_size ;

  siidx = Adb_malloc( NULL, sizeof(adb_field_t *) * n  , db->mm , __FUNCTION__ );
  si->__idx = (void *)siidx - db->addr ;
  strncpy(si->name, name, ADB_NAME_MAX - 1 );
  si->__parent = (void *) mfile - db->addr ;
  si->block_size = ADB_SEG_SIZE ;
/*   si->secondary = TRUE ; */

  if ( nth_ext && db->nfd > 1 )
    si->nfd = nth_ext ;

  /* real work follows */
  va_start(ap, name );
  for ( idx = 0 ; ( fld_name = va_arg(ap, char *)) ; ++idx )
    {
      adb_field_t f ;
      strncpy(f.name, fld_name, ADB_NAME_MAX - 1 );

      fld = adb_tavl_find(&fbh, &f );
      if ( ! fld )
	{
	  ADB_WARNING("Can't find field `%s' in file `%s' database `%s'",
		      f.name, mfile->name, db->name );
	  return NULL ;
	}

      siidx[idx] = (void *)fld - db->addr ;
      fprintf(stderr,"%s [%d] : %s\n",__FUNCTION__, idx, fld->name);
    }

  for ( ; idx < n ; ++idx )
    siidx[idx] = NULL ; /* ZERO remaining */
  
  r = adb_tavl_insert( &ibh, &si->node, si );
  assert(!r);

  return si ;
}

int
Adb_field_cmp(const void *a, const void *b, const void *param UNUSED , int uuu UNUSED )
{
  adb_field_t *field_a = ( adb_field_t *) a ;
  adb_field_t *field_b = ( adb_field_t *) b ;
#if 0
  assert( field_a->magic == ADB_MAGIC_FIELD );
  assert( field_b->magic == ADB_MAGIC_FIELD );
  fprintf(stderr,"field_a->name %s field_b->name %s\n",field_a->name, field_b->name );
  fprintf(stderr,"field_a->magic %x field_b->magic %x\n",field_a->magic, field_b->magic );  fprintf(stderr,"field_a %p field_b %p\n",field_a, field_b );
#endif /* 0 */
  return strcmp (field_a->name, field_b->name ) ;
}


static adb_field_t **
Adb_field_malloc(adb_db_t *db, int len, int *newsize)
{
  int size ;
  adb_field_t **new ;
  if ( len < ADB_MM_MIN )
    size = ADB_MM_MIN * sizeof( adb_field_t *);
  else
    size = ADB_MM_MIN * sizeof( adb_field_t *) * ( len / ADB_MM_MIN + 1 ) ;
  if ( ! ( new = Adb_malloc( NULL, size , db->mm , __func__ )))
    {
      g_warning( "No space left (request size %d bytes)" , len );
      return NULL ;
    }
#if 0
  fprintf(stderr, "Request: %d => size: %d\n",len, size);
#endif
  *newsize = size / sizeof( adb_field_t *) ;
  return new ;
}

inline void
Adb_fld_copy(adb_field_t **dst, adb_field_t **src)
{
  assert((*src)->magic == ADB_MAGIC_FIELD );
  *dst = *src ;
}

static adb_field_t **
Adb_adjust_fld_array(adb_db_t *db, adb_field_t **idx, int idx_pos , int *idx_size )
{
  if ( ! idx )
    { /* alloc idx */
      idx = Adb_field_malloc( db, idx_pos , idx_size ) ; assert(idx);
    }
  if ( idx_pos >= *idx_size  )
    {
      int i ;
      int size = *idx_size ;
      adb_field_t *tmp[size] ;
      for ( i = 0 ; i < size ; ++i )
	Adb_fld_copy( &tmp[i], &idx[i] );
/*       adb_field_t *tmp = &stuff[0] ; */
#if 0
      fprintf(stderr, "\aResize !!!!\n");
#endif
/*       memcpy( stuff , idx , *idx_size ); */
      Adb_free(NULL, (unsigned char *)idx , *idx_size * sizeof(char *), db->mm, __func__);
      idx = Adb_field_malloc( db, idx_pos , idx_size ); assert(idx);
/*       memcpy( idx , stuff , *idx_size ); */
      for ( i = 0 ; i < size ; ++i )
	Adb_fld_copy( &idx[i], &tmp[i] );
    }
  return (void *)idx - db->addr ;
}

adb_field_t *
adb_create_field( adb_db_t *db, adb_meta_file_t *mfile, char *name, 
		  adb_field_type_t type, unsigned int len, int idx_pos )
{
  adb_field_t *field ;
  int data_size ;
  unsigned int sb = db->addr ;
  adb_bh_t fbh = { .magic = ADB_MAGIC_BH, 
		   .cmp = Adb_field_cmp, 
		   .addr = db->addr, 
		   .tree = &mfile->fields } ;

  adb_field_t **idx = (void *)mfile->__idx + db->addr ;
  adb_field_t **order = (void *)mfile->__order + db->addr ;

  assert( mfile->magic == ADB_MAGIC_META_FILE );

  field = Adb_malloc0(NULL, sizeof(*field), db->mm , __FUNCTION__ );

  field->magic = ADB_MAGIC_FIELD ;
  strncpy(field->name, name, ADB_NAME_MAX - 1 ) ;
  field->len = len ;
  field->type = type ;

  field->id = db->sb->last_id++ ;

  if ( field->type == ADB_FIELD_LINK && ! mfile->has_link )
    mfile->has_link = TRUE ;

  if ( adb_tavl_insert( &fbh , &field->node, field ) )
    {
      g_warning("*** field exists : `%s' ***", name);
      Adb_free(NULL, (unsigned char *) field , sizeof(*field), db->mm, __func__);
      return NULL ;
    }
  
  if ( idx_pos != EOF )
    {
      if ( idx_pos >= mfile->idx_alloc )
	{
	  if ( ! ( mfile->__idx = Adb_adjust_fld_array(db, idx, idx_pos, 
						 &mfile->idx_alloc )))
	    {
	      g_warning( "No space for `%s'", field->name );
	      return NULL ;
	    }
	  idx = (void *) mfile->__idx + db->addr ;
	  mfile->idx_size = idx_pos + 1 ;
	}
      
      idx[ idx_pos ] = (void *)field - sb ;
    }
  if ( ! ( mfile->__order = Adb_adjust_fld_array(db, order, 
					 mfile->last_order , 
					 &mfile->order_size )))
    {
      g_warning( "No space for `%s'",field->name );
      return NULL ;
    }
  order = (void *)mfile->__order + db->addr ;
  
  order[ mfile->last_order++ ] = (void *)field - sb ;

  switch ( type )
    {
    case ADB_FIELD_UNSIGNED_INT:
    case ADB_FIELD_INT:
      data_size = sizeof( int ) ;
      break ;
    case ADB_FIELD_STRING:
      data_size = len ;
      break ;
    case ADB_FIELD_VSTRING:
      data_size = sizeof (adb_vstring_t) ;
      break ;
    case ADB_FIELD_WCHAR:
      data_size = sizeof (adb_wchar_t) ;
      break ;
    case ADB_FIELD_LONG_LONG:
      data_size = sizeof (long long) ;
      break ;
    case ADB_FIELD_UNSIGNED_LONG_LONG:
      data_size = sizeof ( unsigned long long );
      break ;
    case ADB_FIELD_DOUBLE:
      data_size = sizeof ( long double );
      break ;
    case ADB_FIELD_BLOB:
    case ADB_FIELD_BLOB_STRING:
      data_size = sizeof( adb_blob_t ); /* store only ptrs to such ugliness */
      break ;
    case ADB_FIELD_LINK:
    case ADB_FIELD_CONTAINER:
      data_size = sizeof (adb_link_t) ;
      break ;
    case ADB_FIELD_PTR_DIFF:
      data_size = sizeof (__PTRDIFF_TYPE__) ;
      break ;
    default:
      assert(0);
      break ;
    }
  
  field->offset = mfile->all_data_size ;
  mfile->all_data_size += data_size ;
/*   adb_add_seg( adb_fd, ADB_SEG_SIZE  * 4); */
#if 0
  adb_show_files(file->db->sb);
#endif /* 0 */

  return field ;
}

void
adb_mode_show(adb_mode_t mode)
{
 int i ;
 fputs("=== Datadase status ===\n",stderr);
 for ( i = 0 ; i < ADB_MODE_BIT_MAX ; ++i )
   fprintf(stderr,"%s : %s\n", _adb_mode_list[i], mode & 1 << i ? "On" : "Off" );
}

void
adb_mode_set(adb_db_t *db, adb_mode_t new_mode)
{
  new_mode &= ~0x1F ; /* keep only lower 5 bits */
  adb_mode_show(new_mode);
  db->sb->mode = new_mode ;
}



void
adb_server(char *dbname)
{
  key_t shm_key , sem_key , shm_pid_to_sem_key ;
  void *shm ;
  int mode = 0666 | IPC_CREAT | IPC_EXCL ;
  char *fn =  dbname;
  
  int size = ADB_SEG_SIZE + ADB_PTS_SIZE + ADB_ZBUF_SIZE * ADB_ZBUF_COUNT ;
  adb_db_t *db = alloca(sizeof(*db));

  sem_key = ftok(fn, 'S');
  shm_key = ftok(fn, 'M');
  shm_pid_to_sem_key = ftok(fn, 'P');

  if ( shm_key == EOF ) abort();
  if ( sem_key == EOF ) abort();

  db->shmid = shmget( shm_key, size, mode );
  if ( db->shmid == EOF )
    {
      ADB_ERROR("Can't open lock shm, shmget (creat) failed");
      abort();
    }

  db->semid = semget( sem_key, ADB_USERS_MAX + ADB_ZBUF_COUNT , mode);
  if ( db->semid == EOF ) 
    {
      ADB_WARNING("cat /proc/sys/kernel/sem should be at least `%d * * *'\n",
		ADB_USERS_MAX + ADB_ZBUF_COUNT);
      ADB_ERROR("Can't open semaphore array, semget (creat) failed\n");
      abort();
    }

  shm = shmat ( db->shmid, 0, 0);
  if ( shm == (void *)EOF )
    ADB_ERROR("Can't shmat @ %p (ADB_LOCK_SHM)",(void *)ADB_LOCK_SHM);

  printf("Lock shared memory @ %p\n",shm);

  db->pts_shm = (void *) shm + ADB_SEG_SIZE; /*   shmat( db->ptsid, 0, 0); */
  db->zbuf = (void *) shm + ADB_SEG_SIZE + ADB_PTS_SIZE ;

/*   if ( db->pts_shm == (void *)EOF ) */
/*     ADB_ERROR("Can't shmat @ %p (ADB_PTS_SHM)",(void *)ADB_PTS_SHM); */
  
  printf("PTS (login) shared memory @ %p\n",db->pts_shm);

  if ( ( L = fopen("/tmp/l.txt", "w" )) == NULL )
    abort();

  if ( 1 )
    {
      ushort *array ;
      ushort values[ADB_USERS_MAX + ADB_ZBUF_COUNT] ;

      adb_mm_t *m ;
      adb_lock_sb_t *sb ;

      memset( values, 0 , sizeof(values));
      array = values ;
      if ( semctl( db->semid, 0, SETALL, array ) == EOF )
	abort();

      fprintf(L,"%ld [%d] Semaphore set initialized\n",time(NULL), getpid()); 
      fflush(L);

      m = (adb_mm_t *)shm ;
      sb = (adb_lock_sb_t *)shm ;
      db->lock_sb = sb ;
#if 0
      db->lock_sb->mode = db->sb->mode ;
#endif

      m->magic = ADB_MAGIC_MM ;
      m->size = ADB_SEG_SIZE ;
      m->map_size = ADB_SEG_MAP_SIZE ;

      /* BUGS!!! */

      sb->file_tree = tavl_create(&sb->__file_tree);
      sb->seg_tree = tavl_create(&sb->__seg_tree);
      sb->rec_tree = tavl_create(&sb->__rec_tree);

/*       sb->mode = main_mode ; */ /* main_mode usually comes from persistent superblock */
      
    }
}

static void
adb_death_wait(adb_db_t *db)
{
  sh_db = db ;

  setsid();

  signal( SIGTERM, Adb_bye );

  pause();
}

static void
adb_log_open(char *dbname)
{
  char *log = Adb_name(dbname, "log") ;

  freopen( log, "w", stdout);
  freopen( log, "w" ,stderr);

  fclose(stdin);
}



static int
Adb_create_lock_file(char *dbname, int multi_user, int *stale)
{
  char *lock = Adb_name(dbname, "lock");
  int lfd ;
  pid_t pid ;
  struct stat sb ;
  adb_mode_t mode ;
  int open_mode = O_EXCL | O_CREAT | O_TRUNC | O_RDWR ;
  if ( ! stat(lock, &sb ))
    { /* .lock exist */
      char *dir = NULL ;
      int pid ;
      if ( (lfd = open(lock, O_RDONLY)) == EOF )
	ADB_ERROR("Can't open lock file : `%s'\n", lock );

      read(lfd, &mode, sizeof(mode));
      read(lfd, &pid, sizeof(pid));

      ADB_WARNING("Lock file exists, created by pid %d\n",pid);
      asprintf(&dir,"/proc/%d",pid);
      
      if ( ! stat(dir, &sb) && S_ISDIR(sb.st_mode))
	{
	  ADB_ERROR("PID's %d Lock file is NOT stale, resign....\n",pid);
	}
      else
	{
	  ADB_WARNING("Stale lock-file `%s' found. I'll try undo all transactions\n",
		      lock );
	  *stale = TRUE ;
	  unlink(lock);
	}
    }
  
  if ( ( lfd = open(lock, open_mode, 0666 )) == EOF )
    ADB_ERROR("Can't open lock file : `%s'\n", lock );

  if ( multi_user )
    mode = ADB_MODE_MULTI ;
  else
    mode = ADB_MODE_SINGLE ;

  write(lfd, &mode, sizeof(mode));

  pid = getpid();
  write( lfd, &pid, sizeof(pid));

  sh_lock = lock ;

  return lfd ;
}


adb_db_t *
adb_start_db_single(char *dbname, char **ext)
{
  adb_db_t *db ;
  struct stat sb ;
  int lfd ;
  int stale = FALSE ;

  lfd = Adb_create_lock_file(dbname, TRUE, &stale ); 
  
  if ( stat(dbname , &sb))
    db = adb_creat( dbname, ext, FALSE );
  else
    {
      db = adb_open( dbname , FALSE );
  
      if ( ADB_MODE(db) & ADB_MODE_BI && stale )
	Adb_tnx_mass_undo(db);
    }
  
  return db ;
}

adb_db_t *
adb_start_db_multi(char *dbname, char **ext)
{
  adb_db_t *db ;
  struct stat sb ;
  int lfd ;
  int stale = FALSE ;
  int pid  ;


  if ( stat(dbname , &sb))
    {
      db = adb_creat( dbname, NULL, FALSE );
      lfd = Adb_create_lock_file(dbname, TRUE, &stale); /* bit racy */
    }
  else
    {
      db = adb_open( dbname , FALSE );
      lfd = Adb_create_lock_file(dbname, TRUE, &stale); /* bit racy */
      
      if ( ADB_MODE(db) & ADB_MODE_BI/*  && stale */ )
	Adb_tnx_mass_undo(db);
    }

  adb_close(db); /* single user end */

  adb_server(dbname); /* create IPC */
  db = adb_open( dbname , TRUE );

  switch ( pid = fork() )
    {
    case 0: /* child */
      {
	int mode ; /* rewrite lock file */
	lseek(lfd, 0LL, SEEK_SET );
	mode = ADB_MODE_MULTI ;
	write(lfd, &mode, sizeof(mode));
	
	pid = getpid();
	write( lfd, &pid, sizeof(pid));
      }
      adb_log_open(dbname);
      adb_death_wait(db);
      break ;
    case EOF:
      ADB_ERROR("Fork failed");
      break ;
    default:
      break ;
    }
  return db ; 
}

