/* AVLDB - AVL DataBase library, locking

   Copyright (C) 2002,2003 Petr Silhavy <silhavy@mef.cz>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.

*/

#define  __ADB 1
#include "adb.h"


/* !!!!!!!!!!! shmat AT FIXED address */

/*
 *                           *** Locking ***
 *
 * "levels" :                                  tree of:
 * R/W whole file                              idx: file_id misc: lock,queue, seg_tree
 * R/W segment => A in file level              idx: offset misc: lock,queue, rec_tree
 * R/W record => A in file & segment level     idx: offset misc: lock,queue
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */




static int
Adb_file_id_cmp(const void *a, const void *b, void *param UNUSED )
{
  adb_lock_file_t *fa = (adb_lock_file_t *)a , *fb = (adb_lock_file_t *)b ;
  fprintf(stderr,"%s %p %p\n", __FUNCTION__, fa->file, fb->file );
  return ADB_LCMP(fa->file, fb->file) ; /* ( a > b ) - ( a < b) */
}

static int
Adb_offset_cmp(const void *a, const void *b, void *param UNUSED )
{
  adb_lock_seg_t *sa = (adb_lock_seg_t *)a , *sb = (adb_lock_seg_t *)b ;
  return ADB_LCMP(sa->offset, sb->offset) ; /* ( a > b ) - ( a < b) */
}

static int
Adb_sem_get(adb_db_t *db)
{
  int i , j , max = sizeof(int) * 8 ;
  if ( db->lock_sb )
    for ( i = 0 ; i < ADB_NSEMS ; ++i )
      for ( j = 0 ; j < max ; ++j )
	{
	  int /* bit = 1 << j , */ sem ;
	  
	  if ( ! test_and_set_bit ( j, &db->lock_sb->sems[i] ) )
	    { /* we locked it ! */
	      fflush(L);
	      sem = i * max + j ;
	      fprintf(L,"%ld [%d] %s Locked sem# %d (bank %d bit %d) \n",time(NULL), 
		      getpid(), __FUNCTION__, sem, i, j);
	      return sem ;
	    }
	}
  return EOF ;
}

static int
Adb_sem_free(adb_db_t *db, int sem )
{
  int max = sizeof(int) * 8 ;
  int i = sem / max , j = sem % max ;
  if ( db->lock_sb )
    if ( ! test_and_clear_bit( j, &db->lock_sb->sems[i] ) )
      abort();
  return 0 ;
}

/* interface */
int 
adb_login(adb_db_t *db)
{
  int i ;
  adb_pts_t *pts = ( adb_pts_t *) db->pts_shm ;
  for ( i = 0 ; i < ADB_USERS_MAX ; ++i , ++pts )
    if ( ! test_and_set_bit( 1, &pts->lock ))
      return i ;
  abort();
}

/* interface */
void * /* return SHM addr e.g. superblock */
adb_lock_init(adb_db_t *db, char *dbname)
{
  key_t shm_key , sem_key , shm_pid_to_sem_key ;
  void *shm ;
  int mode = 0666 ;
  char *fn =  dbname;
  int error = 0 ;

  int size = ADB_SEG_SIZE + ADB_PTS_SIZE + ADB_ZBUF_SIZE * ADB_ZBUF_COUNT ;
  int i ;

  sem_key = ftok(fn, 'S');
  shm_key = ftok(fn, 'M');
  shm_pid_to_sem_key = ftok(fn, 'P');

  if ( shm_key == EOF ) abort();
  if ( sem_key == EOF ) abort();

  for ( i = 1 , db->shmid = EOF; i < 16 && db->shmid == EOF ; ++i )
    {
      db->shmid = shmget( shm_key, size, mode );
      if ( db->shmid == EOF )
	{
	  ADB_WARNING("Can't open lock shm, retrying...");
	  sleep(i);
	}
    }

  if ( db->shmid == EOF )
    {
      ADB_ERROR("Can't open lock shm, shmget (open) failed ");
      abort();
    }

  db->semid = semget( sem_key, ADB_USERS_MAX + ADB_ZBUF_COUNT , mode);
  if ( db->semid == EOF ) 
    {
      ADB_WARNING("cat /proc/sys/kernel/sem should be at least `%d * * *'\n",
		ADB_USERS_MAX);
      ADB_ERROR("Can't open semaphore array, semget (open) failed\n");
      abort();
    }

  shm = shmat ( db->shmid, 0, 0);
  if ( shm == (void *)EOF )
    ADB_ERROR("Can't shmat @ %p (ADB_LOCK_SHM)",(void *)ADB_LOCK_SHM);

  printf("Lock shared memory @ %p\n",shm);

  db->pts_shm = (void *) shm + ADB_SEG_SIZE; /*   shmat( db->ptsid, 0, 0); */
  db->zbuf = (void *) shm + ADB_SEG_SIZE + ADB_PTS_SIZE ;

/*   if ( db->pts_shm == (void *)EOF ) */
/*     ADB_ERROR("Can't shmat @ %p (ADB_PTS_SHM)",(void *)ADB_PTS_SHM); */
  
  printf("PTS (login) shared memory @ %p\n",db->pts_shm);

  if ( ( L = fopen("/tmp/l.txt", "w" )) == NULL )
    abort();

  db->shm = shm ;
  db->lock_sb = (adb_lock_sb_t *)shm ;

  if ( error )
    return NULL ; /* for administration request line AI begin to verify NO clients */

  return shm ;
}

static /* inline */ void
Adb_lock_claim_validate(adb_lock_t scope, adb_lock_t type)
{
  switch ( scope )
    {
    case ADB_LOCK_FILE:
    case ADB_LOCK_SEG:
    case ADB_LOCK_REC:
      break ;
    default:
      assert("bug" == 0);
      break ;
    }

  switch ( type )
    {
    case ADB_LOCK_READ:
    case ADB_LOCK_WRITE:
      break ;
    default:
      assert("bug" == 0);
      break ;
    }
}

inline int
Adb_lock_test_sb_with( adb_db_t *db, _adb_lock_t type)
{
  return  test_bit( type, &db->lock_sb->la.lock );
}

inline void
Adb_lock_claim_sb_with(adb_db_t *db, _adb_lock_t type)
{
  while (db->lock_sb)
    {
      fprintf(L,"%ld [%d] %s Trying ... <0x%x> \n",time(NULL), getpid(), __FUNCTION__, 
	      db->lock_sb->la.lock);
      fflush(L);
      if ( ! test_and_set_bit ( type, &db->lock_sb->la.lock ) )
	{ /* we locked it ! */
	  fprintf(L,"%ld [%d] %s Locked <0x%x> \n",time(NULL), 
		  getpid(), __FUNCTION__, db->lock_sb->la.lock);
	  fflush(L);
	  return ;
	}
      /* damn!, locked */
      sleep(1);
      /* usleep(1000); */ /* 1ms */
    }
}


inline void
Adb_lock_free_sb_with(adb_db_t *db, _adb_lock_t type)
{
  if ( db->lock_sb )
    if ( ! test_and_clear_bit( type, &db->lock_sb->la.lock ) )
      abort();
}

static int
Adb_lock_safe( adb_db_t *db, adb_lock_atom_t *la, 
		adb_lock_cb_func_t func, adb_lock_t param)
{
  int r ;
  _adb_lock_t bit = ffs(param) - 1 ;
  assert( bit != ADB__LOCK_UPDATE );
  while (1)
    {
      if ( ! test_and_set_bit ( ADB__LOCK_UPDATE, &la->lock ) )
	{
	  r = func(db, la, bit);
	  if ( ! test_and_clear_bit( ADB__LOCK_UPDATE, &la->lock ) )
	    abort();
	  if ( test_bit( ADB__LOCK_UPDATE, &la->lock ) )
	    {
	      fprintf(L,"%ld [%d] Strange bit is set <0x%x>\n",time(NULL),  getpid(), la->lock);
	      fflush(L);
	    }
	  return r;
	}
      sleep(1);
      /* usleep(1000); */
    }
}

static int
Adb_lock_inc_usage(adb_db_t *db UNUSED, 
		     adb_lock_atom_t *la, _adb_lock_t type)
{
  switch ( type )
    {
    case ADB__LOCK_READ:
      ++la->usage ;
      break ;
    case ADB__LOCK_AREAD:
      ++la->ausage ;
      break ;
    case ADB__LOCK_AWRITE:
      ++la->awusage ;
      break ;
    default:
      abort();
      break ;
    }
  return 1 ;
}

static int
Adb_lock_inc_waiting(adb_db_t *db UNUSED , adb_lock_atom_t *la, _adb_lock_t type UNUSED )
{
  ++la->waiting ;
  return 1 ;
}

static int
Adb_lock_dec_waiting(adb_db_t *db UNUSED , 
		       adb_lock_atom_t *la, _adb_lock_t type UNUSED )
{
  la->waiting-- ;
  return 1 ;
}

static int
Adb_get_semval(adb_db_t *db, int sem)
{
  int val ;
  return semctl( db->semid, sem, GETVAL, &val );
}


/* Bloatware ? */
#if 0
int
Adb_lock_upgrade_read(adb_lock_atom_t *la, void *p )
{
  adb_lock_t type = ( adb_lock_t ) p ;
  ++la->usage ;

  if ( ! test_and_set_bit ( ffs(type), &la->lock ))
    return 1 ;
  return 0 ;
}
#endif /* 0 */

static int
Adb_lock_set( adb_db_t *db, adb_lock_atom_t *la, _adb_lock_t type )
{
  int do_sem = 0 ;
  switch ( type )
    {
    case ADB__LOCK_READ:
      if ( ! la->usage++ )
	++do_sem ;
      break ;
    case ADB__LOCK_AREAD:
      if ( ! la->ausage++ )
	++do_sem ;
      break ;
    case ADB__LOCK_AWRITE:
      if ( ! la->awusage++ )
	++do_sem ;
      break ;
    default:
      break ;
    }

  if ( do_sem )
    assert(la->sem = EOF) ;
#if 0
    {
      struct sembuf sb ;
      sb.sem_num = adb_login_id ;        /* semaphore index in array */
      sb.sem_op = 1 ;         /* semaphore operation */
      sb.sem_flg = SEM_UNDO;        /* operation flags */
      
      if ( semop( db->semid, &sb, 1 ) == EOF )
	abort();
      la->sem = sb.sem_num ;
      la->pid = getpid();
      fprintf(L,"%ld [%d] Semaphore %d up, set by %d value [%d]\n",time(NULL), getpid(),la->sem, la->pid,
	     Adb_get_semval(db, la->sem));
      fflush(L);
      fflush(L);
    }
#endif /* 0 */
  if ( ! test_and_set_bit ( type, &la->lock ) )
    return 1 ;
  return 0 ;

}

/* ??? interface ??? */
void *
Adb_zbuf_claim(adb_db_t *db, int *sem_num)
{
  int i ;
  struct sembuf sb = { .sem_op = 1 , .sem_flg = SEM_UNDO } ;
  for ( i = 0 ; i < ADB_ZBUF_COUNT ; ++i )
    {
      sb.sem_num = ADB_USERS_MAX + i ;
      if ( ! semop( db->semid, &sb, 1 ) )
	{
	  *sem_num = sb.sem_num ;
	  return (void *)db->zbuf + ADB_ZBUF_SIZE * i ;
	}
    }
  return NULL ;
}

static int
Adb_lock_clear(adb_db_t *db, adb_lock_atom_t *la, _adb_lock_t type )
{
  int do_sem = 0 ;
  switch ( type )
    {
    case ADB__LOCK_READ:
      if ( ! --la->usage )
	++do_sem ;
      break ;
    case ADB__LOCK_AREAD:
      if ( ! --la->ausage )
	++do_sem ;
      break ;
    case ADB__LOCK_AWRITE:
      if ( ! --la->awusage )
	++do_sem ;
      break ;
    default:
      break ;
    }

  if ( do_sem )
    {
      struct sembuf sb ;

      sb.sem_num = la->sem ;        /* semaphore index in array */
      sb.sem_op = -1 ;         /* semaphore operation */
      sb.sem_flg = 0 /* IPC_NOWAIT */;        /* operation flags */

      fprintf(L,"%ld [%d] Semaphore %d down, set by %d value [%d]\n",time(NULL), 
	      getpid(),la->sem, la->pid,
	     Adb_get_semval(db, la->sem));
      fflush(L);

      if ( ! test_and_clear_bit ( type, &la->lock ) )
	abort();
#if 0
      if ( semop( adb_semid, &sb, 1 ) == EOF )
	abort(); /* IPC_NOWAIT */
#else
      if ( la->waiting )
	{
	  int val ;
	  val = 0 ;
	  if ( semctl(db->semid, la->sem, SETVAL, val ) == EOF )
	    abort();
	}
#endif
      fprintf(L,"%ld [%d] Semaphore %d zero value [%d] OK\n",time(NULL), getpid(), 
	      la->sem , Adb_get_semval(db, la->sem) );
      fflush(L);

      la->sem = EOF ;
      la->pid = 0 ;

    }

  return 1 ;
}

/* !!! this MUST NOT run protected by LOCK_UPDATE !!! */

static void
Adb_lock_wait(adb_db_t *db, adb_lock_atom_t *la, adb_lock_t type)
{
  struct sembuf sb ;

  assert(! (type & ADB_LOCK_UPDATE ));

  while ( la->sem == EOF )
    {
      if ( ! test_and_set_bit ( ADB__LOCK_UPDATE, &la->lock ) )
	{
	  if ( la->sem == EOF )
	    {
	      assert( la->waiting == 0 );

	      la->sem = Adb_sem_get(db);
	      la->pid = getpid();

	      /* set semaphore */
	      sb.sem_num = la->sem ;        /* semaphore index in array */
	      sb.sem_op = 1 ;         /* semaphore operation */
	      sb.sem_flg = SEM_UNDO;        /* operation flags */
	      
	      if ( semop( db->semid, &sb, 1 ) == EOF )
		abort();
      
	      fprintf(L,"%ld [%d] Semaphore %d up, set by %d value [%d]\n",
		      time(NULL), getpid(),la->sem, la->pid,
		      Adb_get_semval(db, la->sem));
	    }
	  /* direct call is ok, UPDATE lock is held */
	  Adb_lock_inc_waiting(NULL, la, 0);

	  if ( ! test_and_clear_bit( ADB__LOCK_UPDATE, &la->lock ) )
	    abort();
	  break ;
	}
    }

  sb.sem_num = la->sem ;        /* semaphore index in array */
  sb.sem_op = 0 ; /* WAIT */    /* semaphore operation */
  sb.sem_flg = 0;

/*   Adb_lock_safe(db, la, Adb_lock_inc_waiting, -1); */

  fprintf(L,"%ld [%d] <0x%x> Waiting for lock <0x%x> by pid %d value [%d] sem# %d\n",
	  time(NULL), getpid(), type, 
	  la->lock, la->pid , Adb_get_semval(db, la->sem), la->sem);

  if ( semop( db->semid, &sb, 1 ) == EOF )
    abort();
   
  fprintf(L,"%ld [%d] <0x%x> Wake UP value [%d] # %d\n",
	  time(NULL), getpid(), type,
	  Adb_get_semval(db, la->sem), la->sem);
  fflush(L);

  Adb_lock_safe(db, la, Adb_lock_dec_waiting, -1);

  
  while (1)
    {
      /* free semaphore if no other process is waiting */
      if ( ! test_and_set_bit ( ADB__LOCK_UPDATE, &la->lock ) )
	{
	  if ( la->waiting == 0 )
	    Adb_sem_free(db, la->sem);

	  if ( ! test_and_clear_bit( ADB__LOCK_UPDATE, &la->lock ) )
	    abort();
	  break ;
	}
    }
}

static void 
Adb_lock_claim_atom( adb_db_t *db, adb_lock_atom_t *la, adb_lock_t type)
{
  while (1)
    {
      fprintf(L,"%ld [%d] %s current <0x%x> need <0x%x>\n",time(NULL), getpid(), __FUNCTION__, la->lock, type);
      fflush(L);
      if ( la->lock & type ) /* great ! , only inc usage - if possible ... */
	{
	  switch ( type )
	    {
	    case ADB_LOCK_READ: /* needed */
	    case ADB_LOCK_AREAD:
	    case ADB_LOCK_AWRITE:
	      if ( Adb_lock_safe(db, la, Adb_lock_inc_usage, type))
		return ;
	      break ;
	    default: 
	      assert( type == ADB_LOCK_WRITE );
	      Adb_lock_wait(db, la, type);
	      break ;
	    }
	}
      else if ( la->lock == ADB_LOCK_NULL ) /* empty lock */
	{
	  if ( Adb_lock_safe(db, la, Adb_lock_set, type) )
	    return ;
	}
      else  /* Damn, I'm running into troubles */
	{
	  switch ( type )
	    {
	    case ADB_LOCK_READ:
	      if ( la->lock & ADB_LOCK_WRITE ||
		   la->lock & ADB_LOCK_AWRITE )
		{
		  Adb_lock_wait(db, la, type);/* WAIT */
		  break ;
		}
	      if ( la->lock & ADB_LOCK_AREAD ) /* AWRITE is handled above */
		{
		  if ( Adb_lock_safe(db, la, Adb_lock_set, type) )
		    return ;
		  break ;
		}
	      abort();
	      break ;
	    case ADB_LOCK_AREAD:
	      if ( la->lock & ADB_LOCK_WRITE )
		{
		  Adb_lock_wait(db, la, type);/* WAIT */
		  break ;
		}
	      if ( la->lock & ADB_LOCK_READ ||
		   la->lock & ADB_LOCK_AWRITE )
		{
		  if ( Adb_lock_safe(db, la, Adb_lock_set, type ))
		    return ;
		  break ;
		}
	      abort();
	      break ;
	    case ADB_LOCK_WRITE:
	      Adb_lock_wait(db, la, type);/* WAIT */
	      break ;
	    case ADB_LOCK_AWRITE:
	      if ( la->lock & ADB_LOCK_AREAD )
		{
		  if ( Adb_lock_safe(db, la, Adb_lock_set, type))
		    return ;
		  break ;
		}
	      Adb_lock_wait(db, la, type);/* WAIT */
	      break ;
	    default:
	      abort();
	      break ;
	    }
	}
    }

}

/* TRUE if delete ok , otherwise ZERO */

static int
Adb_lock_free_atom(adb_db_t *db, adb_lock_atom_t *la, adb_lock_t type)
{
  while (1)
    {
      if ( la->lock == type )
	{
	  if ( Adb_lock_safe(db, la, Adb_lock_clear, type))
	    {
	      fprintf(L,"%ld [%d] %s  usage: %d ausage: %d awusage: %d\n",time(NULL), getpid(), 
		     __FUNCTION__, la->usage, la->ausage, la->awusage );
	      fflush(L);
	      if ( la->usage == 0 &&
		   la->ausage == 0 &&
		   la->awusage == 0 &&
		   la->waiting == 0 ) 
		return 1 ; /* ok to free */
	      return 0 ; /* Don't free */
	    }
	  abort();
	}
      else
	{
	  switch ( type )
	    {
	    case ADB_LOCK_READ: /* only AREAD should coexists */
	      if ( la->lock & ADB_LOCK_WRITE ||
		   la->lock & ADB_LOCK_AWRITE )
		abort();
	      
	      if ( la->lock & ADB_LOCK_AREAD )
		if ( Adb_lock_safe(db, la, Adb_lock_clear, type))
		  return 0 ;
	      break ;

	    case ADB_LOCK_AREAD: /* Non-bug others READ/ AWRITE */
	      if ( la->lock & ADB_LOCK_WRITE )
		abort();

	      if (  Adb_lock_safe(db, la, Adb_lock_clear, type))
		return 0 ;
	      break ;
	    case ADB_LOCK_AWRITE: /* Non-bug others AREAD */
	      if ( la->lock & ADB_LOCK_WRITE ||
		   la->lock & ADB_LOCK_READ )
		abort();

	      if (  Adb_lock_safe(db, la, Adb_lock_clear, type))
		return 0 ;
	      break ;
	    case ADB_LOCK_WRITE:
	      if ( la->lock & ADB_LOCK_READ ||
		   la->lock & ADB_LOCK_AREAD ||
		   la->lock & ADB_LOCK_AWRITE )
		abort();
	      if (  Adb_lock_safe(db, la, Adb_lock_clear, type))
		return 0 ;
	      break ;
	    default:
	      abort();
	      break ;
	    }
	}
    }
  return 0 ;
}

/* segment / record */
static adb_lock_seg_t *
Adb_lock_find_or_insert_sr(adb_file_t *file, off_t offset, 
			     tavl_table_t *tree)
{
  adb_lock_seg_t _lock , *flock ;
  memset(&_lock, 0 , sizeof(_lock));
  _lock.offset = offset ;
  if ( ! ( flock = (adb_lock_seg_t *)tavl_find( tree, Adb_offset_cmp, &_lock ))) 
    { /* nonexistent */
      Adb_lock_claim_sb(file->db);
      flock = Adb_malloc0( NULL, sizeof(*flock), 
			     (adb_mm_t *)file->db->lock_sb ,
			     __FUNCTION__ );
      flock->offset = offset ;
      flock->file = file ;
      flock->la.sem = EOF ;

      tavl_insert( tree, Adb_offset_cmp, &flock->node, flock ) ; 
      assert(flock);
      Adb_lock_free_sb(file->db);
    }
  return flock ;
}

static adb_lock_file_t *
Adb_lock_find_or_insert_file(adb_file_t *file, tavl_table_t *tree)
{
  adb_lock_file_t _lock , *flock ;

  memset(&_lock, 0 , sizeof(_lock));
  _lock.file = file ;

  if ( ! ( flock = (adb_lock_file_t *)tavl_find( tree, Adb_file_id_cmp, &_lock ))) 
    { /* nonexistent */
      Adb_lock_claim_sb(file->db);
      flock = Adb_malloc0( NULL, sizeof(*flock), 
			     (adb_mm_t *)file->db->lock_sb,
			     __FUNCTION__ );
      flock->file = file ;
      flock->la.sem = EOF ;
      tavl_insert( tree, Adb_file_id_cmp, &flock->node, flock ) ; 
      assert(flock);
      Adb_lock_free_sb(file->db);
    }
  return flock ;
}



static void
Adb_lock_claim_file( adb_file_t *file, adb_lock_t type)
{
  adb_lock_file_t *flock = 
    Adb_lock_find_or_insert_file(file, file->db->lock_sb->file_tree );

  fprintf(L,"%ld [%d] %s current <0x%x> need <0x%x>\n",time(NULL), getpid(), __FUNCTION__, flock->la.lock, type);
  fflush(L);

  Adb_lock_claim_atom(file->db, &flock->la, type );
}


static void
Adb_lock_free_file( adb_file_t *file, adb_lock_t type)
{
  adb_lock_file_t _lock , *lock ;
  memset(&_lock, 0 , sizeof(_lock));
  _lock.file = file ;

  lock = tavl_find( file->db->lock_sb->file_tree, Adb_file_id_cmp, &_lock );

  if ( ! lock )
    {
      fprintf(stderr, "Count: %d\n",file->db->lock_sb->file_tree->tavl_count);
      assert ( tavl_count( file->db->lock_sb->file_tree ) == 0 );
    }
  else
    {
      if ( Adb_lock_free_atom(file->db, &lock->la, type) )
	{
	  fprintf(L,"%ld [%d] Deleting file from lock-tree\n",
		  time(NULL), getpid() ); 
	  fflush(L);
	  Adb_lock_claim_sb(file->db);
	  tavl_delete(  file->db->lock_sb->file_tree, Adb_file_id_cmp, lock );
	  Adb_free( NULL, (void *)lock , sizeof(*lock), 
		      (adb_mm_t *)file->db->lock_sb, __FUNCTION__);

	  Adb_lock_free_sb(file->db);
	}
    }
}

static adb_lock_t
Adb_lock_get_atype(adb_lock_t type)
{
  adb_lock_t atype ;
  switch ( type )
    {
    case ADB_LOCK_READ:
    case ADB_LOCK_AREAD:
      atype = ADB_LOCK_AREAD ;
      break ;
    case ADB_LOCK_WRITE:
    case ADB_LOCK_AWRITE:
      atype = ADB_LOCK_AWRITE ;
      break ;
    default:
      abort();
    }
  return atype ;
}

static void
Adb_lock_claim_seg( adb_file_t *file, off_t offset, adb_lock_t type)
{
  adb_lock_file_t *flock = Adb_lock_find_or_insert_file(file,  
							  file->db->lock_sb->file_tree );
  adb_lock_seg_t *lck = Adb_lock_find_or_insert_sr(file, offset, 
						     file->db->lock_sb->seg_tree);

  adb_lock_t atype = Adb_lock_get_atype(type) ;

  Adb_lock_claim_atom(file->db, &flock->la, atype );

  Adb_lock_claim_atom(file->db, &lck->la, type);
}

static void
Adb_lock_free_seg( adb_file_t *file, off_t offset, adb_lock_t type)
{
  adb_lock_seg_t _lock , *lock ;

/*   adb_lock_t atype = Adb_lock_get_atype(type); */

  memset(&_lock, 0 , sizeof(_lock));
  _lock.offset = offset ;

  lock = tavl_find( file->db->lock_sb->seg_tree, Adb_offset_cmp, &_lock );
  
  if ( ! lock )
    assert ( tavl_count( file->db->lock_sb->seg_tree ) == 0 );
  else
    {
      /* returns 1 if usage is zero */
      if ( Adb_lock_free_atom(file->db, &lock->la, type) ) 
	{
	  fprintf(L,"%ld [%d] Deleting segment from lock-tree\n",
		  time(NULL), getpid() ); 
	  fflush(L);
	  Adb_lock_claim_sb(file->db);
	  tavl_delete( file->db->lock_sb->seg_tree, Adb_offset_cmp, lock );
	  Adb_free( NULL, (void *)lock, sizeof(*lock), 
		      (adb_mm_t *)file->db->lock_sb, __FUNCTION__ );

	  Adb_lock_free_sb(file->db);
	}
    }

/*   _adb_lock_free_file(file, atype); */

}

static void
Adb_lock_claim_rec( adb_file_t *file, off_t seg_offset, 
		    off_t rec_offset, adb_lock_t type)
{
  adb_lock_file_t *flock = Adb_lock_find_or_insert_file(file, 
							  file->db->lock_sb->file_tree );

  adb_lock_seg_t *seg = 
    Adb_lock_find_or_insert_sr(file, seg_offset, file->db->lock_sb->seg_tree);
  adb_lock_seg_t *rec = 
    Adb_lock_find_or_insert_sr(file, rec_offset, file->db->lock_sb->rec_tree);

  adb_lock_t atype = Adb_lock_get_atype(type);

  Adb_lock_claim_atom(file->db, &flock->la, atype );

  Adb_lock_claim_atom(file->db, &seg->la, atype);

  Adb_lock_claim_atom(file->db, &rec->la, type);
}

static void
Adb_lock_free_rec( adb_file_t *file, off_t seg_offset UNUSED, off_t rec_offset, adb_lock_t type)
{
  adb_lock_seg_t _lock , *lock ;

/*   adb_lock_t atype = Adb_lock_get_atype(type); */

  memset(&_lock, 0 , sizeof(_lock));
  _lock.offset = rec_offset ;

  lock = tavl_find( file->db->lock_sb->rec_tree, Adb_offset_cmp, &_lock );

  if ( ! lock )
    assert ( tavl_count( file->db->lock_sb->rec_tree ) == 0 );
  else
    {
      /* returns 1 if usage is zero */
      if ( Adb_lock_free_atom(file->db, &lock->la, type) ) 
	{
	  fprintf(L,"%ld [%d] Deleting file from lock-tree\n",
		  time(NULL), getpid() ); 
	  fflush(L);
	  Adb_lock_claim_sb(file->db);
	  tavl_delete( file->db->lock_sb->rec_tree, Adb_offset_cmp, lock );
	  Adb_free( NULL, (void *)lock, sizeof(*lock), 
		      (adb_mm_t *)file->db->lock_sb, __FUNCTION__ );
	  Adb_lock_free_sb(file->db);
	}
    }
/*   _adb_lock_free_seg(file, seg_offset, atype); */
  
/*   _adb_lock_free_file(file, atype); */
}

/* Interface */
int
adb_lock_claim( adb_file_t *file, off_t seg_offset , 
	       off_t rec_offset, adb_lock_scope_t scope, adb_lock_t type )
{
  Adb_lock_claim_validate(scope, type );
  
   switch ( scope )
    {
    case ADB_LOCK_FILE:
      Adb_lock_claim_file(file, type);
      break ;
    case ADB_LOCK_SEG:
      Adb_lock_claim_seg(file, seg_offset, type);
      break ;
    case ADB_LOCK_REC:
      Adb_lock_claim_rec(file, seg_offset, rec_offset, type);
      break ;
    default:
      assert("bug" == 0);
      break ;
    }
   return 0 ;
}

int
adb_lock_free( adb_file_t *file, off_t seg_offset, off_t rec_offset, adb_lock_scope_t scope, adb_lock_t type)
{
  adb_lock_t atype = Adb_lock_get_atype(type);
  
  Adb_lock_claim_validate(scope, type );
  
   switch ( scope )
     {
     case ADB_LOCK_FILE:
       Adb_lock_free_file(file, type);
       break ;
     case ADB_LOCK_SEG:
       Adb_lock_free_seg(file, seg_offset, type);
       Adb_lock_free_file(file, atype);
       break ;
     case ADB_LOCK_REC:
       Adb_lock_free_rec(file, seg_offset, rec_offset, type);
       Adb_lock_free_seg(file, seg_offset, atype);
       Adb_lock_free_file(file, atype);
       break ;
     default:
       assert("bug" == 0);
       break ;
     }
   return 0 ;
}
