/* AVLDB - AVL Abuse DataBase library
   
   Copyright (C) 2002-2003 Petr Silhavy <silhavy@mef.cz>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA.

*/

#define  __ADB 1
#include "adb.h"

/*
 * Object (record) has a (target) :
 *
 *
 * 0. $Link file uses record's offset as primary parts of index, target's
 * offset as secondary.
 *
 * 1. $BackLink file uses target's offset as primary parts of index,  
 * link's offset as secondary.
 *
 * 2. When record moves, update is necessary in $Link file so $Link moves
 *
 * 3. When $Link moves, update is necessary in $BackLink file 
 *
 * 4. if file contains link(s) fields and link->count != 0 
 * => alloc new $Link and delete old later
 *
 * 5. When target moves, update is necessary in $Link file so $Link moves
 * 
 * 6. When $Link moves, update is necessary in $BackLink file so $BackLink 
 * moves 
 *
 * 7. if target can be fount by find_close and there are really
 * some nodes in $BackLink matching target 
 * => alloc new $BackLink and $Link , delete old later
 *
 * 8. Whan $BackLink moves, nobody cares 
 *
 * 9. To avoid endless madness updata stuff above only when 
 */

#ifdef MORE_BUGS
/* magically transforms the biggest link field in shared link
 * block to pseudo file 
 */

static void
adb_link_to_file(adb_tnx_t *tnx, adb_file_t *file, 
		 adb_bh_t *bh, adb_bh_t *lbh )
{
  adb_link_t *link = NULL , *l , new_link = { .magic = ADB_MAGIC_LINK,
					      .type = ADB_LINK_TYPE_PRIVATE ,
					      .u.ds.data_start_type = ADB_MAGIC_TREE } ;
  int i, max ;
  unsigned int sb = file->db->addr ;
  adb_field_t *fld ;

  adb_ref_t *ref ;
  adb_tavl_traverser_t tr ;
  adb_rec_t *rec ;

  /* find the largest link field in bh */
  for ( max = 0 , i = 0 ; i < file->meta->last_order ; ++i )
    {
      if ( ( fld = (void *)file->order[i] + sb ) && fld->magic == ADB_MAGIC_FIELD )
	{
	  if ( fld->type == ADB_FIELD_LINK )
	    {
	      for ( rec = adb_tavl_t_first(bh, &tr) ; 
		    rec ; 
		    rec = adb_tavl_t_next(bh, &tr) )
		{
		  l =  (void *)rec + fld->offset ;
		  if ( l->type == ADB_LINK_TYPE_SHARED &&  
		       l->u.tree.tavl_count > max )
		    {
		      max = l->u.tree.tavl_count ;
		      link = l ;
		    }
		}
	    }
	}
    }
  
  assert( link != NULL );

  fprintf(stderr, "%s : I'll move link %p to file\n", __func__, link);

  /* holy move operation follows */
  {
    adb_bh_t fbh = { .magic = ADB_MAGIC_BH, .tree = &link->u.tree, 
		     .m = lbh->m, .state = ADB_TNX_DIRTY,
		     .addr = lbh->addr, .size = lbh->size,
		     .db = lbh->db, .offset = lbh->offset,
		     .cmp = lbh->cmp, .file = lbh->file } ;
    adb_bh_t *new_lbh ;
    adb_rec_t *new ;

    new_lbh = _adb_seg_add( tnx, file, &new_link.u.ds.data_start ,
			    ADB_MAGIC_TREE, 0 , -1LL );

    for ( ref = adb_tavl_t_first(&fbh, &tr) ;
	  ref ;
	  ref = adb_tavl_t_first(&fbh, &tr) )
      {
	const void *item ;

	assert( ref->magic == ADB_MAGIC_RECORD );
	new = __adb_malloc( file, sizeof *ref, new_lbh->m, __func__ );	
	assert( new );
	
	memcpy(new, ref, sizeof *ref);
	
	item = adb_tavl_delete(&fbh, ref);

	assert( item );

	__adb_free( file, (void *)ref, sizeof *ref, lbh->m, __func__ );

	item = adb_tavl_insert( new_lbh, &new->node, (void *)new );
	assert( ! item );
	
      }
    memcpy( link, &new_link, sizeof *link);
  }
}

/*
 * returns obj_rec 
 */

static const adb_rec_t *
__adb_container_add(adb_tnx_t *tnx, adb_file_t *file, adb_bh_t *bh, 
		    const adb_rec_t *rec, adb_rec_t *obj_rec_backup, 
		    adb_link_t *link /* int offset */ /* of link field */, 
		    adb_ref_t *ref)
{
  adb_ref_t *new ;
  adb_bh_t *lbh ;
  void *ret ;

  switch ( link->type )
    {
    case ADB_LINK_TYPE_SHARED:
      {
	if ( ! bh->link_block_offset )
	  {
	    lbh = _adb_seg_add( tnx, file, &bh->link_block_offset ,
				ADB_MAGIC_TREE, 0 , -1LL );
	  }
	else
	  {
	    lbh = _adb_seg_load( tnx, file, bh->link_block_offset , 
				ADB_TNX_DIRTY );
	  }
	
	if ( ! ( lbh->state & ADB_TNX_DIRTY )) /* RO segment */
	  _adb_seg_dirty(tnx, lbh);

	/* "unsafe" malloc OK here */
	new = __adb_malloc( file, sizeof *ref, lbh->m, __FUNCTION__ );

	if ( ! new )
	  {
	    adb_link_to_file(tnx, file, bh, lbh );
	    return __adb_container_add( tnx, file, bh, 
					rec, obj_rec_backup, 
					link /* int offset */ /* of link field */, 
					ref);
	  }
	else
	  {
	    adb_bh_t fbh = { .magic = ADB_MAGIC_BH, .tree = &link->u.tree, 
			     .m = lbh->m, .state = ADB_TNX_DIRTY,
			     .addr = lbh->addr, .size = lbh->size,
			     .db = lbh->db, .offset = lbh->offset,
			     .cmp = lbh->cmp, .file = lbh->file } ;
	    memcpy(new, ref, sizeof *ref);
	    ret = adb_tavl_insert( &fbh, &new->node, new);
	    if ( ret )
	      ADB_WARNING("ref item exists!");
	    memcpy(&link->u.tree, &fbh.tree, sizeof link->u.tree);
	  }
		 
      }
      break ;
    case ADB_LINK_TYPE_PRIVATE:
      {
	adb_file_t *fake = alloca(sizeof *fake); /* pseudo file */
	adb_meta_file_t *mfake = alloca(sizeof *mfake);/* pseudo meta file */
	memcpy(fake, file->db->link, sizeof *fake); 
	memcpy(mfake, file->meta, sizeof(*file->meta) );
  
	fake->meta = mfake ; /* don't modify meta-scheme */
	fake->meta->data_start = link->u.ds.data_start ;
	fake->meta->data_start_type = link->u.ds.data_start_type ;
	fake->pseudo = TRUE ;
  

	new = __adb_safe_malloc(tnx, fake , (adb_rec_t *) ref, 
				&lbh, sizeof *ref, NULL);

	if ( fake->meta->data_start != link->u.ds.data_start )
	  link->u.ds.data_start = fake->meta->data_start ;

	if ( fake->meta->data_start_type != link->u.ds.data_start_type )
	  link->u.ds.data_start_type = fake->meta->data_start_type ;
	
	memcpy(new, ref, sizeof *ref);
	ret = adb_tavl_insert( lbh, &new->node, new );
	assert( !ret );

      }
      break ;
    default:
      abort();
      break ;
    }
  return rec ;

#if 0
  adb_ref_t *tmp = NULL ;
  void *refs = (void *)c->__refs + _bh->addr , *new ;
  size_t old_size = c->count * sizeof(adb_ref_t) , new_size /* bytes */;
  adb_bh_t *bh ;
  adb_rec_t *new_rec = NULL ;
/*   adb_container_t *c ; */
  int count = c->count ;
  int fail = 0 ;
  int c_offset = (void *)c - (void *)rec ;
  size_t old_alloc = c->alloc ;

  adb_ref_t *r ;

  assert((void *)ref->__mfile < (void *)0x1000000 );
  assert( c->magic == ADB_MAGIC_LINK );
  __adb_check_bh( _bh, __FUNCTION__, FALSE);

  new_size = ( c->count + 1 ) * sizeof(adb_ref_t) ;

  if ( new_size < c->alloc )
    { /* enough space */
      new = (void *)c->__refs + _bh->addr ;
      goto L_memcpy ;
    }
  /* save & free old refs storage */
  if ( count )
    {
      tmp = Malloc( old_size );
      memcpy(tmp, refs , old_size );
    }
      

  new_size = (( new_size / 32 ) + 1 ) * 32 ; /* new_size % 32 == 0 */
  new = __adb_safe_malloc(tnx, file, (adb_rec_t *)obj_rec_backup, &bh, new_size, &fail);
  /* search for the record ( when malloc failed in safe_malloc() ) or everthing 
     get stuffed */
  /* if ( malloc_err ) */
  if ( fail )
    {
/*       bh = new_bh = __adb_bh_dig(tnx, file, (adb_rec_t *)rec, NULL, ADB_BH_DIG_INSERT ); */
      new_rec = adb_tavl_find( bh, obj_rec_backup );  assert(new_rec);
      c = (void *)new_rec + c_offset ;
      c->__refs = (void *)new - bh->addr ;
      if ( __adb_debug && ADB_DEBUG_LINK )
	{
	  fprintf(stderr,"(link-fail) New bh: %p new rec: %p new: %p\n",
		  bh->m, new_rec, new );
	  fprintf(stderr,"(link-fail) c->__refs: %p c_offset %d\n",
		  c->__refs, c_offset);
	  fprintf(stderr,"(link-fail) new + old_size: %p next: %p\n",
		  (void *) new + old_size, (void *) new + old_size + sizeof(*ref));
	}
    }
/*   __adb_check_bh( bh, __FUNCTION__, FALSE); */

  c->__refs = (void *)new - bh->addr ;
  
  fprintf(stderr,"(link) New bh: %p new rec: %p new: %p\n",
	  bh->m, new_rec, new );
  fprintf(stderr,"(link) c->__refs: %p c_offset %d\n",c->__refs, c_offset);

  c->alloc = new_size ;
/*   assert( sizeof(*ref) == c->size ); */


  if ( count ) /* copy former part */
    {
      memcpy( new, tmp, old_size );

      if ( ! fail )
	__adb_free(file, refs, old_alloc, bh->m , __FUNCTION__ );
    }
  
  __adb_check_bh( bh, __FUNCTION__, FALSE);
 L_memcpy:

  c->count++ ;
  memcpy( (void *) new + old_size, ref , sizeof(*ref) );

  r = (void *) new + old_size ;
  assert(r->bh_offset);
  assert(r->diff);
  assert(r->__mfile);

  if ( __adb_debug && ADB_DEBUG_LINK )
    {
      fprintf(stderr, "-link- rec: %p\n", new);
      fprintf(stderr, "-link- link: %p __refs: %p\n", c, c->__refs);
    }
  __adb_check_bh( _bh, __FUNCTION__, FALSE);

  if ( count )
    free(tmp);

  return new_rec ? new_rec : rec ;
#endif /* 0 */
}

/* OBJ has A */
void
_adb_link_add(adb_tnx_t *tnx, adb_file_t *obj_file, adb_bh_t *obj_bh, 
	      const adb_rec_t *obj_rec, adb_rec_t *obj_rec_backup, adb_link_t *obj_c,
	      adb_file_t *a_file, adb_bh_t *a_bh, adb_rec_t *a_rec)
{
  unsigned int diff = (void *)a_rec - (void *) a_bh->m ;
  off_t offset = a_bh->offset + diff ;
  adb_ref_t Ref ;
  const adb_rec_t *back_ref ; /* back reference */
  adb_tr_t tr ;
  adb_link_t *bc = NULL ;
  adb_bh_t *bbh ;

  /* add new reference */

  __adb_tnx_check(tnx);

  Ref.magic = ADB_MAGIC_RECORD ; /* to be happy as pseudo file */
  Ref.diff = diff ;
  Ref.bh_offset = a_bh->offset ;
  Ref.size = __adb_vsize_get(a_file, a_rec ) + a_file->meta->all_data_size ;
  Ref.__mfile = (void *)a_file->meta - a_file->db->addr ;

  adb_lock_claim( obj_file, Ref.bh_offset, Ref.bh_offset + (off_t )Ref.diff, 
		  ADB_LOCK_REC, ADB_LOCK_WRITE);

  obj_rec = __adb_container_add(tnx, obj_file, obj_bh, obj_rec, obj_rec_backup, 
		      obj_c, &Ref);

  adb_lock_free( obj_file, Ref.bh_offset, Ref.bh_offset + (off_t )Ref.diff, 
		 ADB_LOCK_REC, ADB_LOCK_WRITE);

  /* back reference */

  if ( adb_find_by_name(tnx, obj_file->db->back_refs, &tr, &back_ref, TRUE,
			  "offset", offset ,
			  NULL ) ) 
    { /* no back_ref now */ 
      adb_lock_claim( obj_file->db->back_refs, 0LL, 0LL, ADB_LOCK_FILE, ADB_LOCK_WRITE );
      back_ref = adb_insert_by_name( tnx, obj_file->db->back_refs,
				     "offset", offset ,
				     NULL );

      adb_lock_free( obj_file->db->back_refs, 0LL, 0LL, ADB_LOCK_FILE, ADB_LOCK_WRITE );

      assert( back_ref != NULL ) ; 
      /* yep again to fill bh */
      bbh = __adb_bh_dig(tnx, obj_file->db->back_refs, (adb_rec_t *)back_ref, NULL, 
			 ADB_BH_DIG_INSERT ); /* to init empty file */
    }
  else
    bbh = tr.bhs[0] ;
  
  {
    size_t back_ref_size ;
    adb_rec_t *back_ref_backup ;
    
    diff = (void *)obj_rec - (void *)obj_bh->m ;

    /* back refs points to LINK field */
    Ref.magic = ADB_MAGIC_RECORD ; /* to be happy as pseudo file */
    Ref.diff = diff ;
    Ref.bh_offset = obj_bh->offset ;
    Ref.size = __adb_vsize_get(obj_file, (adb_rec_t *)obj_rec) + 
      obj_file->meta->all_data_size;

    Ref.__mfile = (void *) obj_file->meta - obj_file->db->addr ;

    adb_fld_get(a_file->db->back_refs, back_ref, "refs", &bc , NULL );

    adb_lock_claim( a_file->db->back_refs, Ref.bh_offset, 
		    Ref.bh_offset + (off_t )Ref.diff, 
		    ADB_LOCK_REC, ADB_LOCK_WRITE);

    back_ref_size = __adb_vsize_get( a_file->db->back_refs, (adb_rec_t *) back_ref) +
      a_file->db->back_refs->meta->all_data_size ;

    back_ref_backup = Malloc(back_ref_size);
    memcpy(back_ref_backup, back_ref, back_ref_size );

    __adb_container_add(tnx, a_file->db->back_refs, bbh, back_ref, back_ref_backup, 
			bc, &Ref );

    adb_lock_free( a_file->db->back_refs, Ref.bh_offset, 
		   Ref.bh_offset + (off_t )Ref.diff, 
		   ADB_LOCK_REC, ADB_LOCK_WRITE);

    free(bc);
  }
  __adb_tnx_check(tnx);
}

static int
Adb_check_magic(adb_magic_t magic, adb_magic_t expected)
{
  if ( magig != expected )
    {
      adb_errno = ADB_ERR_CROSS_DB_LINK ;
      return adb_errno ;
    }
  return 0 ;
}

#endif /* MORE_BUGS */

adb_field_t *
Adb_fld_find(adb_file_t *file, char *name)
{
  adb_field_t *fld, tmp_fld ;
  adb_bh_t fbh = { .magic = ADB_MAGIC_BH, 
		   .cmp = Adb_field_cmp, 
		   .addr = file->db->addr, 
		   .tree = file->fields } ;

  strncpy( tmp_fld.name , name , ADB_NAME_MAX - 1 );

  if ( ( fld = ( adb_field_t *) adb_tavl_find ( &fbh, &tmp_fld ) ) )
    return fld ;
  ADB_WARNING("Can't find field `%s' in file `%s'",name, file->meta->name);
  adb_errno = ADB_ERR_NOT_FIELD;
  return NULL ;
}


inline void
__adb_get_ref(adb_ref_t *ref, adb_bh_t *bh, void *p )
{
  ref->bh_offset = bh->offset ;
  ref->diff = p - (void *)bh->m  ;
}

inline off_t
__adb_get_offset(adb_bh_t *bh, void *p )
{
  off_t diff = p - (void *)bh->m ;
  return bh->offset + diff ;
}

int
adb_obj_has_a(adb_tnx_t *tnx, char *name, 
	      adb_file_t *obj_file, adb_rec_t *obj_rec,
	      adb_file_t *a_file, adb_rec_t *a_rec )
{
  adb_bh_t *obj_bh =  Adb_bh_dig(tnx, obj_file, obj_rec, NULL, ADB_BH_DIG_FIND );
  adb_bh_t *a_bh =  Adb_bh_dig(tnx, a_file, a_rec, NULL, ADB_BH_DIG_FIND );
  adb_field_t *fld, tmp_fld ;

  size_t size ;
  adb_link_t *l ;
  void *r ;

  ADB_ERROK();

  if ( obj_file->db != a_file->db )
    {
      ADB_WARNING("Cross database link %s => %s\n",
		  obj_file->meta->name, 
		  a_file->meta->name);

      adb_errno = ADB_ERR_CROSS_DB_LINK ;
      return adb_errno ;
    }

  if (obj_rec->magic != ADB_MAGIC_RECORD )
    {
      ADB_ERROR("Bad magic in obj_rec 0x%x\n",obj_rec->magic );
      adb_errno = ADB_ERR_INVALID_DATA ;
      return adb_errno ;
    }

  if ( a_rec->magic != ADB_MAGIC_RECORD )
    {
      ADB_ERROR("Bad magic in a_rec 0x%x\n",a_rec->magic );
      adb_errno = ADB_ERR_INVALID_DATA ;
      return adb_errno ;
    }

  if ( obj_file->magic != ADB_MAGIC_FILE )
    {
      ADB_ERROR("Bad magic in obj_file 0x%x\n", obj_file->magic );
      adb_errno = ADB_ERR_INVALID_DATA ;
      return adb_errno ;
    }
  
  if ( a_file->magic != ADB_MAGIC_FILE )
    {
      ADB_ERROR("Bad magic in a_file 0x%x\n", a_file->magic );
      adb_errno = ADB_ERR_INVALID_DATA ;
      return adb_errno ;
    }

  
  strncpy( tmp_fld.name , name , ADB_NAME_MAX - 1 );
  if ( ( fld = Adb_fld_find (obj_file, name ) ) )
    {
      switch ( fld->type )
	{
	case ADB_FIELD_LINK:
	  l = (void *)obj_rec + fld->offset ;
	  break ;
	default:
	  ADB_WARNING("Invalid field type %d, %d LINK needed ",
		      fld->type, ADB_FIELD_LINK);
	  adb_errno = ADB_ERR_NOT_FIELD;
	  return adb_errno ;
	  break ;
	}

      size = Adb_vsize_get(obj_file, (adb_rec_t *)obj_rec) + 
	obj_file->meta->all_data_size ;

      Adb_check_bh(obj_bh, __FUNCTION__, FALSE);
      Adb_check_bh(a_bh, __FUNCTION__, FALSE);

      adb_lock_claim( obj_file->db->link, 0LL, 0LL, 
		      ADB_LOCK_FILE, ADB_LOCK_WRITE);

      adb_lock_claim( obj_file->db->back_link, 0LL, 0LL, 
		      ADB_LOCK_FILE, ADB_LOCK_WRITE);

      
      if ( ! l->id ) /* first use */
	{
	   Adb_lock_claim_sb_with(obj_file->db, ADB__LOCK_LINK);
	   if ( obj_file->db->last_link_id == ULLONG_MAX )
	     {
	       ADB_WARNING("Database %s reached maximum link count %llu\n",
			   obj_file->db->name, obj_file->db->last_link_id);
	       adb_errno = ADB_ERR_TOO_MANY_LINKS ;
	       return adb_errno ;
	     }
	   else
	     {
	       l->id = obj_file->db->last_link_id++ ;
	     }
	   Adb_lock_free_sb_with(obj_file->db, ADB__LOCK_LINK);
	}

      r = adb_insert_by_name(tnx, obj_file->db->link , 
			     "id", l->id,
			     /*
			      * pointer part 
			      */
			     "__mfile", (void *)a_file->meta - a_file->db->addr,
			     "dst_bh_offset", a_bh->offset,
			     "dst_rec_offset", (void *)a_rec - a_bh->addr ,
			     NULL );
      if ( r )
	{
/* 	  adb_bh_t *lbh = Adb_bh_dig(tnx, a_file->db->link, r, NULL, ADB_BH_DIG_FIND ); */
	  if ( l->count == ULLONG_MAX )
	    ADB_WARNING("Can't increase count field `%s' has %lld links\n",
			fld->name,
			l->count);
	  else
	    l->count++ ;
	  
	  r = adb_insert_by_name(tnx, obj_file->db->back_link,
				 "__mfile", (void *)a_file->meta - a_file->db->addr,
				 "dst_bh_offset", a_bh->offset,
				 "dst_rec_offset", (void *)a_rec - a_bh->addr ,
				 "id", l->id,
				 NULL); 
	  if ( ! r )
	    {
	      ADB_BUG("Can't insert info $back_link file");
	    }
	}
      else
	{
	  ADB_BUG("Can't insert into $link file");
	}
      adb_lock_free( obj_file->db->link, 0LL, 0LL, 
		     ADB_LOCK_FILE, ADB_LOCK_WRITE);
      adb_lock_free( obj_file->db->back_link, 0LL, 0LL, 
		     ADB_LOCK_FILE, ADB_LOCK_WRITE);

      Adb_check_bh(obj_bh, __FUNCTION__, FALSE);
      Adb_check_bh(a_bh, __FUNCTION__, FALSE);

    }
  return adb_errno ;
}




/* should be called when moving record e.g. from split_tree() */
void
Adb_link_update(adb_tnx_t *tnx, adb_file_t *file, 
		 adb_bh_t *old_bh, adb_rec_t *old_rec,
		 adb_bh_t *new_bh, adb_rec_t *new_rec)
{
/*   unsigned int sb = file->db->addr ; */
  adb_tr_t tr ;
  const adb_rec_t *rec ;
  off_t bh_offset = old_bh->offset, bho ; /* of old record */
  int rec_offset = (int)((void *)old_rec - old_bh->addr) , ro ; /* of old record */
  if ( file == file->db->back_link ||
       file == file->db->link )
    return ;

  /* do I exist in back-link ?? */
  if ( adb_find_first_where(tnx, file->db->back_link, &tr, &rec, 
			    "__mfile", (void *)file->meta - file->db->addr,
			    "dst_bh_offset", bh_offset,
			    "dst_rec_offset", rec_offset ,
			    NULL ))
    return ; /* Nothing can be found */

  do
    {
/*     off_t link_bh_offset ; */
/*     int link_rec_offset ; */
    unsigned long long id ;
    int mfile ;
    adb_tr_t ltr ;
    const adb_rec_t *link_rec ;
    const void *to_free ;
    size_t size ;
    adb_rec_t *rv ;

    if ( ! adb_fld_get(file->db->back_link, rec, 
		       "dst_bh_offset", &bho , 
		       "dst_rec_offset", &ro,
		       "id", &id,
		       NULL ))
      ADB_BUG("adb_fld_get failed");

    
    if ( adb_find_first_where(tnx, file->db->link, &ltr, &link_rec, 
			      "id", id,
			      "__mfile", (void *)file->meta - file->db->addr ,
			      "dst_bh_offset", bh_offset,
			      "dst_rec_offset", rec_offset ,
			    NULL ))
      ADB_BUG("find_first_where failed");

    /* paranoid check if first link record is really poiting to moving rec */
    if ( ! adb_fld_get(file->db->link, link_rec, 
		       "id", &id,
		       "__mfile", &mfile,
		       "dst_bh_offset", &bho , 
		       "dst_rec_offset", &ro,
		       NULL ))
      ADB_BUG("adb_fld_get failed");

    assert( bho == bh_offset );
    assert( ro == rec_offset );
    
    do /* update of all link records pointing to moving record */
      {
	to_free = adb_tavl_delete( ltr.bhs[0], link_rec ); assert( to_free);

	size = Adb_vsize_get( file, (void *) link_rec) 
	  + file->meta->all_data_size ;
	Adb_free(file, (void *)to_free, size, ltr.bhs[0]->m, __func__ );

	rv = adb_insert_by_name(tnx, file->db->link , 
				 "id", id,
				 "__mfile", (void *)file->meta - file->db->addr,
				 "dst_bh_offset", new_bh->offset,
				 "dst_rec_offset", (void *)new_rec - new_bh->addr ,
				 NULL );
	assert( rv );
	
	adb_next(tnx, file->db->link , &ltr, &link_rec);
	if ( ! adb_fld_get(file->db->link, link_rec, 
			   "id", &id,
			   "__mfile", &mfile,
			   "dst_bh_offset", &bho , 
			   "dst_rec_offset", &ro,
			   NULL ))
	  ADB_BUG("adb_fld_get failed");
      } while ( bho == bh_offset && ro == rec_offset );
    /* END do {} while - link */
    /* update all back-link records "owned" by moving record */
    to_free = adb_tavl_delete( tr.bhs[0], rec ); assert(to_free);
    
    size = Adb_vsize_get( file, (void *) link_rec) 
      + file->meta->all_data_size ;
    Adb_free(file, (void *)to_free, size, tr.bhs[0]->m, __func__ );

    rv = adb_insert_by_name(tnx, file->db->back_link , 
				 "__mfile", (void *)file->meta - file->db->addr,
				 "dst_bh_offset", new_bh->offset,
				 "dst_rec_offset", (void *)new_rec - new_bh->addr ,
				 "id", id,
				 NULL );
    assert( rv );
    adb_next(tnx, file->db->back_link, &tr, &rec);
    
  } while  ( bho == bh_offset && ro == rec_offset ) ;
      /* END do {} while - back-link */ 
}


#if 0
  for ( i = 0 ; i < file->meta->last_order ; ++i )
    {
      if ( (void *)file->order[i] + sb )
	{
	  f = (void *)file->order[i] + sb ;
	  if ( f->magic == ADB_MAGIC_FIELD )
	    {
	      if ( f->type == ADB_FIELD_LINK || f->type == ADB_FIELD_CONTAINER )
		{
		  adb_bh_t *old_lbh , *new_lbh ;
		  old_link = (void *)old_rec + f->offset ;
		  if ( old_link->type == ADB_LINK_TYPE_SHARED &&
		       old_bh->link_block_offset )
		    {
		      old_lbh = _adb_seg_load( tnx, file, 
					       old_bh->link_block_offset , 
					       ADB_TNX_DIRTY );
		      if ( new_bh->link_block_offset )
			{
			  new_lbh = _adb_seg_add( tnx, file, &new_bh->link_block_offset ,
						  ADB_MAGIC_TREE, 0 , -1LL );
			}
		      else
			{
			  new_lbh = _adb_seg_load( tnx, file, 
						   new_bh->link_block_offset , 
						   ADB_TNX_DIRTY );
			}
		      {
			adb_ref_t *ref , *new_ref ;
			adb_tavl_traverser_t tr ;
			void *item ;
			adb_bh_t old_link_bh = { .magic = ADB_MAGIC_BH, 
					 .tree = &old_link->u.tree, 
					 .m = old_lbh->m, .state = ADB_TNX_DIRTY,
					 .addr = old_lbh->addr, .size = old_lbh->size,
					 .db = old_lbh->db, .offset = old_lbh->offset,
					 .cmp = old_lbh->cmp, .file = old_lbh->file } ;
			adb_bh_t new_link_bh = {  .magic = ADB_MAGIC_BH, 
					 .tree = &new_link->u.tree, 
					 .m = new_lbh->m, .state = ADB_TNX_DIRTY,
					 .addr = new_lbh->addr, .size = new_lbh->size,
					 .db = new_lbh->db, .offset = new_lbh->offset,
					 .cmp = new_lbh->cmp, .file = new_lbh->file } ;

			for ( ref = adb_tavl_t_first( &old_link_bh, &tr ) ;
			      ref ;
			      ref = adb_tavl_t_first( &old_link_bh, &tr ) )
			  {
			    new_ref = Adb_malloc0(file, sizeof *new_ref , 
						    new_link_bh.m, __func__ );
			    assert( new_ref );
			    memcpy(new_ref, ref, sizeof *ref );

			    item = adb_tavl_delete( &old_link_bh, ref);
			    assert( item );

			    Adb_free( file, (void *)ref, sizeof *ref, 
					old_link_bh.m, __func__ );

			    item = adb_tavl_insert( &new_link_bh, &new_ref->node, 
						    (void *)new_ref );
			    assert( ! item );
			  }
		      }
		    }
		}
	    }
	}
    }
  /*
   * back-reference stuff 
   */
  if ( file != file->db->back_refs &&
       adb_find_by_name(tnx, file->db->back_refs, &tr, &back_ref, TRUE,
			"offset", old_off ,
			 NULL ) == ADB_ERR_SUCCESS )
    { /* Argggh something is pointing to this record */
      adb_link_t *link = (void *) old_rec + fld->offset ;
      adb_ref_t *ref = (void *)link->__refs + old_bh->addr , *r , *s ;
      for ( i = 0 ; i < link->count ; ++i )
	{
	  size_t size ;
	  void *m ;
	  off_t offset;
	  int ret ;
	  r = ref + i ;
	  offset = r->bh_offset + r->diff ;

	  size = (( r->size / 4096 ) + 1 ) * 4096 ;
	  assert ( ! ( size % 4096 ));

	  m = mmap(NULL, size, PROT_READ|PROT_WRITE, MAP_SHARED, file->fd, offset );
	  assert( m != (void *)EOF );

	  s = m ;
	  
	  s->bh_offset = new_bh->offset ;
	  s->diff = (void *)new_rec - (void *)new_bh->m ;
	  ret = msync(m, size, MS_SYNC );
	  assert( ret != EOF );
	}
    }
#endif /* 0 BUGWARE */


void
Adb_link_test(adb_bh_t *bh, adb_rec_t *rec)
{
  int i ,/*  j, */ size ;
  adb_link_t *link ;
  adb_field_t *fld ;
/*   adb_ref_t *ref ; */
  adb_file_t *file = bh->file ;
  unsigned int sb = file->db->addr ;

  for ( i = 0 , size = 0 ; i < file->meta->last_order ; ++i )
    {
      if ( (void *)file->order[i] + sb )
	{
	  fld = (void *)file->order[i] + sb ;
	  if ( fld->magic == ADB_MAGIC_FIELD )
	    {
	      if ( fld->type == ADB_FIELD_LINK )
		{
		  link = (void *)rec + fld->offset ;
#if 0
		  if ( link->__refs )
		    {
		      ref = (void *)link->__refs + bh->addr ;
		      for ( j = 0 ; j < link->count ; ++j , ++ref )
			{
			  assert(ref->bh_offset);
			  assert(ref->diff >=  0x2000);
			  assert(ref->diff < 0x102000);
			  assert(ref->__mfile);
			}
		    }
#endif /* 0 */
		}
	    }
	}
    }
}

/* interface */
adb_rec_t *
adb_link_read(adb_tnx_t *tnx, adb_link_traverser_t *lt)
{
#if 0
  adb_ref_t *base , *ref ;
  adb_file_t *a_file ;
  adb_meta_file_t *a_mfile ;
  adb_bh_t *bh_tmp ;
  adb_rec_t *rec ;

  ADB_ERROK();
  
  if ( lt->pos == lt->link->count )
    return NULL ;

  base = (void *)lt->link->__refs + lt->bh->addr ;
  ref = base + lt->pos++ ;
  
  a_mfile = (void *)ref->__mfile + lt->file->db->addr ;
  a_file = Adb_file_open( lt->file->db, a_mfile );

  bh_tmp = _adb_seg_load( tnx, a_file, ref->bh_offset, ADB_TNX_NULL ) ;
  
  rec = (void *)bh_tmp->addr + ref->diff ;

  assert( rec->magic == ADB_MAGIC_RECORD);
  
#endif /* 0 */
  return (adb_rec_t *)NULL ;
}

/* interface */
adb_link_traverser_t *
adb_link_get(adb_tnx_t *tnx, adb_file_t *file, const adb_rec_t *rec, char *name)
{
  adb_link_traverser_t *lt = Malloc(sizeof(*lt));
  adb_field_t *fld ;

  ADB_ERROK();

  if ( ( fld = Adb_fld_find (file, name ) ) )
    {
      switch ( fld->type )
	{
	case ADB_FIELD_LINK:
	case ADB_FIELD_CONTAINER:
	  lt->bh = Adb_bh_dig(tnx, file, rec, NULL, ADB_BH_DIG_FIND);
	  assert( lt->bh );
	  lt->file = file ;
	  lt->link = (void *)rec + fld->offset ;
	  lt->pos = 0 ;
	  lt->rec = rec ;
	  return lt ;
	  break ;
	default:
	  ADB_WARNING("Invalid field type %d, %d (LINK or CONTAINER) needed ",
		      fld->type, ADB_FIELD_LINK);
	  adb_errno = ADB_ERR_NOT_FIELD;
	  return NULL ;
	  break ;
	}
    }
  else
    {
      adb_errno = ADB_ERR_NOT_FIELD;
      return NULL ;
    }
}

void
adb_link_free(adb_link_traverser_t *lt)
{
  free(lt);
}
