-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "aviation"
require "director"
require "shading"
require "shapes"
require "frames"
require "transforms"
require "moremath"
require "moreframes"
require "moreparticles"
require "units"

return bodies.system {
   bumper = bodies.ball {
      position = units.meters{3, 0, 800},
      velocity = units.knots{150, 0, 0},

      mass = {1, {0, 0, 0}, {1, 0, 0,  0, 1, 0,  0, 0, 1}},

      radius = 0.1,
   },

   shock = joints.slider {
      axis = {1, 0, 0},
      motor = {0, 5e3},
   },

   link = function (self)
	     self.shock.bodies = {self.fuselage, self.bumper}
	  end,

--    traverse = function (self)
-- 		 print (unpack(self.parent.shock.state))
-- 	      end,

   fuselage = resources.polyhedron "aviation/polyhedra/fuselage.lc" {
      position = units.meters {0, 0, 800},
      
      orientation = transforms.euler (0, 0, 0),
      velocity = units.knots{150, 0, 0},
      
      mass = {units.pounds (1170),
	      {0, 0, 0},
	      units.slugsquarefeet {361,  0, 0,
				    0, 466, 0,
				    0, 0, 749}},
      
      isairplane = true,
      
      pilot = frames.observer {
         position = units.meters {-0.2, 0, 0.5},
         orientation = { 0, 0, -1,
			 0, 1,  0,
		        -1, 0,  0}
      },
      
      --    camera = director.orbit {
      --       damping = 3000,
      
      --       radius = 50,
      --       azimuth = math.pi / 3,
      --       elevation = math.pi / 2,
      
      --       controls = frames.event {
      -- 	 x = 0,
      -- 	 y = 0,
      
      -- 	 motion = function (self, button, x, y, dx, dy)
      -- 		     local dx, dy
      -- 		     local c = 0.03
      
      -- 		     dx, dy = x - self.x, y - self.y
      
      -- 		     if button == 1 then
      -- 			self.parent.azimuth = self.parent.azimuth - c * dx
      -- 			self.parent.elevation = self.parent.elevation - c * dy
      -- 		     elseif button == 2 then
      -- 			self.parent.radius = math.clamp(self.parent.radius -
      -- 							c * dy, 0, 100)
      -- 			print(self.parent.radius)
      -- 		     end
      -- 		     self.x, self.y = x, y
      -- 		  end,
      --       },
      --    },
      
      -- Aerodynamic specs.
      
      dynamics = aircraft.airplane {
         area = units.squarefeet (98),
         span = units.feet (24.4),
         chord = units.feet (4.1),
	 
         drag = {
	    attack = {
		  -3.14159, 0.15000,
		  -2.93215, 0.20000,
		  -2.61799, 0.40000,
		  -2.09439, 0.65000,
		  -1.57079, 1.00000,
		  -1.04719, 0.65000,
		  -0.52359, 0.40000,
		  -0.26179, 0.08000,
		  -0.10471, 0.03500,
		  -0.08726, 0.03300,
		  -0.06981, 0.03100,
		  -0.05235, 0.03000,
		  -0.03490, 0.03000,
		  -0.01745, 0.03100,
	       0.00000, 0.03300,
	       0.01745, 0.03500,
	       0.03491, 0.03800,
	       0.05236, 0.04200,
	       0.06981, 0.04600,
	       0.08727, 0.05100,
	       0.10472, 0.05600,
	       0.13963, 0.06900,
	       0.15708, 0.07600,
	       0.17453, 0.08400,
	       0.19199, 0.09300,
	       0.20944, 0.10200,
	       0.22689, 0.11200,
	       0.24435, 0.11500,
	       0.26180, 0.12000,
	       0.27925, 0.13000,
	       0.29671, 0.13800,
	       0.31416, 0.14500,
	       0.33161, 0.15000,
	       0.34907, 0.16500,
	       0.36652, 0.17500,
	       0.38397, 0.35000,
	       0.52360, 0.65000,
	       1.04720, 1.00000,
	       1.57080, 0.65000,
	       2.09439, 0.40000,
	       2.61799, 0.20000,
	       3.05433, 0.15000,
	       3.14159, 0.15000
	    },
         },
         
         sideforce = {
	    sideslip = {
		  -math.pi, -math.pi * -0.31,
	       math.pi,  math.pi * -0.31
	    },
	    
	    roll = {
		  -math.pi, -math.pi * -0.037,
	       math.pi,  math.pi * -0.037
	    },
	    
	    yaw = {
		  -math.pi, -math.pi * 0.21,
	       math.pi,  math.pi * 0.21
	    },
	    
	    rudder = {
		  -math.pi, -math.pi * 0.187,
	       math.pi,  math.pi * 0.187
	    }
         },
         
         lift = {
	    attack = {
		  -2.93215, 0.30000,
		  -2.61799, 0.15000,
		  -2.09439, 0.10000,
		  -1.57079, 0.00000,
		  -0.06981, -0.15300,
		  -0.05235, -0.07600,
		  -0.03490, 0.00100,
		  -0.01745, 0.07800,
	       0.00000, 0.15500,
	       0.01745, 0.23200,
	       0.03491, 0.30900,
	       0.05236, 0.38600,
	       0.06981, 0.46300,
	       0.08727, 0.54000,
	       0.10472, 0.61700,
	       0.12217, 0.69400,
	       0.13963, 0.77100,
	       0.15708, 0.84800,
	       0.17453, 0.92500,
	       0.19199, 1.00200,
	       0.20944, 1.07900,
	       0.22689, 1.15600,
	       0.24435, 1.23300,
	       0.26180, 1.28000,
	       0.27925, 1.30000,
	       0.29671, 1.30000,
	       0.31416, 1.23000,
	       0.33161, 0.90000,
	       0.34907, 0.60000,
	       0.36652, 0.57500,
	       0.38397, 0.55000,
	       0.52360, 0.30000,
	       1.04720, 0.20000,
	       1.57080, 0.00000,
	       2.09439, -0.10000,
	       2.61799, -0.15000,
	       3.05433, -0.30000,
	       3.14159, -0.10000
	    },
	    
	    pitch = {
		  -math.pi, -math.pi * 3.9,
	       math.pi,  math.pi * 3.9
	    },
	    
	    elevators = {
		  -math.pi, -math.pi * 0.43,
	       math.pi,  math.pi * 0.43
	    }
         },
         
         roll = {
	    sideslip = {
		  -math.pi, -math.pi * -0.089,
	       math.pi,  math.pi * -0.089
	    },
	    
	    roll = {
		  -math.pi, -math.pi * -0.47,
	       math.pi,  math.pi * -0.47
	    },
	    
	    yaw = {
		  -math.pi, -math.pi * 0.096,
	       math.pi,  math.pi * 0.096
	    },
	    
	    ailerons = {
		  -math.pi, -math.pi * -0.178,
	       math.pi,  math.pi * -0.178
	    },
	    
	    rudder = {
		  -math.pi, -math.pi * 0.0147,
	       math.pi,  math.pi * 0.0147
	    },
         },
         
         pitch = {
	    attack = {
		  -3.14159, 0.00000,
		  -2.79253, 0.37000,
		  -2.44346, 0.59000,
		  -2.09439, 0.73000,
		  -1.74533, 0.79000,
		  -1.57080, 0.80000,
		  -1.39626, 0.78630,
		  -1.04720, 0.72469,
		  -0.69813, 0.60145,
		  -0.34907, 0.38920,
		  -0.17453, 0.11533,
		  -0.08727, 0.03767,
	       0.00000, -0.04000,
	       0.08727, -0.11767,
	       0.17453, -0.19533,
	       0.26148, -0.27300,
	       0.27925, -0.29000,
	       0.29671, -0.31500,
	       0.33161, -0.37500,
	       0.34907, -0.40500,
	       0.36652, -0.42000,
	       0.38397, -0.42000,
	       0.41888, -0.39000,
	       0.47124, -0.31500,
	       0.52360, -0.21500,
	       0.61086, -0.01000,
	       0.62832, 0.01500,
	       0.68068, 0.06500,
	       0.71558, 0.08500,
	       0.78540, 0.06000,
	       0.85521, 0.00500,
	       0.95993, -0.15000,
	       1.22173, -0.59500,
	       1.30900, -0.72000,
	       1.39626, -0.81500,
	       1.48353, -0.87500,
	       1.57080, -0.90000,
	       1.74533, -0.89000,
	       2.09439, -0.80000,
	       2.44346, -0.64000,
	       2.79253, -0.36000,
	       3.14159, 0.00000
	    },
	    
	    attackrate = {
		  -math.pi, -math.pi * -5.2,
	       math.pi,  math.pi * -5.2
	    },
	    
	    pitch = {
		  -math.pi, -math.pi * -12.4,
	       math.pi,  math.pi * -12.4
	    },
	    
	    elevators = {
		  -math.pi, -math.pi * -1.28,
	       math.pi,  math.pi * -1.28
	    },
         },
	 
         yaw = {
	    sideslip = {
		  -math.pi, -math.pi * 0.065,
	       math.pi,  math.pi * 0.065
	    },	 
	    
	    roll = {
		  -math.pi, -math.pi * -0.03,
	       math.pi,  math.pi * -0.03
	    },
	    
	    yaw = {
		  -math.pi, -math.pi * -0.099,
	       math.pi,  math.pi * -0.099
	    },
	    
	    ailerons = {
		  -math.pi, -math.pi * -0.053,
	       math.pi,  math.pi * -0.053
	    },
	    
	    rudder = {
		  -math.pi, -math.pi * -0.0657,
	       math.pi,  math.pi * -0.0657
	    }
         },
	 
         controls = frames.event {
	    buttonpress = function(self, button, x, y)
	       local x_0, y_0 = x, y
			     
	       if button == 1 then
		  self.motion = function (self, button, x, y)
		     local w = graphics.window
						 
		     self.parent.ailerons = math.clamp(-(x - x_0) / w[1],
						       units.degrees (-15),
						       units.degrees (20))
		     
		     self.parent.elevators = math.clamp(-(y - y_0) / w[2],
							units.degrees (-23),
							units.degrees (28))
		  end
	       elseif button == 2 then
		  self.motion = function (self, button, x, y)
		     local w = graphics.window
				   
		     self.parent.rudder = math.clamp((x - x_0) / w[1],
						     units.degrees (-16),
						     units.degrees (16))
		  end
	       end
	    end,
            
	    buttonrelease = function(self, button)
	       self.motion = nil
			       
	       self.parent.ailerons = 0
	       self.parent.elevators = 0
	       self.parent.rudder = 0
	    end
         },
      },
      
      engine = propulsion.piston {
         orientation = transforms.euler (0, 90, 0),
	 
         diameter = units.feet (6.25),
         inertia = units.slugsquarefeet(1.67),
         throttle = 1.0,
         ratio = 1.3,
	 
         --        traverse = function (self)
         -- 		     print (self.speed)
         -- 		  end,
	 
         brakepower = {
	    units.rotationsperminute(0)      , units.metrichorsepower(0), 
	    units.rotationsperminute(401.07) , units.metrichorsepower(0), 
	    units.rotationsperminute(1002.68), units.metrichorsepower(120), 
	    units.rotationsperminute(2005.35), units.metrichorsepower(234), 
	    units.rotationsperminute(2750.20), units.metrichorsepower(274), 
	    units.rotationsperminute(2998.48), units.metrichorsepower(240), 
	    units.rotationsperminute(3504.59), units.metrichorsepower(0)
         },
         
         thrust = {0.0,   0.068,
		   0.1,   0.068,
		   0.2,   0.067,
		   0.3,   0.066,
		   0.4,   0.064,
		   0.5,   0.061,
		   0.6,   0.057,
		   0.7,   0.050,
		   0.8,   0.040,
		   0.9,   0.029,
		   1.0,   0.019,
		   1.1,   0.008,
		   1.2,   -0.001,
		   1.3,   -0.008,
		   1.4,   -0.019,
		   1.5,   -0.029,
		   1.6,   -0.040,
		   1.7,   -0.050,
		   1.8,   -0.057,
		   1.9,   -0.061,
		   2.0,   -0.064,
		   2.1,   -0.066,
		   2.2,   -0.067,
		   2.3,   -0.068,
		   5.0,   -0.068,
		   10.0,   -0.068},
	 
         power = {0.0,   0.0580,
		  0.1,   0.0620,
		  0.2,   0.0600,
		  0.3,   0.0580,
		  0.4,   0.0520,
		  0.5,   0.0450,
		  0.6,   0.0421,
		  0.7,   0.0389,
		  0.8,   0.0346,
		  0.9,   0.0280,
		  1.0,   0.0202,
		  1.1,   0.0111,
		  1.2,   0.0075,
		  1.3,   0.0111,
		  1.4,   0.0202,
		  1.5,   0.0280,
		  1.6,   0.0346,
		  1.7,   0.0389,
		  1.8,   0.0421,
		  1.9,   0.0436,
		  2.0,   0.0445,
		  2.1,   0.0445,
		  2.2,   0.0442,
		  2.3,   0.0431,
		  2.4,   0.0424,
		  5.0,   0.0413,
		  10.0,  0.0413},
	 
         controls = frames.event {
	    buttonpress = function(self, button)
	       if button == 4 then
		  self.parent.throttle = self.parent.throttle + 0.1
	       elseif button == 5 then
		  self.parent.throttle = self.parent.throttle - 0.1
	       end
	    end,
         }
      },
      
--       autopilot = frames.transform {
-- 	 transform = function (self)
-- 			self.parent.dynamics.elevators = 0.01 * (self.parent.velocity[3] + 0)
-- 			self.parent.dynamics.ailerons = 0.0
-- 		     end
--       },

      -- The display.
      
      display = options.simple and widgets.display {
	 size = {graphics.window[1] / graphics.window[2], 1},
	 align = {-1, -1},
	 
	 widgets.row {
	    size = {0, 1},
	    align = {0, -1},

	    altimeter = widgets.column {
	       padding = {0.01, 0},

	       face = widgets.altimeter {
		  color = {1.0, 0.8, 0.2},
		  opacity = 0.6,

		  radius = {0.1, 0.07},

		  traverse = function (self)
		     self.reading = self.ancestry[4].position[3]
	          end,

		  [1] = resources.large {text = "0"},
		  [2] = resources.large {text = "1"},
		  [3] = resources.large {text = "2"},
		  [4] = resources.large {text = "3"},
		  [5] = resources.large {text = "4"},
		  [6] = resources.large {text = "5"},
		  [7] = resources.large {text = "6"},
		  [8] = resources.large {text = "7"},
		  [9] = resources.large {text = "8"},
		  [10] = resources.large {text = "9"},
	       },

	       label = resources.regular {
		  color = math.scale ({1.0, 0.8, 0.2}, 1.2),
		  padding = {0.01, 0.01},

		  text = "Altimeter",
	       }
	    },

	    airspeed = widgets.column {
	       padding = {0.01, 0},

	       face = widgets.clock {
		  color = {1.0, 0.8, 0.2},
		  opacity = 0.6,
		  
		  radius = {0.1, 0.07},
		  range = {60, 240},
		  spacing = {5, 4},
		  spread = {2 * math.pi / 5, 9 * math.pi / 5},

		  traverse = function (self)
		     self.reading = math.length(self.ancestry[4].velocity) /
		     units.milesperhour (1)
	          end,

		  [1] = resources.small {text = "60"},
		  [2] = resources.small {text = "80"},
		  [3] = resources.small {text = "100"},
		  [4] = resources.small {text = "120"},
		  [5] = resources.small {text = "140"},
		  [6] = resources.small {text = "160"},
		  [7] = resources.small {text = "180"},
		  [8] = resources.small {text = "200"},
		  [9] = resources.small {text = "220"},
		  [10] = resources.small {text = "240"},
	       },

	       label = resources.regular {
		  color = math.scale ({1.0, 0.8, 0.2}, 1.2),
		  padding = {0.01, 0.01},

		  text = "Airspeed",
	       }
	    },

	    heading = widgets.column {
	       padding = {0.01, 0},
	       
	       face = widgets.compass {
		  color = {1.0, 0.8, 0.2},
		  opacity = 0.6,

		  radius = {0.07, 0.05},
		  range = {0, 360},
		  spacing = {5, 2},
		  spread = {0, 2 * math.pi},

	       
		  traverse = function (self)
		     local v = math.project (transforms.fromnode (
						self.ancestry[4],
						{1, 0, 0}),
					     {0, 0, 1})
				
		     self.reading = math.deg(math.acos(v[1]))
		  end,

		  [1] = resources.small {text = "N"},
		  [4] = resources.small {text = "3"},
		  [7] = resources.small {text = "6"},
		  [10] = resources.small {text = "E"},
		  [13] = resources.small {text = "12"},
		  [16] = resources.small {text = "15"},
		  [19] = resources.small {text = "S"},
		  [22] = resources.small {text = "21"},
		  [25] = resources.small {text = "24"},
		  [28] = resources.small {text = "W"},
		  [31] = resources.small {text = "30"},
		  [34] = resources.small {text = "33"},
	       },

	       label = resources.regular {
		  color = math.scale ({1.0, 0.8, 0.2}, 1.2),
		  padding = {0.01, 0.01},

		  text = "Heading"
	       }
	    },

	    tachometer = widgets.column {
	       padding = {0.01, 0},

	       face = widgets.clock {
		  color = {1.0, 0.8, 0.2},
		  opacity = 0.6,

		  radius = {0.07, 0.05},
		  range = {0, 3500},
		  spacing = {100, 5},
		  spread = {-4 * math.pi / 5, 4 * math.pi / 5},

		  traverse = function (self)
		     self.reading = self.ancestry[4].engine.speed / math.pi * 30
	          end,

		  [1] = resources.small {text = "0"},
		  [2] = resources.small {text = "5"},
		  [3] = resources.small {text = "10"},
		  [4] = resources.small {text = "15"},
		  [5] = resources.small {text = "20"},
		  [6] = resources.small {text = "25"},
		  [7] = resources.small {text = "30"},
		  [8] = resources.small {text = "35"},
	       },

	       label = resources.regular {
		  color = math.scale ({1.0, 0.8, 0.2}, 1.2),
		  padding = {0.01, 0.01},

		  text = "Tachometer"
	       }
	    },
 	 }
      },
      
      -- The cockpit.
      
      cockpit = not options.simple and frames.mustache {
	 --    frames.transform {
	 --       position = {-0, 0, 0.1},
	 --       orientation = {0, 0, -1,
	 -- 		     -1, 0,  0,
	 -- 		     0, 1,  0},
	 
         position = units.meters{0, -0.34, -0.11},
         
         section = shading.cook {
	    diffuse = resources.mirrored "aviation/imagery/section.lc",
	    specular = resources.periodic "aviation/imagery/sectiongloss.lc",
	    parameter = {1.5, 0.1},
	    
	    mesh = resources.static "aviation/meshes/section.lc" {}
         },
         
         star = shading.cook {
	    diffuse = math.scale({1, 0.3, 0.2}, 0.6),
	    specular = resources.periodic "aviation/imagery/sectiongloss.lc",
	    parameter = {1.5, 0.1},
	    
	    mesh = resources.static "aviation/meshes/star.lc" {}
         },
	 
         propeller = frames.top {
	    position = units.meters{0, -0.12, -2.574},
	    axis = {0, 0, 1},
	    
	    -- The trails.
	    
	    [1] = shapes.pie {
	       orientation = transforms.euler (0, 0, 60),
	       color = {0.5, 0.5, 0.5},
	       opacity = 0.4,
	       radius = 0.9,
	       segments = 16,
	    },
	    
	    [2] = shapes.pie {
	       orientation = transforms.euler (0, 0, 180),
	       color = {0.5, 0.5, 0.5},
	       opacity = 0.4,
	       radius = 0.9,
	       segments = 16,
	    },
	    
	    [3] = shapes.pie {
	       orientation = transforms.euler (0, 0, 300),
	       color = {0.5, 0.5, 0.5},
	       opacity = 0.4,
	       radius = 0.9,
	       segments = 16,
	    },
	    
	    -- The blades.
	    
	    blades = shading.anisotropic {
	       diffuse = math.scale({1, 1, 1}, 0.7),
	       specular = math.scale({1, 1, 1}, 0.7),
	       parameter = {16, 0.1},
	       
	       mesh = resources.static "aviation/meshes/propeller.lc" {},
	    },
	    
	    transform = function (self)
	       local w = self.ancestry[2].engine.speed /
		  (2 * math.pi)
	       local w_0 = 24
	       local N = 3
			   
	       self.spin = math.mod(2 * math.pi * w / w_0,
				    2 * math.pi / N) * w_0
			   
	       if self.spin > math.pi / N * w_0 then
		  self.spin = self.spin - 2 * math.pi / N * w_0
	       end
			   
	       for i = 1, 3 do
		  self[i].arc = -self.spin / w_0
	       end
			   
-- 		      print (self.parent.engine.throttle,
-- 			     self.parent.engine.speed / (2 * math.pi) * 60,
-- 		             self.parent.engine.speed / (2 * math.pi) * 60 *
-- 			     self.parent.engine.ratio,
-- 		             self.parent.engine.thrust)
	    end
         },
	 
         dashboard = shading.cook {
	    diffuse = math.scale ({1, 1, 1}, 0.01),
	    specular = resources.mirrored "aviation/imagery/scratches.lc",
	    parameter = {0.75, 0.3},
	    
	    mesh = resources.static "aviation/meshes/dashboard.lc" {},
         },
	 
	 border = shapes.lines {
	    position = units.meters {-0.19, 0.125, -0.52},
	    
	    color = {1, 1, 0.7},
	    opacity = 0.25,
	    width = 2,
	    
	    units.meters {0.09, 0.05, 0}, 
	    units.meters {0.1, 0.04, 0}, 
	    units.meters {-0.1, 0.04, 0}, 
	    units.meters {-0.09, 0.05, 0}, 
	    units.meters {-0.09, -0.05, 0}, 
	    units.meters {-0.1, -0.04, 0}, 
	    units.meters {0.1, -0.04, 0}, 
	    units.meters {0.09, -0.05, 0}, 
	    units.meters {-0.09, -0.05, 0}, 
	    units.meters {0.09, -0.05, 0}, 
	    units.meters {0.1, 0.04, 0}, 
	    units.meters {0.1, -0.04, 0}, 
	    units.meters {0.09, 0.05, 0}, 
	    units.meters {-0.09, 0.05, 0}, 
	    units.meters {-0.1, -0.04, 0}, 
	    units.meters {-0.1, 0.04, 0}, 
	 },

	 smallborder = shapes.lines {
	    position = units.meters {0.175, 0.133, -0.52},
	    
	    color = {1, 1, 0.7},
	    opacity = 0.25,
	    width = 2,

	    units.meters {0.063, 0.034369, 0}, 
	    units.meters {0.07, 0.027369, 0}, 
	    units.meters {-0.07, 0.027369, 0}, 
	    units.meters {-0.063, 0.034369, 0}, 
	    units.meters {-0.063, -0.035631, 0}, 
	    units.meters {-0.07, -0.028631, 0}, 
	    units.meters {0.07, -0.028631, 0}, 
	    units.meters {0.063, -0.035631, 0}, 
	    units.meters {-0.063, -0.035631, 0}, 
	    units.meters {0.063, -0.035631, 0}, 
	    units.meters {0.07, 0.027369, 0}, 
	    units.meters {0.07, -0.028631, 0}, 
	    units.meters {0.063, 0.034369, 0}, 
	    units.meters {-0.063, 0.034369, 0}, 
	    units.meters {-0.07, 0.027369, 0}, 
	    units.meters {-0.07, -0.028631, 0}, 
	 },

	 switchborder = shapes.lines {
	    position = units.meters {0.175, 0.084, -0.52},
	    
	    color = {1, 1, 0.7},
	    opacity = 0.25,
	    width = 2,

	    {0.063, 0.009784, 0}, 
	    {0.07, 0.002784, 0}, 
	    {-0.07, 0.002784, 0}, 
	    {-0.063, 0.009784, 0}, 
	    {-0.063, -0.011046, 0}, 
	    {-0.07, -0.004046, 0}, 
	    {0.07, -0.004046, 0}, 
	    {0.063, -0.011046, 0}, 
	    {-0.063, -0.011046, 0}, 
	    {0.063, -0.011046, 0}, 
	    {0.07, 0.002784, 0}, 
	    {0.07, -0.004046, 0}, 
	    {0.063, 0.009784, 0}, 
	    {-0.063, 0.009784, 0}, 
	    {-0.07, 0.002784, 0}, 
	    {-0.07, -0.004046, 0}, 
	 },
	 
         altimeter = frames.transform {
	    position = units.meters {-0.24, 0.125, -0.513},
	    
	    bezel = shading.cook {
	       diffuse = math.scale ({1, 1, 1}, 0.05),
	       specular = resources.mirrored "aviation/imagery/scratches.lc",
	       parameter = {0.75, 0.3},
	    
	       mesh = resources.static "aviation/meshes/bezel.lc" {
		  position = units.meters {0, 0, -0.007},
	       },
	    },
  
	    face = shading.cook {
	       diffuse = resources.clamped "aviation/imagery/altimeter.lc",
	       specular = resources.mirrored "aviation/imagery/specularity.lc",
	       parameter = {1.5, 0.25},
	       
	       mesh = resources.static "aviation/meshes/altimeter.lc" {},
	    },
	    
	    hundreds = shading.lambert {
	       diffuse = {1, 1, 1},
	       mesh = resources.static "aviation/meshes/hundreds.lc" {},
	       
	       traverse = function (self)
		  local h = self.ancestry[3].position[3] 
		  self.orientation = transforms.euler (0, 0,
						       -(h % 1e3) / 1e2 * 36)
	       end
	    },
	    
	    thousands = shading.lambert {
	       diffuse = {1, 1, 1},
	       
	       mesh = resources.static "aviation/meshes/thousands.lc" {},
	       
	       traverse = function (self)
	          local h = self.ancestry[3].position[3] 
		  self.orientation = transforms.euler (0, 0,
						       -(h % 1e4) / 1e3 * 36)
	       end
	    },
	    
	    tenthousands = shading.lambert {
	       diffuse = {1, 1, 1},
	       mesh = resources.static "aviation/meshes/tenthousands.lc" {},
	       
	       traverse = function (self)
	          local h = self.ancestry[3].position[3] 
		  self.orientation = transforms.euler (0, 0,
						       -(h % 1e5) / 1e4 * 36)
	       end
	    }
         },
         
         airspeed = frames.transform {
	    position = units.meters {-0.14, 0.125, -0.513},
	    
	    bezel = shading.phong {
	       diffuse = math.scale ({1, 1, 1}, 0.05),
	       specular = resources.mirrored "aviation/imagery/scratches.lc",
	       parameter = {0.75, 0.3},
	    
	       mesh = resources.static "aviation/meshes/bezel.lc" {
		  position = units.meters {0, 0, -0.007},
	       },
	    },
	    
	    face = shading.cook {
	       diffuse = resources.clamped "aviation/imagery/airspeed.lc",
	       specular = resources.mirrored "aviation/imagery/specularity.lc",
	       parameter = {1.5, 0.25},
	       
	       mesh = resources.static "aviation/meshes/airspeed.lc" {},
	    },
	    
	    knots = shading.lambert {
	       diffuse = {1, 1, 1},
	       mesh = resources.static "aviation/meshes/knots.lc" {},
	       
	       traverse = function (self)
		  local v = math.length(self.ancestry[3].velocity) /
		            units.milesperhour (1)
			  
	          self.orientation = transforms.euler (0, 0,
						       -(v % 260) / 260 * 360)
	       end
	    },
         },
         
         tachometer = frames.transform {
	    position = units.meters {0.14, 0.133, -0.512},
	    
	    bezel = shading.phong {
	       diffuse = math.scale ({1, 1, 1}, 0.05),
	       specular = resources.mirrored "aviation/imagery/scratches.lc",
	       parameter = {0.75, 0.3},
	    
	       mesh = resources.static "aviation/meshes/smallbezel.lc" {
		  position = units.meters {0, 0, -0.008},
	       },
	    },
	    
	    face = shading.cook {
	       diffuse = resources.clamped "aviation/imagery/tachometer.lc",
	       specular = resources.mirrored "aviation/imagery/specularity.lc",
	       parameter = {1.5, 0.25},
	       
	       mesh = resources.static "aviation/meshes/tachometer.lc" {},
	    },
	    
	    needle = shading.lambert {
	       diffuse = {1, 1, 1},
	       mesh = resources.static "aviation/meshes/rpms.lc" {},
	       
	       traverse = function (self)
		  local nu = self.ancestry[3].engine.speed / (2 * math.pi) * 60
			     
		  self.orientation = transforms.euler (0, 0, 121 - nu / 14)
	       end
	    },
         },
         
         compass = frames.transform {
	    position = units.meters {0.21, 0.133, -0.512},
	    
	    bezel = shading.phong {
	       diffuse = math.scale ({1, 1, 1}, 0.05),
	       specular = resources.mirrored "aviation/imagery/scratches.lc",
	       parameter = {0.75, 0.3},
	    
	       mesh = resources.static "aviation/meshes/smallbezel.lc" {
		  position = units.meters {0, 0, -0.008},
	       },
	    },
	    
	    face = shading.cook {
	       diffuse = resources.clamped "aviation/imagery/compassface.lc",
	       specular = resources.mirrored "aviation/imagery/specularity.lc",
	       parameter = {1.5, 0.25},
	       
	       mesh = resources.static "aviation/meshes/compassface.lc" {},
	    },
	    
	    dial = shading.lambert {	       
	       diffuse = resources.clamped "aviation/imagery/compassdial.lc",
	       specular = math.scale ({0.12, 0.17, 0.19}, 1), 
	       
	       mesh = resources.static "aviation/meshes/compassdial.lc" {
		  position = units.meters {0, 0, -0.022},
	       
		  traverse = function (self)
		     local v = math.project (transforms.fromnode (
						self.ancestry[4],
						{1, 0, 0}),
					     {0, 0, 1})
				
		     self.orientation = transforms.rotation (
			math.deg(math.acos(v[1])),
			{0, 1, 0}
		     )
		  end
	       },
	    },
         },
	 
         switches = shading.cook {
	    position = units.meters {0.116, 0.085, -0.52},
	    
	    diffuse = math.scale ({1.5, 0.9, 1}, 0.04),
	    specular = math.scale ({0.12, 0.17, 0.19}, 0.7), 
	    parameter = {1.3, 0.15},
	    
	    meshes = arrays.linear {
	       mold = resources.static "aviation/meshes/switch.lc",
	       size = 6,
	       length = 0.143
	    }
         },
	 
         light = frames.transform {
	    position = units.meters {0.265, 0.085, -0.52},
	    
	    nut = shading.cook {
	       diffuse = math.scale ({1.5, 0.9, 1}, 0.04),
	       specular = math.scale ({0.12, 0.17, 0.19}, 0.7), 
	       parameter = {1.3, 0.15},
	       
	       mesh = resources.static "aviation/meshes/nut.lc" {},
	    },
	    
	    bulb = shading.phong {
	       diffuse = math.scale ({0.9, 0.2, 0.2}, 0.3),
	       specular = math.scale ({1, 1, 1}, -0.1),
	       parameter = 1,
	       
	       mesh = resources.static "aviation/meshes/bulb.lc" {},
	    },
	 },
      },
      
      dust = shading.sprites {
         position = units.meters {3, 0, -1},
         texture = resources.clamped "aviation/imagery/puff.lc",
	 
         particles = particulate.stream {
	    orientation = transforms.euler (0, 90, 0),
	    
	    granularity = 1,
	    
	    size = 32,
	    clustering = 24,
	    
	    scale = {1, 0.2},
	    density = 3e-3,
	    diffusion = 0.5,
	    support = 25,
	    stiffness = {10, 0.15},
	    viscosity = {1, 5},
	    mobility = 7e6,
	    
	    color = {0.8, 0.75, 0.55},
	    radius = {5e-4, 1e-4},
	    albedo = {0.5, 0.1},
	    opacity = {0.1, 0.03},
	    
	    clock = frames.timer {
	       period = 0,
	       
	       tick = function (self)
			 local V = math.length(self.ancestry[3].velocity)
			 self.parent.velocity = {1 * V, 0.2 * V, 0.15}
		      end
	    },
	 },
      },
   } 
}
