-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "network"
require "resources"
require "derived"
require "aviation"
require "nature"
require "widgets"
require "shapes"
require "shading"
require "transforms"
require "frames"
require "moremath"
require "moremeshes"
require "particulate"
require "physics"
require "units"

-- These values are fixed.

graphics.title = "Aviation"
graphics.window = {800, 600}
-- graphics.hide = true
graphics.canvas = {0.07, 0.07, 0.075}

derived.gee = units.meters(-9.81)
derived.range = units.meters{0.1, 100000}
derived.field = 47

dynamics.tolerance = {1e-6, 0.2}
dynamics.popvelocity = 10
dynamics.surfacelayer = 1e-2
dynamics.stepsize = 0.01
dynamics.collision = {}

dynamics.collision.ground = function (a, b)
   if a.isground or b.isground then
      local other = a.isground and b or a

      if other.isairplane then
	 other.dynamics.controls = nil
	 other.engine.throttle = 0
	 other.dust.particles.rate = 1

	 if not graph.fadeout then
	    graph.fadeout = director.script {
	       [0.1] = function (self)
			  graph.transition = transitions.fade {
			     duration = 5
			  }
		       end,
	       [5] = function (self)
			graph.sky = nil
		     end,
	       
	       [7] = function (self) 
			common.iterate = false
		     end
	    }
	 end

	 return 5, 0.6, 0
      else
	 return 1, 0.6, 0
      end	 
   end

   return 0, 0, 0
end

graphics.close = function ()
		    common.iterate = false
		 end

-- Define some basic global resources.

resources.small = widgets.face "Sans-10"
resources.regular = widgets.face "Sans-12"
resources.large = widgets.face "Sans-14"
resources.huge = widgets.face "Sans-48"

resources.instruments = function (values)
			   values.color = {1, 0, 0}
			   return resources.faces.small (values)
			end

resources.altimeter = function (values)
			 values.color = {1, 0, 0}
			 return resources.faces.large (values)
		      end

-- Load all the rest.

resources.dofile "common/atmosphere.lua"
resources.dofile "common/kolmogorov.lua"

-- Set up the scene.

graph.sky = nature.atmosphere {
   size = {512, 128},

   sun = {0, math.pi / 8},
   turbidity = 3.5,

   rayleigh = math.scale({6.95e-06, 1.18e-05, 2.44e-05}, 1),
   mie = 7e-5,

--       frames.timer {
--          period = 0.2,

--          tick = function (self, tick, delta, elapsed)
-- 		local n = 2 + tick * 0.1

--     	        print (n)
-- 		self.parent.rayleigh = math.scale({6.95e-06, 1.18e-05, 2.44e-05}, n)
-- 		self.parent.mie = math.scale({4e-7, 6e-7, 2.4e-6}, n)
--    		self.parent.sun = {math.pi, math.pi / n}
--  		self.parent.turbidity = n
--  		self.parent.eccentricity = self.parent.eccentricity + 0.05
-- 		print (self.parent.eccentricity)
--    	     end
--       },

   sunlight = shading.light {
      position = math.transform(
	 transforms.euler (
	    math.deg(math.pi / 4),
	    0,
	    180 - math.deg(0)
	 ),
	 {0, 0, 1e2}
      ),

      offset = {1, 2},
      volume = {-1000, 1000, -1000, 1000, 0.1, 1000},
      intensity = math.scale({1, 1, 0.8}, 1),
      size = {0, 0},
   },

   skylight = shading.ambient {
      orientation = transforms.euler (0, 180, 0),
      intensity = resources.clamped "aviation/imagery/horizon.lc",
   },

   system = bodies.system {
      plane = resources.dofile "aviation/edge.lua",

      land = (options.simple and nature.earth or nature.land) {
	 albedo = 0.45,
	 
	 elevation = resources.elevation "aviation/columbia/columbia.lua" {
	    isground = true,

	    position = units.meters{-4.5 * 256 * 93, -5.5 * 256 * 93, 0},
	    target = 15000,
	 
	    link = function(self)
	       local t_0, d = os.clock(), 0

	       print ("\nLoading imagery and elevation data.  " ..
		      "This may take a while.\n")

	       for i = 0 + d, 10 - d do
		  for j = 0 + d, 8 - d do
		     local suffix = string.format("%d.%d.lc", j, i)
		     local samples, bounds =
			resources.dofile ("aviation/columbia/elevation_" .. suffix)
		     local imagery = not options.noimagery and
			resources.dofile ("aviation/columbia/imagery_" .. suffix)

		     self[i * 9 + j] = {samples, bounds, imagery}
		  end
	       end

	       print (string.format("\nImagery and elevation data loaded " ..
				    "in %.1f seconds.\n", os.clock() - t_0))

	       -- No need to load more than once.

	       self.link = nil
	    end
	 }
      }
   }
}
   
graph.events = frames.event {
   keypress = function(self, key)
		 print ("'" .. key .. "' pressed")

		 if key == 'q' then
		    common.iterate = false
		 elseif key == 'p' then
		    dynamics.timescale = dynamics.timescale > 0 and 0 or 1
		 end
	      end
}

-- Load per-user configuration.

local rcscript = loadfile (os.getenv ("HOME") .. "/.aviation")

if rcscript then
   print "\nReading local user configuration."
   
   print ("  " .. os.getenv ("HOME") .. "/.aviation")
   rcscript ()
end

graph.framerate = options.framerate and widgets.display {
   size = {graphics.window[1] / graphics.window[2], 1},
   align = {1, 1},

   gauge = widgets.clock {
      padding = {0.01, 0.01},
      color = {1.0, 0.8, 0.2},
      opacity = 0.6,
      
      radius = {0.07, 0.05},
      range = {0, 220},
      spacing = {5, 4},
      spread = {-4 * math.pi / 5, 3 * math.pi / 5},

      frames.timer {
	 period = 0.2,
	 frames = 0,

	 tick = function (self, tick, delta, elapsed)
		   self.parent.reading = (graphics.frames - self.frames) / delta
		   self.frames = graphics.frames
		end
      },

      [1] = resources.small {text = "0"},
      [2] = resources.small {text = "20"},
      [3] = resources.small {text = "40"},
      [4] = resources.small {text = "60"},
      [5] = resources.small {text = "80"},
      [7] = resources.small {text = "120"},
      [9] = resources.small {text = "160"},
      [11] = resources.small {text = "200"},
   },
}
