-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "units"

local penalty = 0
local alpha = time.physics + 3

airrace.crashed.restart = function ()
   airrace.nodes.fadeout.parent = graph
   airrace.nodes.fadeout.blur = transitions.blur {
      progress = 0.75,
   }

   table.insert(airrace.notes, string.format("Crashed the plane at %.2fs.",
					     time.physics - alpha))
end

-- What to do when we've been through a gate

airrace.crossedgate.checksequence = function (gate)
   local index, next

   -- Just flew through a gate.  Check if
   -- this is the next in line and add it.

   for i, other in ipairs(airrace.gates) do
      if gate == other then
	 index = i
      end
   end

   next = airrace.sequence[#airrace.sections + 1]

   if index == next then
      airrace.nodes.logger.parent = nil
      table.insert(airrace.sections, {time.physics - alpha,
				      airrace.nodes.logger.velocity,
				      airrace.nodes.logger.acceleration,
				      airrace.nodes.logger.attack})
      airrace.nodes.logger.parent = aviation.nodes.plane     

      table.insert(airrace.notes,
		   string.format("Crossed gate %d at %.2fs.",
				 index, time.physics - alpha))
   else
      airrace.nodes.fadeout.parent = graph

      table.insert(airrace.notes,
		   string.format("Crossed gate %d at %.2fs, " ..
				 "next in sequence was %d.",
			         index, time.physics - alpha, next))
   end

   -- If we've passed through all gates...

   if #airrace.sequence == #airrace.sections then
      local trackset, track, delta, entry

      delta = time.physics - alpha

      table.insert(airrace.notes,
		   string.format("Course completed at %.2fs with %d " ..
				 "seconds of penalty time.", delta, penalty))

      -- Check wether we've set a new record.

      trackset, track = string.match (airrace.track, "(.*)/track(%d*).lua")
      trackset = airrace.tracksets[trackset]
      entry = trackset and trackset[tonumber(track)]

      if entry and (not entry.best or entry.best > delta + penalty) then
      	 local file

      	 table.insert(airrace.notes, "New track record set.")
      	 entry.best = delta + penalty

	 -- Update the career records.

	 file = io.open (os.getenv ("HOME") .. "/.airrace.d/career", "w")
	 
	 for setname, trackset in pairs (airrace.tracksets) do
	    for i, track in ipairs (trackset) do
	       if track.best then
	 	  file:write (string.format('airrace.tracksets["%s"][%d].best = %.2f\n', setname, i, track.best))
	       end
	    end
	 end

	 file:close()
      end

      airrace.nodes.fadeout.parent = graph
   end
end

-- What to do when we've been through a pylon

airrace.crossedpylon.penalize = function (pylon)
   local index

   penalty = penalty + 5

   for i, other in ipairs(airrace.gates) do
      if pylon.ancestry[2] == other then
	 index = i
      end
   end

   table.insert(airrace.notes,
		string.format("Went through a pylon at " ..
			      "gate %d at %.2fs.",
			      index, time.physics - alpha))
end

aviation.nodes.plane.cockpit.switchgear.button.press = function (self)
   if self.pressed then
      table.insert(airrace.notes, "Pilot signaled a resignation.")
   end

   airrace.nodes.fadeout.parent = graph
end

airrace.nodes.fadeout = director.script {
   link = function(self)
	     -- Disable any further rules checking.

	     airrace.crossedgate.checksequence = nil
	     aviation.crashed.restart = nil
	     airrace.crossedpylon.penalize = nil
	     aviation.nodes.plane.cockpit.switchgear.button.press = nil
	  end,

   [0.5] = function(self)
	    self.transition = transitions.fade {
	       duration = 2.5
	    }
	 end,

   [3] = function(self)
	    if options.track then
	       -- Single track selected, quit.

	       common.iterate = false
	       network.block = false
	       graphics.hide = true
	    else
	       -- Clip the scene tree.

	       resources.dofile "airrace/stop.lua"
	       resources.dofile "aviation/stop.lua"

	       -- Suspend and wait for an HTTP request.

	       graphics.hide = true
	       network.block = true
	    end

	    self.parent = nil
	 end
}

airrace.nodes.logger = frames.timer {
   period = 0.1,

   link = function (self)
	     self.velocity = {1 / 0, -1 / 0, 0}
	     self.attack = {1 / 0, -1 / 0}
	     self.acceleration = {1 / 0, -1 / 0}
	  end,

   tick = function (self, tick, delta, elapsed)
	     local uvw, alpha, V, F, gee

	     -- Calculate speed, angle of attack and acceleration.

	     uvw = transforms.tonode(self.parent, self.parent.velocity)
	     alpha = -math.deg (math.atan2 (uvw[3], uvw[1]))
	     V = uvw[1] / units.milesperhour(1)
	     F = transforms.tonode(self.parent.belt.torso,
				   self.parent.belt.forces[2])

	     gee = -F[3] / self.parent.belt.torso.mass[1] / derived.gee
	     
	     if V < self.velocity[1] then
		self.velocity[1] = V
	     end

	     if V > self.velocity[2] then
		self.velocity[2] = V
	     end

	     self.velocity[3] = self.velocity[3] + V
	     	     
	     if alpha < self.attack[1] then
		self.attack[1] = alpha
	     end

	     if alpha > self.attack[2] then
		self.attack[2] = alpha
	     end

	     if gee < self.acceleration[1] then
		self.acceleration[1] = gee
	     end

	     if gee > self.acceleration[2] then
		self.acceleration[2] = gee
	     end	     
	  end,

   unlink = function (self)
	       self.velocity[3] = self.velocity[3] / self.state[1]
	    end
}

airrace.nodes.layout = widgets.layout {
   opacity = 0.75,
   scale = 0.45,
   text = [[<span color="black" font="Sans 32">Aviation</span>]],

   timer = frames.timer {
      period = 0.1,

      tick = function (self, tick, delta, elapsed)
		self.parent.text = string.format ([[
<span color="black" font="Sans 20">time:	%.2fs+%ds</span>
<span color="black" font="Sans 20">section:	%d</span>]],
	                           time.physics - alpha, penalty, #airrace.sections + 1)
		   
		if #airrace.sequence == 0 then
		   self.parent = nil
		end
	     end
   }
}

airrace.nodes.display = widgets.display {
   padding = {0.02, 0.02},
   align = {1, -1},

   frame = widgets.frame {
      color = {1.0, 0.8, 0.2},
      opacity = 0.6,

      clock = frames.timer {
	 period = 0.1,

	 tick = function (self, tick, delta, elapsed)
		   self.parent.layout.text = string.format ([[
<span color="beige" font="Sans 14">time:</span>\t\t\t<span color="floralwhite" font="Sans 18">%.2fs+%ds</span>
<span color="beige" font="Sans 14">section:</span>\t<span color="floralwhite" font="Sans 18">%d</span>]], time.physics - alpha, penalty, #airrace.sections + 1)
		   
		   if #airrace.sequence == 0 then
		      self.parent = nil
		   end
		end
      },

      layout = widgets.layout {
	 padding = {0.02, 0.01},
      },
   },
}

airrace.sections = {}
airrace.notes = {}

airrace.nodes.logger.parent = aviation.nodes.plane
airrace.nodes.layout.parent = not options.simple and aviation.nodes.screen
airrace.nodes.display.parent = options.simple and graph
