//  Copyright (C) 2010, 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <config.h>

#include <fstream>
#include <gtkmm.h>
#include <time.h>
#include <glibmm.h>
#include <giomm.h>
#include "main.h"
#include "tapioca-window.h"
#include "preferences-dialog.h"
#include "gallery-dialog.h"
#include "document-dialog.h"
#include "connect-dialog.h"
#include "changed-gallery.h"
#include "changed-document.h"
#include "progress-dialog.h"
#include "changes-editor-dialog.h"
#include "publish-dialog.h"
#include "ucompose.hpp"
#include "quit-dialog.h"
#include "gallery.h"

TapiocaWindow* TapiocaWindow::create(std::string filename)
{
  TapiocaWindow *window;
  Glib::RefPtr<Gtk::Builder> xml = 
    Gtk::Builder::create_from_file(Main::get_glade_path() + "/tapioca.gtk");

  xml->get_widget_derived("window", window);
  window->on_window_loaded(filename);
  window->signal_delete_event().connect(
	sigc::mem_fun(*window, &TapiocaWindow::on_delete_event));
  return window;
}

TapiocaWindow::TapiocaWindow(BaseObjectType* baseObject, 
                             const Glib::RefPtr<Gtk::Builder>& xml)
 : Gtk::Window(baseObject)
{
  needs_saving = false;

  xml->get_widget("contents", contents);

  set_icon_from_file(Main::get_data_path() + "/icon.png");
  std::list<Gtk::TargetEntry> targets;
  targets.push_back(Gtk::TargetEntry("TapiocaDocument", Gtk::TARGET_SAME_APP));
  targets.push_back(Gtk::TargetEntry("GTK_TREE_MODEL_ROW", Gtk::TARGET_SAME_WIDGET));

  xml->get_widget("treeview", treeview);

  galleries_list = Gtk::ListStore::create(galleries_columns);
  treeview->set_model(galleries_list);
  treeview->get_selection()->set_mode(Gtk::SELECTION_SINGLE);
  treeview->append_column("Galleries", galleries_columns.name);
  session = NULL;
  treeview->get_selection()->signal_changed().connect
    (sigc::mem_fun(*this, &TapiocaWindow::on_gallery_selected));
  treeview->signal_row_activated().connect
    (sigc::mem_fun(*this, &TapiocaWindow::on_gallery_activated));
  treeview->signal_button_press_event().connect_notify
    (sigc::mem_fun(*this, &TapiocaWindow::on_treeview_clicked));
  treeview->enable_model_drag_dest(targets, Gdk::ACTION_MOVE);
  treeview->signal_drag_data_received().connect
    (sigc::mem_fun
     (*this, &TapiocaWindow::on_galleries_drop_drag_data_received));

  Gtk::Menu::MenuList& menulist = treeview_context_menu.items();
  menulist.push_back
    (Gtk::Menu_Helpers::MenuElem
     ("_Properties", 
      sigc::mem_fun(*this, &TapiocaWindow::on_gallery_properties_clicked)));
  menulist.push_back
    (Gtk::Menu_Helpers::MenuElem
     ("_Add", sigc::mem_fun(*this, &TapiocaWindow::on_add_gallery_clicked)));
  menulist.push_back
    (Gtk::Menu_Helpers::MenuElem
     ("_Remove", sigc::mem_fun(*this, 
                               &TapiocaWindow::on_remove_gallery_clicked)));
  treeview_context_menu.accelerate(*this);


  xml->get_widget("iconview", iconview);
  documents_list = Gtk::ListStore::create (documents_columns);
  iconview->set_model(documents_list);
  iconview->set_pixbuf_column(documents_columns.image);
  iconview->set_text_column(documents_columns.name);
  iconview->signal_selection_changed().connect
    (sigc::mem_fun(*this, &TapiocaWindow::on_document_selected));

  iconview->signal_item_activated().connect(sigc::mem_fun(*this, &TapiocaWindow::on_document_activated));
  iconview->signal_selection_changed().connect(sigc::mem_fun(*this, &TapiocaWindow::on_document_selected));
  iconview->signal_button_press_event().connect_notify
    (sigc::mem_fun(*this, &TapiocaWindow::on_iconview_clicked));
  iconview->signal_drag_end().connect_notify
    (sigc::mem_fun(*this, &TapiocaWindow::on_iconview_reordered));
  iconview->enable_model_drag_source(targets, Gdk::MODIFIER_MASK, 
                                     Gdk::ACTION_MOVE);
  iconview->enable_model_drag_source(targets);
  iconview->enable_model_drag_dest(targets, Gdk::ACTION_MOVE);
  iconview->signal_drag_data_get().connect
    (sigc::mem_fun(*this, &TapiocaWindow::on_document_drag_data_get));
  iconview->signal_drag_data_received().connect
    (sigc::mem_fun
     (*this, &TapiocaWindow::on_documents_drop_drag_data_received));


  xml->get_widget("gallery_label", gallery_label);
  menulist = iconview_context_menu.items();
  menulist.push_back
    (Gtk::Menu_Helpers::MenuElem
     ("_Properties", 
      sigc::mem_fun(*this, &TapiocaWindow::on_image_properties_clicked)));
  menulist.push_back
    (Gtk::Menu_Helpers::MenuElem
     ("_View", sigc::mem_fun(*this, 
                             &TapiocaWindow::on_view_document_clicked)));
  menulist.push_back
    (Gtk::Menu_Helpers::MenuElem
     ("_Add", sigc::mem_fun(*this, &TapiocaWindow::on_add_document_clicked)));
  menulist.push_back
    (Gtk::Menu_Helpers::MenuElem
     ("_Remove", sigc::mem_fun(*this, 
                               &TapiocaWindow::on_remove_document_clicked)));
  treeview_context_menu.accelerate(*this);


  //menubar callbacks
  xml->get_widget("connect_menuitem", connect_menuitem);
  connect_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_connect_clicked));
  xml->get_widget("open_offline_menuitem", open_offline_menuitem);
  open_offline_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_open_offline_changes_clicked));
  xml->get_widget("save_offline_menuitem", save_offline_menuitem);
  save_offline_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_save_offline_changes_clicked));
  xml->get_widget("save_offline_as_menuitem", save_offline_as_menuitem);
  save_offline_as_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_save_offline_changes_as_clicked));
  xml->get_widget("save_changes_menuitem", save_changes_menuitem);
  save_changes_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_save_changes_to_atpic_clicked));
  xml->get_widget("export_menuitem", export_menuitem);
  export_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_export_clicked));
  xml->get_widget("exit_menuitem", exit_menuitem);
  exit_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_exit_clicked));
  xml->get_widget("preferences_menuitem", preferences_menuitem);
  preferences_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_preferences_clicked));
  xml->get_widget("image_properties_menuitem", image_properties_menuitem);
  image_properties_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_image_properties_clicked));
  image_properties_menuitem->set_sensitive(false);
  xml->get_widget("gallery_properties_menuitem", gallery_properties_menuitem);
  gallery_properties_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_gallery_properties_clicked));
  gallery_properties_menuitem->set_sensitive(false);
  xml->get_widget("about_menuitem", about_menuitem);
  about_menuitem->signal_activate().connect(sigc::mem_fun(*this,&TapiocaWindow::on_about_clicked));
}

void TapiocaWindow::on_window_loaded(std::string filename)
{
  fill_in_galleries();
  fill_in_documents();
  if (filename.empty() == false)
    {
      current_save_filename = filename;
      load_session();
    }
  update_menuitems();

  temporary_directory = "/tmp/" PACKAGE_NAME ".XXXXXXX";
  int fd = Glib::mkstemp(temporary_directory);
  close(fd);
  Document::remove_file(temporary_directory);
  Document::create_dir(temporary_directory);
  progress_aborted = false;
}

TapiocaWindow::~TapiocaWindow()
{
  if (session)
    delete session;
  Document::clean_dir(temporary_directory);
}

void TapiocaWindow::on_connect_clicked()
{
  if (needs_saving)
    {
      Gtk::MessageDialog e("Re-connecting to AtPic destroys local changes.", 
                           Gtk::MESSAGE_ERROR);
      e.set_modal();
      e.set_title("Warning!");
      e.set_icon_from_file(Main::get_data_path() + "/icon.png");
      e.run();
    }

  Profile *profile = new Profile();
  if (session)
    {
      profile->set_username(session->get_profile()->get_username());
      profile->set_password(session->get_profile()->get_password());
    }
  ConnectDialog *d = ConnectDialog::create(profile);
  d->set_transient_for(*this);
  int response = d->run();
  d->hide();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      profile->set_url(Main::get_homepage());

      Session *new_session = Session::create(profile);
      new_session->login(sigc::bind(sigc::mem_fun(*this, &TapiocaWindow::on_connect_attempted), new_session));
      //session gets set later on, after all of the callbacks.
    }
  else
    delete profile;
  delete d;
}

        
void TapiocaWindow::on_pull_galleries_failed(Session *s, ProgressDialog &d)
{
  if (progress_aborted)
    return;
  d.hide();
  Gtk::MessageDialog e("Couldn't download documents from AtPic.  Try again.", 
                       Gtk::MESSAGE_ERROR);
  e.set_modal();
  e.set_title("Whoops!");
  e.set_icon_from_file(Main::get_data_path() + "/icon.png");
  e.run();
}

void TapiocaWindow::pull_galleries_in_the_background(Session *s, ProgressDialog &d)
{
      s->pulled_galleries.connect
        (sigc::bind<Session*, ProgressDialog&>(sigc::mem_fun(this, 
                                  &TapiocaWindow::on_pulled_gallery_listing),
                    s, d));
      s->pull_galleries_failed.connect
        (sigc::bind<ProgressDialog &>(sigc::mem_fun(*this, &TapiocaWindow::on_pull_galleries_failed), d));
  s->pull_galleries();
}

void TapiocaWindow::on_connect_attempted(bool login_successful, 
                                         Session *new_session)
{
  if (!login_successful)
    {
      delete new_session;
      on_connect_clicked();
    }
  else
    {
      ProgressDialog d;
      d.set_parent_window(*this);
      d.aborted.connect
        (sigc::mem_fun(*this, &TapiocaWindow::on_progress_aborted));
      Glib::signal_idle().connect_once(sigc::bind<Session*, ProgressDialog&>(sigc::mem_fun(*this, &TapiocaWindow::pull_galleries_in_the_background), new_session, d));
      d.run();
    }
}

void TapiocaWindow::on_galleries_pulled(Tapioca::Session *new_session)
{
  new_session->copy_galleries_from_profile();

  if (session)
    {
      //wait a minute.  if the usernames differ, then we're SOL.
      if (new_session->get_profile()->get_id() ==
          session->get_profile()->get_id())
        new_session->merge_changes(session);
      delete session;
    }
  session = new_session;
  needs_saving = session->is_changed();

  fill_in_galleries();
  fill_in_documents();
  if (session->size() > 0)
    treeview->set_cursor(Gtk::TreePath("0"));
}

void TapiocaWindow::on_open_offline_changes_clicked()
{
    Gtk::FileChooserDialog chooser(*this, "Choose File to Load");
    Gtk::FileFilter sav_filter;
    sav_filter.add_pattern("*" + std::string(TAPIOCA_EXT));
    chooser.set_filter(sav_filter);
    chooser.set_current_folder(Glib::get_home_dir());

    chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
    chooser.set_default_response(Gtk::RESPONSE_ACCEPT);
	
    chooser.show_all();
    int res = chooser.run();
    
    if (res == Gtk::RESPONSE_ACCEPT)
      {
        std::string old_save_filename = current_save_filename;
	current_save_filename = chooser.get_filename();
	chooser.hide();
        load_session();
      }
    update_menuitems();
}

void TapiocaWindow::load_session_in_the_background(ProgressDialog &d)
{
  d.progress(1.0/6.0);
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
  try
    {
      std::ifstream ifs(current_save_filename.c_str());
      d.progress(2.0/6.0);
      while (g_main_context_iteration(NULL, FALSE)); //doEvents
      boost::archive::text_iarchive ia(ifs);
      d.progress(3.0/6.0);
      Gallery::loading.connect(sigc::mem_fun(d, &ProgressDialog::pulse));
      ia >> session;
    }
  catch (std::exception &ex)
    {
      d.hide();
      return;
    }

  d.progress(4.0/6.0);
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
  session->update_thumbnails();
  d.progress(5.0/6.0);
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
  session->get_profile()->update_thumbnails();
  d.progress(6.0/6.0);
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
  d.hide();
  return;
}
      
void TapiocaWindow::load_session()
{
  if (session)
    delete session;
  ProgressDialog d;
  d.set_parent_window(*this);
  session = new Session();
  Glib::signal_idle().connect_once(sigc::bind<ProgressDialog&>(sigc::mem_fun(*this, &TapiocaWindow::load_session_in_the_background), d));

  d.run();
  if (session->size() == 0)
    {
      delete session;
      session = NULL;
      return;
    }
  remove_deleted_documents();
  fill_in_galleries();
  fill_in_documents();
  if (session->size() > 0)
    treeview->set_cursor(Gtk::TreePath("0"));
  needs_saving = false; 
}

void TapiocaWindow::remove_deleted_documents()
{
  std::list<Document*> removed_docs;
  session->find_deleted_documents(removed_docs);
  std::string removed = "The following documents were not found:\n";
  size_t count = 0;
  size_t max_listed = 10;
  for (std::list<Document*>::iterator it = removed_docs.begin();
       it != removed_docs.end(); it++)
    {
      Gallery *gallery = session->find_by_id((*it)->get_gallery_id());
      if (count < max_listed)
        removed += (*it)->get_image_filename() + "\n";
      count++;
      gallery->remove_document(*it);
    }
  needs_saving = true; 
  if (removed_docs.size() > max_listed)
    removed += 
      String::ucompose("and %1 more", removed_docs.size() - max_listed);

  if (removed_docs.size() > 0)
    {
      Gtk::MessageDialog d(removed, Gtk::MESSAGE_ERROR);
      d.set_modal();
      d.set_title("Some documents were not loaded");
      d.set_icon_from_file(Main::get_data_path() + "/icon.png");
      d.run();
    }
  update_window_title();
}

void TapiocaWindow::save_session()
{
  ProgressDialog d;
  d.set_parent_window(*this);
  Glib::signal_idle().connect_once(sigc::bind<ProgressDialog&>(sigc::mem_fun(*this, &TapiocaWindow::save_session_in_the_background), d));

  d.run();
  save_session_in_the_background(d);
  needs_saving = false;
  update_window_title();
}

void TapiocaWindow::save_session_in_the_background(ProgressDialog &d)
{
  std::ofstream ofs(current_save_filename.c_str());
  boost::archive::text_oarchive oa(ofs);
  Gallery::saving.connect(sigc::mem_fun(d, &ProgressDialog::pulse));
  oa << session;
  d.hide();
}

void TapiocaWindow::on_save_offline_changes_clicked()
{
  if (current_save_filename.empty() == true)
    on_save_offline_changes_as_clicked();
  else
    save_session();
}

std::string TapiocaWindow::add_ext_if_necessary(std::string file, std::string ext) const
{
  if (Document::name_ends_with(file, ext) == true)
    return file;
  else
    return file + ext;
}

void TapiocaWindow::on_save_offline_changes_as_clicked()
{
    Gtk::FileChooserDialog chooser(*this, "Choose a File to Save To",
				   Gtk::FILE_CHOOSER_ACTION_SAVE);
    Gtk::FileFilter sav_filter;
    sav_filter.add_pattern("*" + std::string(TAPIOCA_EXT));
    chooser.set_filter(sav_filter);
    chooser.set_current_folder(Glib::get_home_dir());

    chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
    chooser.add_button(Gtk::Stock::SAVE, Gtk::RESPONSE_ACCEPT);
    chooser.set_default_response(Gtk::RESPONSE_ACCEPT);
    
    chooser.show_all();
    int res = chooser.run();
    
    if (res == Gtk::RESPONSE_ACCEPT)
    {
        std::string old_save_filename = current_save_filename;
	current_save_filename = chooser.get_filename();
	chooser.hide();
        current_save_filename = 
          add_ext_if_necessary(current_save_filename, TAPIOCA_EXT);

        save_session();
    }
}

void TapiocaWindow::on_save_changes_to_atpic_clicked()
{
  ChangesEditorDialog *d = ChangesEditorDialog::create(session);
  d->set_transient_for(*this);
  int response = d->run();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      session->revert(d->get_reverted_changes());
      session->apply(d->get_changes());
      if (d->get_reverted_changes().empty() == false ||
          d->get_changes().empty() == false)
        {
          fill_in_galleries();
          fill_in_documents();
          update_menuitems();
        }
      d->hide();
      SessionChanges changes = session->get_changes();
      if (changes.empty() == false)
        {
          PublishDialog *di = PublishDialog::create(changes);
          di->set_transient_for(*this);
          int resp = di->run();
          di->hide();
          if (resp == Gtk::RESPONSE_ACCEPT)
            {
              publish(changes);
              //publish these changes now.
            }
          delete di;
        }
    }
  delete d;
}

void TapiocaWindow::on_published_gallery_removed(Gallery *gallery, sigc::slot<void> slot)
{
  publish_changes.removed_galleries.remove
    (publish_changes.removed_galleries.front());
  std::list<ChangedDocument> shadowed_docs;
  for (std::list<ChangedDocument>::iterator i = 
       publish_changes.removed_documents.begin(); 
       i != publish_changes.removed_documents.end(); i++)
    {
      if ((*i).first.get_gallery_id() == gallery->get_id())
        shadowed_docs.push_back(*i);
    }
  for (std::list<ChangedDocument>::iterator i = shadowed_docs.begin();
       i != shadowed_docs.end(); i++)
    publish_changes.removed_documents.remove(*i);
  session->get_profile()->remove(gallery);
  delete gallery;
  if (publish_changes.removed_galleries.size() > 0)
    remove_published_galleries(slot);
  else
    slot();
}

void TapiocaWindow::on_published_gallery_added(Gallery *gallery, sigc::slot<void> slot)
{
  publish_changes.added_galleries.remove
    (publish_changes.added_galleries.front());
  if (publish_changes.added_galleries.size() > 0)
    add_published_galleries(slot);
  else
    slot();
}

void TapiocaWindow::on_tags_updated(Item *item)
{
}

void TapiocaWindow::export_documents_in_the_background(ProgressDialog &d, std::string export_dir, std::list<Document*> docs)
{


  Document *document;
  while (1)
    {
      if (docs.size() == 0)
        {
          d.hide();
          return;
        }
      document = docs.front();
      docs.remove(document);
      if (Document::exists(export_dir +"/" + document->get_gallery_id() + "/" + document->get_export_filename()))
        continue;
      break;
    }

  session->download_document(document, sigc::bind<std::string, std::list<Document*>, ProgressDialog&>(sigc::mem_fun(*this, &TapiocaWindow::export_downloaded_document), export_dir, docs, d));

}

void TapiocaWindow::on_progress_aborted()
{
  progress_aborted = true;
}

void TapiocaWindow::on_export_clicked()
{
  Gtk::FileChooserDialog chooser(*this, "Choose Folder to Export All Documents to", Gtk::FILE_CHOOSER_ACTION_SELECT_FOLDER);
  chooser.set_current_folder(Glib::get_home_dir());
  chooser.set_create_folders(true);
  chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
  chooser.set_default_response(Gtk::RESPONSE_ACCEPT);
	
  chooser.show_all();
  int response = chooser.run();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      std::string export_dir = chooser.get_current_folder();
      chooser.hide();
      bool have_credentials = true;
      if (session->get_profile()->get_username() == "" ||
          session->get_profile()->get_password() == "")
        have_credentials = false;

      std::list<Document*> documents;
      for (Profile::const_iterator i = 
           session->get_profile()->get_galleries().begin(); 
           i != session->get_profile()->get_galleries().end(); i++)
        {
          Gallery* g = *i;
          for (Gallery::const_iterator j = g->get_docs().begin(); 
               j != g->get_docs().end(); j++)
            documents.push_back(*j);
        }

      ProgressDialog d;
      d.set_parent_window(*this);
      d.push_back(session->download_document_failed.connect
                  (sigc::bind<ProgressDialog&>(sigc::mem_fun(*this, &TapiocaWindow::on_export_document_download_failed), d)));
      d.aborted.connect
        (sigc::mem_fun(*this, &TapiocaWindow::on_progress_aborted));
      on_export_documents_attempted
        (have_credentials, d, export_dir, documents,
         sigc::bind<std::string, std::list<Document*>,ProgressDialog& >(sigc::mem_fun(this, &TapiocaWindow::export_downloaded_document), export_dir, documents,d));

    }
  else
    chooser.hide();
}

bool TapiocaWindow::quit()
{
  if (needs_saving)
    {
      QuitDialog *d = QuitDialog::create();
      d->set_transient_for(*this);
      int response = d->run();
      d->hide();
      if (response == Gtk::RESPONSE_CANCEL) //we don't want to quit
	return false;
      else if (response == Gtk::RESPONSE_ACCEPT) // save and quit
        on_save_offline_changes_clicked();
      exit(0); //makes us exit quicker
    }
  else
    exit(0); //makes us exit quicker
}
    
bool TapiocaWindow::on_delete_event(GdkEventAny *event)
{
  return quit();
}

void TapiocaWindow::on_exit_clicked()
{
  quit();
}

void TapiocaWindow::on_preferences_clicked()
{
  PreferencesDialog *d = PreferencesDialog::create();
  d->set_transient_for(*this);
  d->run();
  d->hide();
  delete d;
}

void TapiocaWindow::on_image_properties_clicked()
{
  Document *document = new Document(*get_selected_document());
  DocumentDialog *d = DocumentDialog::create
    (document, session->get_profile()->find_doc_by_id(document->get_id()));
  d->set_transient_for(*this);
  int response = d->run();
  d->hide();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      Document *orig_document = get_selected_document();
      if (*orig_document == *document)
        ;
      else
        {
          needs_saving = true;
          *orig_document = *document;
          update_window_title();
        }
    }
  delete d;
  update_selected_document_name();
}

void TapiocaWindow::on_gallery_properties_clicked()
{
  Gallery *gallery = new Gallery(*get_selected_gallery());
  GalleryDialog *d = GalleryDialog::create
    (gallery, session->get_profile()->find_by_id(gallery->get_id()));

  d->set_transient_for(*this);
  int response = d->run();
  d->hide();
  if (response == Gtk::RESPONSE_ACCEPT)
    {
      Gallery *orig_gallery = get_selected_gallery();
      if (*orig_gallery == *gallery)
        ;
      else
        {
          needs_saving = true;
          //what's this all about?  ah we're replacing things.
          orig_gallery->remove_all_documents();
          *orig_gallery = *gallery;
          fill_in_documents();
          update_window_title();
        }
    }
  delete d;
  update_selected_gallery_name();
}

void TapiocaWindow::update_selected_document_name()
{
  typedef std::list<Gtk::TreeModel::Path> type_list_paths;
  type_list_paths selected = iconview->get_selected_items();
  if (!selected.empty())
    {
      const Gtk::TreeModel::Path &path = *selected.begin();
      Gtk::TreeModel::iterator iter = documents_list->get_iter(path);
      Gtk::TreeModel::Row row = *iter;
      Document *document = row[documents_columns.data];
      row[documents_columns.name] = document->get_title();
    }
}

void TapiocaWindow::update_selected_gallery_name()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Gallery *gallery = row[galleries_columns.data];
      row[galleries_columns.name] = gallery->get_title();
      gallery_label->set_text(gallery->get_title());
    }
}

void TapiocaWindow::on_about_clicked()
{
  Gtk::AboutDialog* dialog;

  Glib::RefPtr<Gtk::Builder> xml
    = Gtk::Builder::create_from_file(Main::get_glade_path() + "/about-dialog.gtk");

  xml->get_widget("dialog", dialog);
  dialog->set_transient_for(*this);
  dialog->set_icon_from_file(Main::get_data_path() + "/icon.png");

  std::string name = PACKAGE_NAME;
  std::transform(name.begin(), ++name.begin(), name.begin(), toupper);
  dialog->set_program_name(name);
  dialog->set_version(PACKAGE_VERSION);
  dialog->set_logo(Gdk::Pixbuf::create_from_file(Main::get_data_path() + "/icon.png"));
  dialog->show_all();
  dialog->run();
  delete dialog;
  return;
}

Gallery* TapiocaWindow::get_selected_gallery()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Gallery *gallery = row[galleries_columns.data];
      return gallery;
    }
  return NULL;
}

Document* TapiocaWindow::get_selected_document()
{
  typedef std::list<Gtk::TreeModel::Path> type_list_paths;
  type_list_paths selected = iconview->get_selected_items();
  if (!selected.empty())
    {
      const Gtk::TreeModel::Path &path = *selected.begin();
      Gtk::TreeModel::iterator iter = documents_list->get_iter(path);
      Gtk::TreeModel::Row row = *iter;
      return row[documents_columns.data];
    }
  return NULL;
}

void TapiocaWindow::update_menuitems()
{
  Gallery *gallery = get_selected_gallery();
  if (session != NULL)
    {
      save_offline_menuitem->set_sensitive(true);
      save_offline_as_menuitem->set_sensitive(true);
      save_changes_menuitem->set_sensitive(true);
      export_menuitem->set_sensitive(true);
      contents->set_sensitive(true);
      if (gallery)
        gallery_properties_menuitem->set_sensitive(true);
      else
        gallery_properties_menuitem->set_sensitive(false);
    }
  else
    {
      save_offline_menuitem->set_sensitive(false);
      save_offline_as_menuitem->set_sensitive(false);
      save_changes_menuitem->set_sensitive(false);
      gallery_properties_menuitem->set_sensitive(false);
      export_menuitem->set_sensitive(false);
      contents->set_sensitive(false);
    }

  if (get_selected_document() != NULL)
    image_properties_menuitem->set_sensitive(true);
  else
    image_properties_menuitem->set_sensitive(false);
  update_window_title();
}


void TapiocaWindow::on_gallery_selected()
{
  Gallery *gallery = get_selected_gallery();
  fill_in_documents();
  if (gallery != NULL)
    gallery_label->set_text(gallery->get_title());
  if (gallery && gallery->size() > 0)
    iconview->scroll_to_path(Gtk::TreeModel::Path("0"), false, 0.0, 0.0);
  update_menuitems();
}

void TapiocaWindow::on_document_selected()
{
  update_menuitems();
}

void TapiocaWindow::on_gallery_activated(const Gtk::TreeModel::Path& path,
                                      Gtk::TreeViewColumn *col)
{
  update_menuitems();
  on_gallery_properties_clicked();
}

void TapiocaWindow::on_document_activated(const Gtk::TreeModel::Path& path)
{
  update_menuitems();
  on_image_properties_clicked();
}

void TapiocaWindow::add_gallery(Tapioca::Gallery *gallery)
{
  Gtk::TreeIter i = galleries_list->append();
  (*i)[galleries_columns.name] = gallery->get_title();
  (*i)[galleries_columns.data] = gallery;
}

void TapiocaWindow::fill_in_galleries()
{
  galleries_list->clear();
  if (session)
    for (Tapioca::Session::const_iterator it = 
         session->get_galleries().begin(); 
         it != session->get_galleries().end(); it++)
      add_gallery((*it));
}

void TapiocaWindow::fill_in_documents()
{
  Gallery *g = get_selected_gallery();
  if (g)
    {
      documents_list->clear();
      for (Gallery::const_iterator it = g->get_docs().begin(); 
           it != g->get_docs().end(); it++)
        add_document(*it);
    }
  else
    {
      documents_list->clear();
      gallery_label->set_text("");
    }
}

void TapiocaWindow::add_document(Tapioca::Document *document)
{
  if (document)
    {
      Gtk::TreeModel::Row row = *(documents_list->append());
      row[documents_columns.image] = document->get_thumbnail();
      row[documents_columns.name] = document->get_title();
      row[documents_columns.data] = document;
    }
}
      
void TapiocaWindow::on_treeview_clicked(GdkEventButton* event)
{
  if (event->type == GDK_BUTTON_PRESS &&event->button == 3)
    {
      treeview_context_menu.popup(event->button, event->time);
    }
}

void TapiocaWindow::on_iconview_clicked(GdkEventButton* event)
{
  if (event->type == GDK_BUTTON_PRESS &&event->button == 3)
    {
      Gtk::Menu::MenuList& menulist = iconview_context_menu.items();
      Gtk::TreeModel::Path path = iconview->get_path_at_pos(event->x, event->y);
      if (path.empty() == false)
        iconview->select_path(path);
      Document *document = get_selected_document();
      //View and Properties can be turned off.
      Gtk::Menu::MenuList::iterator i = menulist.begin();
      (*i).set_sensitive(document != NULL);
      i++;
      (*i).set_sensitive(document != NULL);
      iconview_context_menu.popup(event->button, event->time);
    }
}

void TapiocaWindow::on_add_gallery_clicked()
{
  Gtk::TreeIter i = galleries_list->append();
  Gallery *gallery = new Gallery();
  gallery->set_title("new gallery");
  gallery->set_id(String::ucompose("%1", time(NULL)));
  session->push_back(gallery);
  (*i)[galleries_columns.name] = gallery->get_title();
  (*i)[galleries_columns.data] = gallery;
  //now select the new gallery
  Glib::ustring p = String::ucompose("%1", session->size()-1);
  treeview->set_cursor(Gtk::TreePath(p));
  needs_saving = true;
  on_gallery_properties_clicked();
}

void TapiocaWindow::on_remove_gallery_clicked()
{
  Glib::RefPtr<Gtk::TreeSelection> selection = treeview->get_selection();
  Gtk::TreeModel::iterator iterrow = selection->get_selected();
  if (iterrow)
    {
      Gtk::TreeModel::Row row = *iterrow;
      Gallery *gallery = row[galleries_columns.data];
      session->remove(gallery);
      delete gallery;
      galleries_list->erase(iterrow);
      needs_saving = true;
      update_menuitems();
    }
}


void TapiocaWindow::on_add_document_clicked()
{
  //ask for a document
  Gtk::FileChooserDialog chooser(*this, "Choose Document to Add");
  Gtk::FileFilter doc_filter;
  doc_filter.add_pixbuf_formats();
  doc_filter.add_pattern("*.avi");
  doc_filter.add_pattern("*.ogv");
  doc_filter.add_pattern("*.mpg");
  doc_filter.add_pattern("*.AVI");
  doc_filter.add_pattern("*.OGV");
  doc_filter.add_pattern("*.MPG");
  chooser.set_filter(doc_filter);
  chooser.set_current_folder(Glib::get_home_dir());

  chooser.add_button(Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL);
  chooser.add_button(Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT);
  chooser.set_default_response(Gtk::RESPONSE_ACCEPT);
	
  chooser.show_all();
  int res = chooser.run();
    
  if (res == Gtk::RESPONSE_ACCEPT)
    {
      std::string filename = chooser.get_filename();
      Document *document = new Document();
      if (Document::is_video_filename(filename) == true)
        document->set_video_filename(filename);
      else
        document->set_image_filename(filename);
      document->set_id(filename);
      document->set_title("new document");
      Gallery *gallery = get_selected_gallery();
      gallery->add_document(document);
      needs_saving = true;
      add_document(document);
    }
}

void TapiocaWindow::on_remove_document_clicked()
{
  typedef std::list<Gtk::TreeModel::Path> type_list_paths;
  type_list_paths selected = iconview->get_selected_items();
  if (!selected.empty())
    {
      const Gtk::TreeModel::Path &path = *selected.begin();
      Gtk::TreeModel::iterator iter = documents_list->get_iter(path);
      Gtk::TreeModel::Row row = *iter;
      Document *document = row[documents_columns.data];
      Gallery *gallery = get_selected_gallery();
      gallery->remove_document(document);
      delete document;
      documents_list->erase(iter);
      needs_saving = true;
      update_menuitems();
    }
}

void TapiocaWindow::on_iconview_reordered(const Glib::RefPtr<Gdk::DragContext>&context)
{
  //sort 'em steve dave.
  //okay only one of the icons is out of order.
      
  Document *prev_document = NULL;
  Document *document = NULL;
  Gallery *gallery = get_selected_gallery();
  for (Gtk::TreeModel::iterator it = documents_list->children().begin();
       it != documents_list->children().end(); it++)
    {
      prev_document = document;
      document = (*it)[documents_columns.data];
      if (prev_document && document)
        {
          if (prev_document->get_order() < document->get_order())
            {
              gallery->set_document_order(document, prev_document);
              needs_saving = true;
              update_window_title();
            }
        }
      else 
        {
          gallery->set_document_order(document, NULL);
          needs_saving = true;
          update_window_title();
        }
    }
}
      
void TapiocaWindow::on_document_drag_data_get(const Glib::RefPtr<Gdk::DragContext> &drag_context,
                                                     
                                           Gtk::SelectionData &data, 
                                           guint info, guint time)

{
  drag_context->get_source_window()->show();
  Document *document = get_selected_document();
  data.set(data.get_target(), 8, (const guchar*)document->get_id().c_str(), strlen(document->get_id().c_str()));

}
       
void TapiocaWindow::on_galleries_drop_drag_data_received(const Glib::RefPtr<Gdk::DragContext> &context, int x, int y, const Gtk::SelectionData& selection_data, guint c, guint time)
{
  const int length = selection_data.get_length();
  if (length >= 0 && selection_data.get_format() == 8)
    {
      std::string id = selection_data.get_data_as_string();
      int nx = 0, ny = 0;
      treeview->convert_widget_to_bin_window_coords(x, y, nx, ny);
      Gtk::TreeModel::Path path;
      treeview->get_path_at_pos(nx, ny, path);
      Gtk::TreeModel::iterator iter = galleries_list->get_iter(path);
      Gtk::TreeModel::Row row = *iter;
      Gallery *dest_gallery = row[galleries_columns.data];
      if (dest_gallery)
        {
          Document *document = session->find_doc_by_id(id);
          Gallery *orig_gallery = 
            session->find_by_id(document->get_gallery_id());
          orig_gallery->remove_document(document);
          fill_in_documents();
          Document *after = NULL;
          if (dest_gallery->size() > 0)
            after = dest_gallery->back();
          document->set_gallery_id(dest_gallery->get_id());
          dest_gallery->set_document_order(document, after);
          needs_saving = true;
          dest_gallery->add_document(document);
          update_menuitems();
        }
    }

  context->drag_finish (false, false, time);

}

void TapiocaWindow::on_documents_drop_drag_data_received(const Glib::RefPtr<Gdk::DragContext> &context, int x, int y, const Gtk::SelectionData& selection_data, guint c, guint time)
{
  bool append = false;
  //Gallery *gallery = get_selected_gallery();
  const int length = selection_data.get_length();
  Gtk::TreeModel::Path dest;
  if (length >= 0 && selection_data.get_format() == 8)
    {
      Gtk::IconViewDropPosition pos;
      int nx = 0, ny = 0;
      iconview->convert_widget_to_bin_window_coords(x, y, nx, ny);
      iconview->get_dest_item_at_pos(nx, ny, pos);
      dest = iconview->get_path_at_pos(nx, ny);
      if (dest.empty())
        {
          dest = documents_list->get_path(--documents_list->children().end());
          pos = Gtk::ICON_VIEW_DROP_RIGHT;
        }

      if (pos == Gtk::ICON_VIEW_DROP_RIGHT || pos == Gtk::ICON_VIEW_DROP_BELOW)
        {
          if (documents_list->get_iter(dest) == documents_list->children().end() ||
              documents_list->get_iter(dest) == --documents_list->children().end())
            append = true;
          else
            dest = documents_list->get_path(++documents_list->get_iter(dest));
        }
      iconview->set_drag_dest_item(dest, pos);
    }
  Gtk::TreeModel::Path src;
  iconview->get_cursor(src);
  if (append)
    documents_list->move(documents_list->get_iter(src), documents_list->children().end());
  else
    documents_list->move(documents_list->get_iter(src), documents_list->get_iter(dest));
  needs_saving = true; 
  update_window_title();
}

void TapiocaWindow::update_window_title()
{
  std::string title = PACKAGE_NAME;
  std::transform(title.begin(), ++title.begin(), title.begin(), toupper);
  if (needs_saving || current_save_filename != "")
    title = " - " + title;
  if (current_save_filename != "")
    title = Glib::path_get_basename(current_save_filename) + title;
  if (needs_saving)
    title = "*" + title;
  set_title(title);
}

void TapiocaWindow::on_view_document_clicked()
{
  Document *document = get_selected_document();
  if (document)
    {
      if (document->is_local() == true)
        view_downloaded_document(document, document->get_filename());
      else
        {
          //short circuit: did we already download it?
          if (document->get_original_filename() != "")
            {
              std::string file = temporary_directory + "/" + document->get_export_filename();
              if (Document::exists(file) == true)
                {
                  view_downloaded_document(document, file);
                  return;
                }
            }
            
          bool have_credentials = true;
          if (session->get_profile()->get_username() == "" ||
              session->get_profile()->get_password() == "")
            have_credentials = false;
          on_download_document_attempted
            (have_credentials, document, 
             sigc::mem_fun(this, &TapiocaWindow::view_downloaded_document));
        }
    }
}

void TapiocaWindow::on_document_downloaded_progress(Document *document, ProgressDialog &d)
{
  if (progress_aborted)
    return;
  d.pulse();
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
}

void TapiocaWindow::on_document_downloaded(Document *document, ProgressDialog &d)
{
  if (progress_aborted)
    return;
  d.hide();
}

void TapiocaWindow::on_document_download_failed(Document *document, ProgressDialog &d, const Session::SlotDownloadDocument slot)
{
  if (progress_aborted)
    return;
  d.hide();
  Gtk::MessageDialog e("Couldn't download document.  Try again.", 
                       Gtk::MESSAGE_ERROR);
  e.set_modal();
  e.set_title("Whoops!");
  e.set_icon_from_file(Main::get_data_path() + "/icon.png");
  e.run();
}

void TapiocaWindow::on_export_document_download_failed(Document *document, ProgressDialog &d)
{
  d.hide();
  Gtk::MessageDialog e("Couldn't download document.  Try again.", 
                       Gtk::MESSAGE_ERROR);
  e.set_modal();
  e.set_title("Whoops!");
  e.set_icon_from_file(Main::get_data_path() + "/icon.png");
  e.run();
}

void TapiocaWindow::view_document_in_the_background(ProgressDialog &d,
                                                    Document *document,
                                                    const Session::SlotDownloadDocument slot)
{
  if (progress_aborted)
    {
      progress_aborted = false;
      return;
    }
  d.push_back(downloaded_document.connect
    (sigc::bind<ProgressDialog&>(sigc::mem_fun(*this, &TapiocaWindow::on_document_downloaded), d)));
  d.push_back(session->download_document_failed.connect
    (sigc::bind<ProgressDialog&, const Session::SlotDownloadDocument>(sigc::mem_fun(*this, &TapiocaWindow::on_document_download_failed), d, slot)));
  d.push_back(session->downloaded_chunk_of_document.connect
    (sigc::bind<ProgressDialog&>(sigc::mem_fun(*this, &TapiocaWindow::on_document_downloaded_progress), d)));
  d.pulse();
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
  session->download_document(document, slot);
  d.pulse();
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
}

void TapiocaWindow::on_download_document_attempted(bool logged_in, 
                                                   Document *document, 
                                                   const Session::SlotDownloadDocument slot)
{
  if (logged_in)
    {
      ProgressDialog d;
      d.set_parent_window(*this);
      d.aborted.connect
        (sigc::mem_fun(*this, &TapiocaWindow::on_progress_aborted));
      Glib::signal_idle().connect_once(sigc::bind<ProgressDialog&, Document*, Session::SlotDownloadDocument>(sigc::mem_fun(*this, &TapiocaWindow::view_document_in_the_background), d, document, slot));
      d.run();
    }
  else
    {
      Profile *profile = new Profile();
      if (session)
        {
          profile->set_username(session->get_profile()->get_username());
          profile->set_password(session->get_profile()->get_password());
        }
      ConnectDialog *d = ConnectDialog::create(profile, false);
      d->set_transient_for(*this);
      int response = d->run();
      d->hide();
      if (response == Gtk::RESPONSE_ACCEPT)
        {
          session->get_profile()->set_username(profile->get_username());
          session->get_profile()->set_password(profile->get_password());
          session->login(sigc::bind(sigc::mem_fun(*this, &TapiocaWindow::on_download_document_attempted), document, slot));
        }
      delete profile;
      delete d;
    }
}

void TapiocaWindow::on_export_documents_attempted(bool logged_in, 
                                                  ProgressDialog &di,
                                                  std::string export_dir,
                                                  std::list<Document*> docs,
                                                 const Session::SlotDownloadDocument slot)
{
  if (logged_in)
    {
      Glib::signal_idle().connect_once(sigc::bind<ProgressDialog&, std::string, std::list<Document*> >(sigc::mem_fun(*this, &TapiocaWindow::export_documents_in_the_background), di, export_dir, docs));
      di.run();
    }
  else
    {
      Profile *profile = new Profile();
      if (session)
        {
          profile->set_username(session->get_profile()->get_username());
          profile->set_password(session->get_profile()->get_password());
        }
      ConnectDialog *d = ConnectDialog::create(profile, false);
      d->set_transient_for(*this);
      int response = d->run();
      d->hide();
      if (response == Gtk::RESPONSE_ACCEPT)
        {
          session->get_profile()->set_username(profile->get_username());
          session->get_profile()->set_password(profile->get_password());
          session->login(sigc::bind<ProgressDialog&, std::string, std::list<Document*>, const Session::SlotDownloadDocument>(sigc::mem_fun(*this, &TapiocaWindow::on_export_documents_attempted), di, export_dir, docs, slot));
        }
      delete profile;
      delete d;
    }
}

void TapiocaWindow::export_downloaded_document(Tapioca::Document *document, std::string file, std::string export_dir, std::list<Document*> docs, ProgressDialog &d)
{
  Glib::RefPtr<Gio::File> f = Gio::File::create_for_path(file);
  Glib::ustring dest = export_dir + "/" + document->get_gallery_id() + "/" + document->get_export_filename();
  Document::create_dir(export_dir + "/" + document->get_gallery_id());
  dest = Document::find_free_file(dest);
  f->move(Gio::File::create_for_path(dest), Gio::FILE_COPY_BACKUP);
  
  if (progress_aborted == true)
    {
      progress_aborted = false;
      return;
    }
  //landed.
  //now we need to kick off the download of the next file.
  size_t numdocs = session->get_profile()->count_documents();
  if (docs.size() == 0)
    {
      session->get_profile()->export_html_pages
        (session->get_profile()->get_username(), export_dir);
      d.hide();
      return;
    }
  d.progress(float(numdocs - docs.size())/(float)numdocs);
  d.set_text("exported "+ document->get_original_filename());
  while (g_main_context_iteration(NULL, FALSE)); //doEvents
  Glib::signal_idle().connect_once(sigc::bind<ProgressDialog&, std::string, std::list<Document*> >(sigc::mem_fun(*this, &TapiocaWindow::export_documents_in_the_background), d, export_dir, docs));
}

void TapiocaWindow::view_downloaded_document(Tapioca::Document *document, std::string file)
{
  if (progress_aborted)
    {
      progress_aborted = false;
      return;
    }
  std::string prog;
  Glib::RefPtr<Gnome::Conf::Client> conf_client;
  conf_client = Gnome::Conf::Client::get_default_client();
  if (Document::is_video_filename(file) == true)
    prog = conf_client->get_string("/apps/" PACKAGE_NAME 
                                   "/general/video_viewer_path");
  else
    prog = conf_client->get_string("/apps/" PACKAGE_NAME 
                                   "/general/image_viewer_path");

  Glib::ustring dest = 
    temporary_directory + "/" + document->get_export_filename();
  if (file != dest)
    {
      dest = Document::find_free_file(dest);
      Glib::RefPtr<Gio::File> f = Gio::File::create_for_path(file);
      f->move(Gio::File::create_for_path(dest), Gio::FILE_COPY_BACKUP);
      downloaded_document.emit(document);
    }

  std::string cmd = prog + " " + dest;
  if (file != "")
    Glib::spawn_command_line_async(cmd);
}

void TapiocaWindow::on_pulled_gallery_listing(std::list<Gallery*> galleries, Session *s, ProgressDialog &d)
{
  if (progress_aborted)
    {
      progress_aborted = false;
      return;
    }

  d.set_text(String::ucompose("%1 galleries left", galleries.size()));
  d_thumbnail_urls.clear();
  d.pulse();
  if (galleries.size() == 0)
    return;
  //now we grab them in order, while removing them from a list.
  std::list<Gallery*> gallery_list;
  gallery_list.merge(galleries);
  Gallery *gallery = gallery_list.front();
  gallery_list.remove(gallery);
  s->pull_documents_for_gallery(gallery,sigc::bind<std::list<Gallery*>, Session*, ProgressDialog &>(sigc::mem_fun(*this, &TapiocaWindow::on_pulled_documents_for_gallery), gallery_list, s, d), d_thumbnail_urls);
}

void TapiocaWindow::on_thumbnail_downloaded(Document *document, Session *s, std::list<std::string> thumbnail_urls, ProgressDialog &d)
{
  d.set_text(String::ucompose("%1 thumbnails left", thumbnail_urls.size()));
  if (progress_aborted)
    {
      printf("aborted!\n");
      progress_aborted = false;
      return;
    }
  d.pulse();
  if (thumbnail_urls.size() == 0)
    {
      d.hide();
      on_galleries_pulled(s);
      return;
    }
  std::string url = thumbnail_urls.front();
  thumbnail_urls.remove(url);
  s->pull_thumbnail(url, sigc::bind<Session*, std::list<std::string>, ProgressDialog>(sigc::mem_fun(this, &TapiocaWindow::on_thumbnail_downloaded), s, thumbnail_urls, d));
}

void TapiocaWindow::on_pulled_documents_for_gallery(Gallery *gallery, std::list<std::string> &thumbnail_urls, std::list<Gallery*> galleries, Session *s, ProgressDialog &d)
{
  if (progress_aborted)
    {
      progress_aborted = false;
      return;
    }
  d.pulse();
  d.set_text(String::ucompose("%1 galleries left", galleries.size()));
  if (galleries.size() == 0)
    {
      if (thumbnail_urls.size() != 0)
        {
      std::string url = thumbnail_urls.front();
      thumbnail_urls.remove(url);
      s->pull_thumbnail(url, sigc::bind<Session*, std::list<std::string>, ProgressDialog& >(sigc::mem_fun(this, &TapiocaWindow::on_thumbnail_downloaded), s, thumbnail_urls, d));
        }
      return;
    }
  Gallery *g = galleries.front();
  galleries.remove(g);
  s->pull_documents_for_gallery(g,sigc::bind<std::list<Gallery*>, Session *, ProgressDialog &>(sigc::mem_fun(*this, &TapiocaWindow::on_pulled_documents_for_gallery), galleries, s, d), thumbnail_urls);
}

void TapiocaWindow::add_published_galleries(sigc::slot<void> slot)
{
  if (publish_changes.added_galleries.size() == 0)
    {
      slot();
      return;
    }
  Gallery *gallery = 
    session->find_by_id(publish_changes.added_galleries.front().first.get_id());
  if (gallery)
    session->add_gallery
      (gallery, sigc::bind(sigc::mem_fun
                           (*this, 
                            &TapiocaWindow::on_published_gallery_added), 
                           slot));
}

void TapiocaWindow::remove_published_galleries(sigc::slot<void> slot)
{
  if (publish_changes.removed_documents.size() == 0)
    {
      slot();
      return;
    }
  Gallery *gallery = 
    session->get_profile()->find_by_id
    (publish_changes.removed_documents.front().first.get_id());
  if (gallery)
    session->remove_gallery
      (gallery, sigc::bind(sigc::mem_fun
                           (*this, 
                            &TapiocaWindow::on_published_gallery_removed), 
                           slot));
}

void TapiocaWindow::publish(SessionChanges &changes)
{
  publish_changes = changes;
  //kick it off
  add_published_galleries(sigc::mem_fun(*this, &TapiocaWindow::on_published_galleries_added));

}

void TapiocaWindow::on_published_galleries_added()
{
  //now it's time to do the move documents.
  publish_changes.find_moved_docs();
  move_published_documents
    (sigc::mem_fun(*this, &TapiocaWindow::on_published_documents_moved));
}

void TapiocaWindow::move_published_documents(sigc::slot<void> slot)
{
  if (publish_changes.moved_documents.size() == 0)
    {
      slot();
      return;
    }
  Document *src = session->get_profile()->find_doc_by_id
    (publish_changes.moved_documents.front().first.first.get_id());
  Document *dest = session->find_doc_by_id
    (publish_changes.moved_documents.front().second.first.get_id());
  if (src && dest)
    {
      session->move_document
        (src, dest, sigc::bind
         (sigc::mem_fun(this, &TapiocaWindow::on_published_document_moved), 
          slot));
    }
}

void TapiocaWindow::on_published_document_moved(Document *src, Document *dest, sigc::slot<void> slot)
{
  publish_changes.moved_documents.erase
    (publish_changes.moved_documents.begin());

  //Q. what do we do when the doc was moved?
  //A. we get rid of the source document, and update the new document's id.

  Gallery *gallery = session->get_profile()->find_by_id(src->get_gallery_id());
  if (gallery)
    gallery->remove_document(src);
  delete src;
  if (publish_changes.moved_documents.size() > 0)
    move_published_documents(slot);
  else
    slot();
}

void TapiocaWindow::on_published_documents_moved()
{
  remove_published_galleries(sigc::mem_fun(*this, &TapiocaWindow::on_published_galleries_removed));
  //now we remove galleries.
}

void TapiocaWindow::on_published_galleries_removed()
{
  add_published_documents(sigc::mem_fun(*this, &TapiocaWindow::on_published_documents_added));
}

void TapiocaWindow::on_published_documents_added()
{
  remove_published_documents(sigc::mem_fun(*this, &TapiocaWindow::on_published_documents_removed));
}

void TapiocaWindow::add_published_documents(sigc::slot<void> slot)
{
  if (publish_changes.added_documents.size() == 0)
    {
      slot();
      return;
    }
  Document *document = session->find_doc_by_id
    (publish_changes.added_documents.front().first.get_id());
  if (document)
    session->add_document
      (document, sigc::bind(sigc::mem_fun
                           (*this, 
                            &TapiocaWindow::on_published_document_added), 
                           slot));
}

void TapiocaWindow::remove_published_documents(sigc::slot<void> slot)
{
  if (publish_changes.removed_documents.size() == 0)
    {
      slot();
      return;
    }
  Document *document = session->get_profile()->find_doc_by_id
    (publish_changes.removed_documents.front().first.get_id());
  if (document)
    session->remove_document
      (document, sigc::bind(sigc::mem_fun
                           (*this, 
                            &TapiocaWindow::on_published_document_removed), 
                           slot));
}

void TapiocaWindow::on_published_document_removed(Document *document, sigc::slot<void> slot)
{
  publish_changes.removed_documents.remove
    (publish_changes.removed_documents.front());
  Gallery *gallery = 
    session->get_profile()->find_by_id(document->get_gallery_id());
  gallery->remove_document(document);
  delete document;
  if (publish_changes.removed_documents.size() > 0)
    remove_published_documents(slot);
  else
    slot();
}

void TapiocaWindow::on_published_document_added(Document *document, sigc::slot<void> slot)
{
  publish_changes.added_documents.remove
    (publish_changes.added_documents.front());
  if (publish_changes.added_documents.size() > 0)
    add_published_documents(slot);
  else
    slot();
}

void TapiocaWindow::on_published_documents_removed()
{
  //now time for modified galleries, and modified documents
  modify_published_galleries(sigc::mem_fun(*this, &TapiocaWindow::on_published_galleries_modified));
}

void TapiocaWindow::modify_published_galleries(sigc::slot<void> slot)
{
  if (publish_changes.modified_galleries.size() == 0)
    {
      slot();
      return;
    }
  Gallery *gallery = session->find_by_id
    (publish_changes.modified_galleries.front().first.get_id());
  if (gallery)
    session->modify_gallery
      (gallery, sigc::bind(sigc::mem_fun
                           (*this, 
                            &TapiocaWindow::on_published_gallery_modified), 
                           slot));
}

void TapiocaWindow::on_published_gallery_modified(Gallery *gallery, sigc::slot<void> slot)
{
  publish_changes.modified_galleries.remove
    (publish_changes.modified_galleries.front());
  if (publish_changes.modified_galleries.size() > 0)
    modify_published_galleries(slot);
  else
    slot();
}

void TapiocaWindow::on_published_galleries_modified()
{
  modify_published_documents(sigc::mem_fun(*this, &TapiocaWindow::on_published_documents_modified));
}

void TapiocaWindow::modify_published_documents(sigc::slot<void> slot)
{
  if (publish_changes.modified_docs.size() == 0)
    {
      slot();
      return;
    }
  Document *document = session->find_doc_by_id
    (publish_changes.modified_docs.front().first.get_id());
  if (document)
    session->modify_document
      (document, sigc::bind(sigc::mem_fun
                           (*this, 
                            &TapiocaWindow::on_published_document_modified), 
                           slot));
}

void TapiocaWindow::on_published_document_modified(Document *document, sigc::slot<void> slot)
{
  publish_changes.modified_docs.remove
    (publish_changes.modified_docs.front());
  if (publish_changes.modified_docs.size() > 0)
    modify_published_documents(slot);
  else
    slot();
}

void TapiocaWindow::on_published_documents_modified()
{
}
