#!/usr/bin/python

# This file is part of asterisk-phonepatch

# Copyright (C) 2006 Arnau Sanchez
#
# Asterisk-phonepatch is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# Standard Python modules
import sys, optparse, re

__version__ = "$Revision: 1.3 $"
__author__ = "Arnau Sanchez <arnau@ehas.org>"
__depends__ = ['Python-2.3']
__copyright__ = """Copyright (C) 2006 Arnau Sanchez <arnau@ehas.org>.
This code is distributed under the terms of the GNU General Public License."""

COMMENT_CHARS = (";", "#")
EQUAL_CHARS = ("=", ":")
STATE_ON = ("on", "yes", "true", "1", "enabled", "enable", "active")

############################
############################
class Parser:
	"""Parse configuration files using a template.
	
	The template is defined by a list of strings (it can be read from a file
	with the read_template() method), and defines the parametrs and
	associated type, help string and, optionally, the default value (if not given, 
	it would be commented on configuration file and None set to dictionary).

	Example of configuration file:
	
	; This is a comment
	[section1]
	parameter1=value1
	parameter2=value2
	...
	[section2]
	parameter1=value3
	parameter2=value4
	...

	example of template file:
	
	parameter1:
		help = text explaing the function of that parameter
		type = string | int | float | state
		default = default_value
	
	parameter2:
		help = text explaing the function of that parameter
		type = string | int | float | state
		default = default_value
	...
	[section2]
	...
"""
	
	###########################
	def __init__(self, template = None, verbose = False):
		self.verbose = verbose
		self.template = template
		self.header = None
	
	## Private functions
	
	###########################
	def __debug(self, text, force_verbose = False):
		"""Print debug information to standard errror if verbose enabled"""
		if not self.verbose and not force_verbose: return
		sys.stderr.write("parser -- " + text + "\n")
		sys.stderr.flush()

	###########################
	def __parse(self, typedef, strvalue):
		"""Check type for string value"""
		check = True
		if typedef == "float": 
			try: value = float(strvalue)
			except: check = False
		elif typedef == "integer":
			try: value = int(strvalue)
			except: check = False
		elif typedef == "state":
			check = True
			value = (strvalue.lower() in STATE_ON)
		elif typedef == "string":
			check &= True
			value = strvalue
		else: 
			value = strvalue
		if not check:
			return None
		return value

	## Public functions

	###########################
	def set_template(self, template, header = None):
		"""Set template (list)"""
		self.template = template
		if header:
			self.header = header

	###########################
	def get_template(self):
		"""Return template variable"""
		return self.template

	#####################################
	def read_configuration(self, file):
		"""Read a configuration file and return a dictionary containing
		section as keys. A nested dictionary contains the value for each
		parameter"""
		if self.template == None:
			raise ValueError, "Template has not been set, cannot read configuration file"
		
		parameters = dict(self.template).keys()
		dtemplate = dict(self.template)
		fd = open(file)
		section = None
		configuration = {}
		
		# Loop configuration lines
		for line in fd.readlines():
			# Strip line and remove string after a ';' and '#'
			line = line.strip()
			if not line: continue
			if [char for char in COMMENT_CHARS if line.find(char) == 0]: continue

			# Check if it's a section: [section]
			if line[0] == "[" and line[-1] == "]":
				section = line[1:-1]
				configuration[section] = {}
				continue
			
			# Check if we are inside a section configuration
			if not section: 
				self.__debug("out of section %s" %line, True); continue
			
			# Split parameter... par = value
			for char in EQUAL_CHARS:
				if line.index(char) >= 0:
					break
			else: continue
			
			try: key, value = line.split(char)
			except: self.__debug("syntax error: %s" %line, True); continue
			
			# Remove char <"> from values
			key, value = key.strip(), value.strip().replace('"', '')
			
			if not dtemplate.has_key(key): 
				self.__debug("parameter unknown in section %s: %s" %(section, key), True); continue
			
			# Parse parameter and transform to given type
			dvalues = dict(dtemplate[key])
			typedef = dvalues["type"]
			
			value = self.__parse(typedef, value)
			if value == None: self.__debug("parsing error: %s (should be of type <%s>)" %(line, typedef), True); continue
			if dvalues.has_key("choices"):
				choices = [x.strip() for x in dvalues["choices"].split(",")]
				if value not in type(value)(choices):
					self.__debug("Value error for parameter %s: %s" %(key, str(value)))			
			configuration[section][key] = value

		configuration[None] = {}
		for parameter, options in dtemplate.items():
			dvalues = dict(dtemplate[parameter])
			typedef = dvalues["type"]
			if dvalues.has_key("default"):
				configuration[None][parameter] = self.__parse(typedef, dvalues["default"])
			else: configuration[None][parameter] = None

		fd.close()

		return configuration

	#####################################
	def print_template(self):
		"""Print configuration template"""
		if self.template == None:
			raise ValueError, "Template has not been set"
		for name, options in self.template:
			print "parameter:", name
			for key, value in options:
				print "\t%s: %s" %(key, value)

	#####################################
	def print_configuration(self, configuration):
		"""Print an already read configuration"""
		for section, params in configuration.items():
			print "section: %s" %section
			for name, value in params.items():
				print "  * %s = %s" %(name, value)

	#####################################
	def generate_conf(self, comment="; ", operator=" = ", help=False):
		"""Generate a configuration file according to template"""
		if self.header:
			print self.header,
		for name, options in self.template:
			doptions = dict(options)
			if help: print "%s%s: %s" %(comment, name, doptions["help"])
			try: default = doptions["default"]
			except: print comment + name + operator
			else: print name + operator + default

	#####################################
	def read_template(self, file):
		"""Read a template from a file"""
		fd = open(file)
		template, lparameter = [], []
		parameter = None
		header = ""
		for line in fd.readlines():
			if not line: continue
			if not parameter and line[0] in COMMENT_CHARS: 
				header += line
				continue
			line = line.strip()
			if not line: continue
						
			# Find parameter
			if line[-1] == ":":
				if parameter:
					template.append([parameter, lparameter])
					lparameter = []
				parameter = line[:-1].strip()
				continue
				
			# It's an option, get value
			option, value = line.split("=", 1)
			option ,value = option.strip(), value.strip()
			lparameter.append((option, value))
		
		# Last parameter
		if parameter: template.append([parameter, lparameter])
		fd.close()
		self.header = header
		self.template = template

#####################################
def main():
	usage = """
	config.py [options] template

	Parse a phonepatch configuration file"""

	parser = optparse.OptionParser(usage)
	parser.add_option('-v', '--verbose', dest='verbose', default = False, action='store_true', help = 'Enable verbose mode')
	parser.add_option('-g', '--generate',  dest='generate', default = False, action='store_true', help = 'Print section definitions')
	parser.add_option('-e', '--generate-help',  dest='generate_help', default = False, action='store_true', help = 'Discard help entries on generated configuration')
	parser.add_option('-p', '--parse-file',  dest='parse_file', type = "string", default = None, help = 'Parse a configuration file')
	options, args = parser.parse_args()

	if len(args) < 1:
		parser.error("You need to specify template file")

	template_file = args[0]
		
	cfg = Parser(verbose = options.verbose)
	cfg.read_template(template_file)
	if options.generate:
		cfg.generate_conf(comment=";", operator="=", help = options.generate_help)
	elif options.parse_file:
		cfg.read_configuration(options.parse_file)
	else:
		cfg.print_template()
	
	sys.exit(0)
		
#####################################
#####################################
if __name__ == "__main__":
	main()
