#!/usr/bin/python

# This file is part of asterisk-phonepatch

# Copyright (C) 2006 Arnau Sanchez
#
# Asterisk-phonepatch is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# Standard Python modules
import sys, optparse

__version__ = "$Revision: 1.1 $"
__author__ = "Arnau Sanchez <arnau@ehas.org>"
__depends__ = ['Python-2.3']
__copyright__ = """Copyright (C) 2006 Arnau Sanchez <arnau@ehas.org>.
This code is distributed under the terms of the GNU General Public License."""

COMMENT_CHARS = (";", "#")
EQUAL_CHARS = ("=", ":")

############################
############################
class Parser:
	"""Parse configuration files using a template.
	
	Configuration files have a syntax similar to RFC822, consisting of [section]
	and "parameter = value" or "parameter: value". Comments must begin 
	with ";" or "#" (inline comments not allowed). 
	
	The template is defined by a list of strings (it can be read from a file
	with the read_template() method), and defines the sections and the
	parameters inside each section. For each parameter, it defines its type, 
	the associated help string and, optionally, the default value (if not given, 
	it would be commented on configuration file and None set to dictionary):

	Example of configuration file:
	
	; This is a comment
	[section1]
	parameter1=value1
	parameter2=value2
	...
	[section2]
	...

	example of template file:
	
	[section1]
	parameter1:
		help = text explaing the function of that parameter
		type = string | int | float | state
		default = default_value
	
	parameter2:
		help = text explaing the function of that parameter
		type = string | int | float | state
		default = default_value
	...
	[section2]
	...
"""
	
	###########################
	def __init__(self, template = None, verbose = False):
		self.verbose = verbose
		self.template = template
		self.header = None
	
	## Private functions
	
	###########################
	def __debug(self, text, force_verbose = False):
		"""Print debug information to standard errror if verbose enabled"""
		if not self.verbose and not force_verbose: return
		sys.stderr.write("parser -- " + text + "\n")
		sys.stderr.flush()

	###########################
	def __parse(self, typedef, strvalue):
		"""Check type for string value"""
		check = True
		if typedef == "float": 
			try: value = float(strvalue)
			except: check = False
		elif typedef == "integer":
			try: value = int(strvalue)
			except: check = False
		elif typedef == "state":
			check &= strvalue in ("on", "off")
			if strvalue == "on": value = True
			else: value = False
		elif typedef == "string":
			check &= True
			value = strvalue
		else: 
			calue = strvalue
		if not check:
			return None
		return value

	## Public functions

	###########################
	def set_template(self, template, header = None):
		"""Set template (list)"""
		self.template = template
		if header:
			self.header = header

	###########################
	def get_template(self):
		"""Return template variable"""
		return self.template

	#####################################
	def read_configuration(self, file):
		"""Read a configuration file and return a dictionary containing
		section as keys. A nested dictionary contains the value for each
		parameter"""
		if self.template == None:
			raise ValueError, "Template has not been set, cannot read configuration file"
		
		sections = dict(self.template).keys()
		dtemplate = dict(self.template)
		fd = open(file)
		section = None
		configuration = {}
		
		# Loop configuration lines
		for line in fd.readlines():
			# Strip line and remove string after a ';' and '#'
			line = line.strip()
			if not line: continue
			if 0 in [line.find(char) for char in COMMENT_CHARS]: continue

			# Check if it's a section: [section]
			if line[0] == "[" and line[-1] == "]":
				section = line[1:-1]
				if section not in sections: 
					self.__debug("warning: unknown section %s" %section, True); continue
				# Create section inside the configuration
				configuration[section] = {}
				continue
			
			# Check if we are inside a section configuration
			if not section: self.__debug("out of section %s" %line, True); continue
			
			# Split parameter... par = value
			for char in EQUAL_CHARS:
				if line.index(char) >= 0:
					break
			else: continue
			
			try: key, value = line.split(char)
			except: self.__debug("syntax error: %s" %line, True); continue
			
			# Remove char <"> from values
			key, value = key.strip(), value.strip().replace('"', '')
			if not dtemplate.has_key(section):
				self.__debug("parameter inside unknown section %s: %s" %(section, key), True); continue
			dsection = dict(dtemplate[section])
			if not dsection.has_key(key): 
				self.__debug("parameter unknown in section %s: %s" %(section, key), True); continue
			
			# Parse parameter and transform to given type
			typedef = dict(dsection[key])["type"]
			
			value = self.__parse(typedef, value)
			if value == None: self.__debug("parsing error: %s (should be of type <%s>)" %(line, typedef), True); continue
			
			configuration[section][key] = value
		
		# Close file
		fd.close()
		
		# Scan section template to look for non-initialized parameters
		for section, values in self.template:
			for value in values:
				key = value[0]
				dvalues = dict(value[1])
				if not configuration.has_key(section):
					configuration[section] = {}
				if not configuration[section].has_key(key):
					# Not in configuration file, set to default and parse
					typedef = dvalues["type"]
					value = dvalues.get("default", None)
					if value != None:
						value = self.__parse(typedef, value)
					configuration[section][key] = value
				
		return configuration

	#####################################
	def print_template(self):
		"""Print configuration template"""
		if self.template == None:
			raise ValueError, "Template has not been set"
		for section, params in self.template:
			print "section: %s" %section
			for name, options in params:
				print "  * parameter:", name
				for key, value in options:
					print "    - %s: %s" %(key, value)

	#####################################
	def print_configuration(self, configuration):
		"""Print an already read configuration"""
		for section, params in configuration.items():
			print "section: %s" %section
			for name, value in params.items():
				print "  * %s = %s" %(name, value)

	#####################################
	def generate_conf(self, comment = "; ", operator = " = ", help = False):
		"""Generate a configuration file according to template"""
		if self.header:
			print self.header,
		for section, params in self.template:
			print "\n[%s]" %section
			for name, options in params:
				doptions = dict(options)
				if help: print "%s%s: %s" %(comment, name, doptions["help"])
				try: default = doptions["default"]
				except: print comment + name + operator
				else: print name + operator + default

	#####################################
	def read_template(self, file):
		"""Read a template from a file"""
		fd = open(file)
		template, lsection, lparameter = [], [], []
		section = parameter = None
		header = ""
		for line in fd.readlines():
			if not line: continue
			if not section and line[0] in COMMENT_CHARS: header += line
			line = line.strip()
			
			# Remove comments
			for char in COMMENT_CHARS:
				index = line.find(char)
				if index >= 0: 
					line = line[:index]
			if not line: continue
			
			# Find section
			if line[0] == "[" and line[-1] == "]":
				if parameter:
					lsection.append([parameter, lparameter])
					lparameter = []
					parameter = None
				if section:
					template.append([section, lsection])
					lsection =[]
				section = line[1:-1]
				continue
			
			# Find parameter
			if line[-1] == ":":
				if parameter:
					lsection.append([parameter, lparameter])
					lparameter = []
				parameter = line[:-1]
				continue
				
			if line.find("=") < 0: continue
				
			# It's an option, get value
			option, value = line.split("=")
			option ,value = option.strip(), value.strip()
			lparameter.append((option, value))
		
		# Last section 
		if parameter: lsection.append([parameter, lparameter])
		if section: template.append([section, lsection])
		
		fd.close()
		self.header = header
		self.template = template

#####################################
def main():
	usage = """
	config.py [options] template

	Parse a phonepatch configuration file"""

	parser = optparse.OptionParser(usage)
	parser.add_option('-v', '--verbose', dest='verbose', default = False, action='store_true', help = 'Enable verbose mode')
	parser.add_option('-g', '--generate',  dest='generate', default = False, action='store_true', help = 'Print section definitions')
	parser.add_option('-e', '--generate-help',  dest='generate_help', default = False, action='store_true', help = 'Discard help entries on generated configuration')
	parser.add_option('-p', '--parse-file',  dest='parse_file', type = "string", default = None, help = 'Parse a configuration file')
	options, args = parser.parse_args()
	
	if len(args) < 1:
		parser.error("You need to specify template file")
	template_file = args[0]
		
	cfg = Parser(verbose = options.verbose)
	cfg.read_template(template_file)
	
	if options.generate:
		cfg.generate_conf(comment=";", operator="=", help = options.generate_help)
	elif options.parse_file:
		cfg.read_configuration(options.parse_file)
	else:
		cfg.print_template()

	sys.exit(0)
		
#####################################
#####################################
if __name__ == "__main__":
	main()
