#!/usr/bin/python

# This file is part of asterisk-phonepatch

# Copyright (C) 2006 Arnau Sanchez
#
# Asterisk-phonepatch is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

import os, sys, re, array
import optparse, fcntl, math
import ossaudiodev as oss
import alsaaudio as alsa

## TODO:

## Implement half/full-duplex mode

__version__ = "$Revision: 1.1 $"
__author__ = "Arnau Sanchez <arnau@ehas.org>"
__depends__ = ['AlsaAudio', 'OSSAudioDev', 'Python-2.4']
__copyright__ = """Copyright (C) 2006 Arnau Sanchez <arnau@ehas.org>.
This code is distributed under the terms of the GNU General Public License."""

AUDIO_FORMATS = {
	"u8": {"oss": oss.AFMT_U8, "alsa": alsa.PCM_FORMAT_U8}, 
	"s8": {"oss": oss.AFMT_S8, "alsa": alsa.PCM_FORMAT_S8},
	"s16le": {"oss": oss.AFMT_S16_LE, "alsa": alsa.PCM_FORMAT_S16_LE},
	"u16le": {"oss": oss.AFMT_U16_LE, "alsa": alsa.PCM_FORMAT_U16_LE},
	"s16be": {"oss": oss.AFMT_S16_BE, "alsa": alsa.PCM_FORMAT_S16_BE},
	"u16be": {"oss": oss.AFMT_U16_BE, "alsa": alsa.PCM_FORMAT_U16_BE},
	"alaw": {"oss": oss.AFMT_A_LAW, "alsa": alsa.PCM_FORMAT_A_LAW},
	"ulaw": {"oss": oss.AFMT_MU_LAW, "alsa": alsa.PCM_FORMAT_MU_LAW}}
	
###
def create_dict(**args): return args
	
###############################################
class Soundcard:
	"""High-level wrapping for OSS (ossaudiodev) and ALSA (alsasound) modules.
	
	Default mode is full-duplex with CD quality (stereo, 44100 sps, s16le: 16-bits 
	signed little-endian samples)"""

	#####
	## Generic functions
	####################################################	
	
	##################################################
	def __init__(self, **args):
		"""Open soundcard with the following args:
		
		library: oss | alsa
		channels: 1 or 2.
		samplerate: soundcard rate
		sampleformat: u8 | s8 | s16le, u16le, s16be, u16be, alaw, ulaw
		mode: "r" | "w" | "rw" (read, write or read/write modes)
		nonblock: whether writes to soundcard block (only OSS)
		fullduplex: full-duplex (send and write on same time) operation
		fragment_size: buffer fragment size (only OSS)
		"""
		self.library = str(args.get("library", "alsa")).lower()
		if self.library not in ("alsa", "oss"):
			raise NameError, "audio library %s unknown, available: 'alsa' or 'oss'" % library
		self.channels = int(args.get("channels", 2))
		self.samplerate = int(args.get("samplerate", 44100))
		self.sampleformatstr = str(args.get("sampleformat", "s16le")).lower()
		try: self.sampleformat = AUDIO_FORMATS[self.sampleformatstr][self.library]
		except: raise NameError, "Sample format not supported: %s" %self.sampleformatstr
		if self.sampleformat == None:
			raise NameError, "Sample format not supported by library %s: %s" %(self.library, self.sampleformatstr)
		self.fullduplex = bool(args.get("fullduplex", True))
		self.fragment_size = int(args.get("fragment_size", 0))
		self.nonblock = bool(args.get("nonblock", False))
		self.mode = str(args.get("mode", "rw"))
		self.opened = True
		self.getmethod("init")(**args)

	##################################################
	def getoptions(self):
		return create_dict(library = self.library, device = self.device, channels = self.channels, \
			samplerate = self.samplerate, sampleformat = self.sampleformatstr, \
			fullduplex = self.fullduplex, mode = self.mode, fragment_size = self.fragment_size)
			
	##################################################
	def getmethod(self, name):
		complete = self.library + "_" + name
		try: method = getattr(self, complete) 
		except: raise NameError, "Method not implemented: %s" %complete
		return method
	
	##################################################
	def read(self, max):
		if not self.opened:
			raise IOError, "cannot read from %s: device is not opened" %self.device
		if "r" not in self.mode:
			raise IOError, "cannot read from %s: device not opened in read mode" %self.device
		return self.getmethod("read")(max)

	##################################################
	def write(self, data):
		if not self.opened:
			raise IOError, "cannot write to %s: device is not opened" %self.device
		if "w" not in self.mode:
			raise IOError, "cannot write to %s, device not opened in write mode" %self.device
		return self.getmethod("write")(data)

	##################################################
	def open(self):
		if  self.opened:
			raise IOError, "cannot open %s: device already opened" %self.device
		self.opened = True
		return self.getmethod("open")()

	##################################################
	def close(self):
		if not self.opened:
			raise IOError, "cannot close %s: device not opened" %self.device
		self.opened = False
		return self.getmethod("close")()

	##################################################
	def fileno(self):
		return self.getmethod("fileno")()

	##################################################
	def noutbuffer(self):
		return self.getmethod("noutbuffer")()

	##################################################
	def sync(self):
		return self.getmethod("sync")()

	#####
	## ALSA specific functions
	####################################################	

	####################################################	
	def alsa_init(self, **args):
		self.device = str(args.get("device", "default"))
		self.periodsize = int(args.get("periodsize", 256))
		self.alsa_open()
		
	####################################################	
	def alsa_open(self):
		block = 0
		if not self.nonblock:  block = alsa.PCM_NONBLOCK
		for mode in self.mode:
			modeflag = {"r": alsa.PCM_CAPTURE, "w": alsa.PCM_PLAYBACK}[mode]
			print self.device
			fd = alsa.PCM(modeflag, block, self.device)
			fd.setchannels(self.channels)
			fd.setrate(self.samplerate)
			fd.setformat(self.sampleformat)
			fd.setperiodsize(self.periodsize)
			dict_name = {"r": "fd_read", "w": "fd_write"}
			setattr(self, dict_name[mode], fd)

	##################################################
	def alsa_write(self, buffer):
		return self.fd_write.write(buffer)

	##################################################
	def alsa_close(self):
		if "r" in self.mode: 
			del self.fd_read
		if "w" in self.mode: 
			del self.fd_write

	##################################################
	def alsa_read(self, max):
		output = ""
		while len(output) < max: 
			output += self.fd_read.read()[1]	
		return output

	####################################################	
	def alsa_fileno(self, mode):
		if mode == "w":
			return self.fd_write.fileno()
		else:
			return self.fd_read.fileno()

	####################################################	
	def alsa_noutbuffer(self):
		# Not implemented
		return 0

	####################################################	
	def alsa_sync(self):
		# Not implemented
		return
		
	#####
	## OSS specific functions
	####################################################	

	####################################################	
	def oss_init(self, **args):
		self.device = str(args.get("device", "/dev/dsp"))
		self.oss_open()
				
	####################################################	
	def oss_open(self):
		self.fd = oss.open(self.device, self.mode)
		
		# Set fragment size operation is not supported by ossaudiodev, do with ioctl
		if self.fragment_size:
			arg = 0x7FFF0000 + int(math.log(self.fragment_size, 2))
			fragment = array.array('L', [arg])
			fcntl.ioctl(self.fd, oss.SNDCTL_DSP_SETFRAGMENT, fragment, 1)
		self.fd.setparameters(self.sampleformat, self.channels, self.samplerate)
		if self.nonblock: 
			self.fd.nonblock()

	##################################################
	def oss_read(self, max):
		return self.fd.read(max)

	##################################################
	def oss_write(self, buffer):
		return self.fd.write(buffer)

	##################################################
	def oss_close(self):
		self.fd.close()

	####################################################	
	def oss_fileno(self):
		return self.fd.fileno()

	####################################################	
	def oss_noutbuffer(self):
		return self.fd.obufcount()

	####################################################	
	def oss_sync(self):
		return self.fd.sync()


###########################
def show_options(sc):
	for parameter, value in sc.getoptions().items():
		sys.stderr.write("%s = %s\n" %(parameter.replace("_", " "), value))
		sys.stderr.flush()

###########################
def main():
	usage = """
	soundcard.py [options] play | record
	
	Soundcard RAW audio player and recorder"""
	
	parser = optparse.OptionParser(usage)

	parser.add_option('-v', '--verbose', dest='verbose', default = False, action='store_true', help = 'enable verbose mode')
	parser.add_option('-d', '--device', dest='device', default = "", metavar='DEVICE', type='str', help = 'audio device')
	parser.add_option('-s', '--samplerate', dest='samplerate', default = 44100, metavar='SPS', type='int', help = 'sampling rate')
	parser.add_option('-f', '--sampleformat', dest='sampleformat', default = "s16le", metavar='s16le', type='string', help = 'sample format (u8/s8/s16le/u16le/s16be/u16be/alaw/ulaw)')
	parser.add_option('-c', '--channels', dest='channels', default = "2", metavar='NUMBER', type='int', help = 'audio channels')
	parser.add_option('-l', '--library', dest='library', default = "oss", metavar='LIBRARY', type='str', help = 'audio library (available: OSS, ALSA)')
	parser.add_option('-b', '--buffersize', dest='buffersize', default = 1024, metavar = "BYTES", type = int, help = 'Buffer size for input/output')

	options, args = parser.parse_args()

	if len(args) != 1: parser.print_help(); sys.exit(1)
	
	command = args[0]
	options.library = options.library.lower()

	if options.library not in ("alsa", "oss"):
		print "audio library unknown:", options.library
		parser.print_help()
		sys.exit(1)

	if options.library == "alsa": options.device = options.device or "default"
	elif options.library == "oss": options.device = options.device or "/dev/dsp"
	
	if command == "play":
		sc = Soundcard(device = options.device, channels = options.channels, \
				mode = "w", library = options.library, samplerate = options.samplerate, \
				sampleformat = options.sampleformat)
		if options.verbose: show_options(sc)
		while 1:
			buffer = os.read(0, options.buffersize)
			if not buffer: break
			sc.write(buffer)
		sc.close()
				
	elif command == "record":
		sc = Soundcard(device = options.device, channels = options.channels, \
				mode = "r", library = options.library, samplerate = options.samplerate, \
				sampleformat = options.sampleformat)
		if options.verbose: show_options(sc)
		while 1:
			try: os.write(1, sc.read(options.buffersize))
			except: break
		sc.close()
			
	sys.exit(0)

############################
if __name__ == "__main__":
	main()