#!/usr/bin/python

# This file is part of asterisk-phonepatch

# Copyright (C) 2006 Arnau Sanchez
#
# Asterisk-phonepatch is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# Standard Python modules
import sys, time
import errno, audioop

# External phonepatch modules
import soundcard

__version__ = "$Revision: 1.1 $"
__author__ = "Arnau Sanchez <arnau@ehas.org>"
__depends__ = ['Python-2.4']
__copyright__ = """Copyright (C) 2006 Arnau Sanchez <arnau@ehas.org>.
This code is distributed under the terms of the GNU General Public License."""

		
###############################
###############################
class Radio:
	"""Use soundcard and external PTT apps to interface with a radio transceiver.
	
	Thie class provices a simple and easy way to send and receive audio voice 
	(even data) with standard transceivers, using soundcard as D/A and D/A 
	converter. Moreover, it allows controlling the PTT (Push-to-Talk) line which
	toggle between the reception (Rx) or transmition (Tx) state. 
	"""

	###############################
	def __init__(self, soundcard_device, samplerate, buffer_samplerate, radio_control, ptt, carrier_detection, verbose = False, soundcard_retries = 1, **args):
		"""Open a soundcard and PTT interface.

		Use radio_control object to set PTT (method-name: set_ptt) and get 
		carrier-detection state (method-name: get_carrier).
		
		Soundcard device should be OSS files (/dev/dspX). Parameter 
		<samplerate> will be the rate used by soundcard while 
		<buffer_samplerate> stands for the rate of the read/write
		buffer that it receives/send. Note that different samplerates
		need a CPU-cost processing.
		
		PTT object is an instance  of ExecInterface with "on" and "off"
		commands defined.
		"""
		self.samplerate = samplerate
		self.buffer_samplerate = buffer_samplerate
		self.verbose = verbose
		self.radio_control = radio_control
		if self.radio_control:
			self.ptt = ptt
			self.carrier_detection = carrier_detection
		else:
			self.ptt = self.carrier_detection = False
		
		# PTT parameters
		self.threshold_def = args.get("ptt_threshold_signal", 0.01)
		self.tailtime_def = args.get("ptt_tail_time", 0)
		self.maxtime_def = args.get("ptt_max_time", 120)
		self.waittime_def = args.get("ptt_wait_time", 5)
		
		# Carrier parameters
		self.fullduplex = args.get("fullduplex", False)
		self.carrier_polling = args.get("carrier_polling", 0.1)
		
		# Soundcard parameters
		self.sampleformat = "s16le"
		self.audio_channels = 1
		self.buffer_size = 1024
		self.sample_width = 2
		self.sample_max = 2.0**(2*8) / 2.0
		
		# Reduce soundcard fragment size (default is 4096), to reduce latency
		self.fragment_size = 128
		
		self.onoff_dict = {False: "off", True: "on"}
		self.offtime = self.ontime = self.tailtime = 0
		self.carrier_state = False

		# Open soundcard
		self.soundcard = None
		self.soundcard_device = soundcard_device
		while 1:
			try: self.soundcard = soundcard.Soundcard(device = soundcard_device, \
					channels = self.audio_channels, mode = "rw", library = "oss", \
					samplerate = samplerate, sampleformat = self.sampleformat, \
					fragment_size = self.fragment_size)
			except IOError, (nerror, detail): 
				if nerror != errno.EBUSY: break
				soundcard_retries -= 1
				if not soundcard_retries: break
				self.debug("soundcard busy, remaining retries: %d" %soundcard_retries)
				time.sleep(1)
			else: break
				
		if not self.soundcard:		
			raise IOError, "cannot open soundcard: %s" %soundcard_device
		
		if samplerate != self.buffer_samplerate:
			self.debug("warning: soundcard samplerate is set to %d, but output is %s" %(samplerate, self.buffer_samplerate))
			self.debug("warning: resampling input/output audio with CPU penalty")
			
		# Turn PTT off at start (for safety)
		self.set_ptt(False)
		
	###################################
	def debug(self, args, exit = False):
		"""Write logs to standard error if enabled"""
		if not self.verbose: return
		log = "radio -- "
		if exit: log += "fatal error - "
		log += str(args) + "\n"
		sys.stderr.write(log)
		sys.stderr.flush()
		if exit: sys.exit(1)
		
	###################################
	def get_audiofd(self):
		"""Get audio file descriptor used to interface soundcard"""
		return self.soundcard

	#####################################
	def read_audio(self, size, resample = False):
		"""Read data from soundcard 
		
		Resample audio buffer to <buffer_samplerate> if resampling 
		enabled. If carrier detection is enabled, return audio buffer only
		if carrier is detected."""
		if not self.soundcard: self.debug("soundcard not opened"); return
		buffer = self.soundcard.read(size)
		if not buffer: return
			
		# Return a void buffer if there is no carrier detection
		if self.carrier_detection and not self.carrier_state:
			return "\x00" * size
		
		# Resample if asked and if it's necessary
		if resample and self.samplerate != self.buffer_samplerate:
			buffer, state = audioop.ratecv(buffer, self.sample_width, self.audio_channels, self.samplerate, self.buffer_samplerate, self.rateread_state)
			self.rateread_state = state
			
		return buffer

	#####################################
	def update_carrier_state(self):
		"""Update carrier_detection state"""
		if not self.carrier_detection: return
		try: next_time = self.time_next_carrier
		except: next_time = 0
		now = time.time()
		if now > next_time:
			try: self.carrier_state = self.radio_control.get_carrier()
			except: self.debug("cannot get carrier state"); return
			self.time_next_carrier = now + self.carrier_polling
		try: old = self.old_carrier_state
		except: old = None
		if old != self.carrier_state:
			self.debug("new carrier state: %s" %self.onoff_dict[self.carrier_state])
			self.old_carrier_state = self.carrier_state
		
	########################################
	def is_ptt_blocked(self):
		"""Return a bool indicating if is possible to set the ptt on, 
		otherwise the carrier detection is blocking it"""
		if not self.carrier_detection or self.fullduplex or not self.carrier_state:
			return False
		return True
		
	#####################################
	def vox_process(self, buffer, buffer_rate = 0):
		"""VOX PTT processing.
		
		Set PTT on if audio data in buffer reaches the threshold. 
		Control minimum and maximum PTT on/off states.
		"""
		if not self.soundcard: raise IOError, "Soundcard not opened"
		
		if not self.ptt: 
			self.send_audio(buffer, buffer_rate)
			return
		
		# Get power of audio fragment for VOX
		power = audioop.rms(buffer, self.sample_width) / self.sample_max
		
		now = time.time()
		
		if power >= self.threshold_def and now >= self.ontime:
			self.tailtime = now + self.tailtime_def
			if not self.get_ptt():
				self.debug("input power threshold reached: %0.4f" %self.threshold_def)
				# Thereshold for PTT reached, but check before if not carrier is detected
				if not self.is_ptt_blocked():
					self.set_ptt(True)
					self.ontime = 0
					self.offtime = now + self.maxtime_def
				else:
					self.debug("PTT blocked due to carrier detection")
			elif self.offtime and now >= self.offtime and not self.ontime:
				self.debug("ptt_max_time timed out: turn PTT off and wait %d seconds" %self.waittime_def)
				self.set_ptt(False)
				self.ontime = now + self.waittime_def
				
		elif self.get_ptt():
			if now >= self.tailtime:
				self.debug("ptt_tail_time reached")
				self.set_ptt(False)
			elif self.is_ptt_blocked():
				self.debug("PTT blocked due to carrier detection")
				self.set_ptt(False)

		#if self.get_ptt():
		self.send_audio(buffer, buffer_rate)
			
	#####################################
	def send_audio(self, buffer, buffer_rate = 0):
		"""Send audio to radio transceiver using the soundcard"""
		if not self.soundcard: self.debug("soundcard not opened"); return
		if not buffer: return
		
		# Resample if asked and if it's necessary
		if buffer_rate and buffer_rate != self.samplerate:
			buffer, state = audioop.ratecv(buffer, self.sample_width, self.audio_channels, buffer_rate, self.samplerate, self.ratewrite_state)
			self.ratewrite_state = state			
		
		self.soundcard.write(buffer)

	#####################################
	def flush_audio(self):
		"""Flush buffer soundcard"""
		self.soundcard.sync()

	###################################
	def close(self):
		"""Close radio interface"""
		self.debug("closing radio interface")
		
		if self.soundcard: 
			self.soundcard.close()
			self.soundcard = None
			self.debug("soundcard closed")
		else:
			self.debug("soundcard was not opened")
		
		if self.radio_control: 
			self.debug("closing radio-control interface")
			self.set_ptt(False)
			self.radio_control.close()

	###################################
	def set_ptt(self, state):
		"""Set PTT state (bool)"""
		if not self.ptt: return
		self.debug("Setting PTT %s" %self.onoff_dict[state])
		try: self.radio_control.set_ptt(state)
		except: self.debug("error setting ptt")

	###################################
	def get_ptt(self):
		"""Get PTT state -> bool"""
		if not self.ptt: return False
		return self.radio_control.get_ptt()
