
from twisted.trial import unittest
import os
import shutil
import logging

class QueueSetup(object):
    """Setup a queue with mock messages in it."""
    def __init__(self):
        self.configFileName = "Foo"
        self.cwd = os.path.abspath(os.path.curdir)
        self.message = ("From: John.Citizen@example.com\n"
                        "Subject: A new action\n"
                        "star-merge archive@example.com/foo--bar--0"
                            "archive2@example.com/bar--foo--1")

    def setUp(self):
        myFile=open(self.configFileName, "w")
        myFile.write("[DEFAULT]\n"
                     "queuedir=%s/queue" % self.cwd)
        myFile.close()
        self.queuedir = os.path.join(self.cwd, "queue")
        try:
            os.mkdir(self.queuedir)
        except OSError:
            pass
        self.messageFileName = os.path.join(self.queuedir, "patch.00000001")
        messageFile = open(self.messageFileName, "w")
        messageFile.write(self.message)
        messageFile.close()

    def tearDown(self):
        os.unlink(self.configFileName)
        shutil.rmtree(self.queuedir, ignore_errors=True)

class TestTwistedUI(unittest.TestCase):

    def testImports(self):
        import arch_pqm.ui.twistd

    def testQueue(self):
        import arch_pqm
        queueSetup = QueueSetup()
        queueSetup.setUp()
        try:
            configp = arch_pqm.ConfigParser()
            configp.read([queueSetup.configFileName])
            queuedir = arch_pqm.get_queuedir(configp, logging, [])
            patches =  arch_pqm.find_patches(queuedir, logging, None)
            self.assertEqual(1, len(patches))
            self.assertEqual(patches[0].filename, queueSetup.messageFileName)
        finally:
            queueSetup.tearDown()

class TestScript(unittest.TestCase):
    """Test the script command objects"""
    
    def testName(self):
        import arch_pqm
        patch = arch_pqm.Script('foo.script', logging, False)
        self.assertEqual(patch.filename, 'foo.script')

    def testFields(self):
        import arch_pqm
        scriptname = 'foo.script'
        message = ("From: John.Citizen@example.com\n"
                   "Subject: A new action\n"
                   "star-merge archive@example.com/foo--bar--0"
                       "archive2@example.com/bar--foo--1")

        scriptFile = open(scriptname, "w")
        scriptFile.write(message)
        scriptFile.close()
        try:
            script = arch_pqm.Script(scriptname, logging, False)
            self.assertEqual(script.getSender(), "John.Citizen@example.com")
            self.assertEqual(script.getSubject(), "A new action")
            self.assertEqual(script.getLines(), 
                [("star-merge archive@example.com/foo--bar--0"
                 "archive2@example.com/bar--foo--1")])
        finally:
            os.unlink(scriptname)

