# -*- mode: python; coding: utf-8 -*-
# vim:smartindent cinwords=if,elif,else,for,while,try,except,finally,def,class:ts=4:sts=4:sta:et:ai:shiftwidth=4
#
# Copyright ©  2004 Canonical Ltd.
#	Author: Robert Collins <robertc@robertcollins.net>

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from PQMConfigParser import ConfigParser
import os
import re
import stat
import sys
import email

# These need thought and layer - but for now, are just moved
# from the binary script intact
keyring = None
used_transactions = {}

def get_queuedir(config_parser, logger, args):
    """Get the queuedir that should be used from the config"""
    if config_parser.has_option('DEFAULT', 'queuedir'):
        return os.path.abspath(os.path.expanduser(
            config_parser.get('DEFAULT', 'queuedir')))
    elif len(args) > 0:
        return os.path.abspath(args[0])
    else:
        logger.error("No queuedir specified on command line or"
                     " in config_parser files.")
        sys.exit(1)
        
configfile_names = ['/etc/arch-pqm.conf']
configfile_names.append(os.path.expanduser('~/.arch-pqm.conf'))
configfile_names.append(os.path.expanduser('~/.tla-pqm.conf'))

class Script(object):
    """A command script."""

    def __init__(self, filename, logger, verify_sigs):
        """Create a script for a given file."""
        self.filename = filename
        self.logger = logger
        self.verify_sigs = verify_sigs
        self.readComplete = False

    def _read(self):
        """Read in the script details."""
        details = read_email(self.logger, open(self.filename))
        (self.sender, self.subject, self.msg, self.sig) = details
        if self.verify_sigs:
            sigid,siguid = verify_sig(self.sender,
                                      self.msg,
                                      self.sig, 
                                      0,
                                      self.logger)
        self.msg = email.message_from_file(open(self.filename))
        self.sender = self.msg['From']
        self.logger.info ('sender: %s' , self.sender)
        self.logger.info ('subject: %s' , self.subject)
        self.readComplete = True

    def getSender(self):
        """Get the sender of the script."""
        if not self.readComplete:
            self._read()
        return self.sender

    def getSubject(self):
        """Get the subject of the script."""
        if not self.readComplete:
            self._read()
        return self.subject

    def getLines(self):
        """Get the lines in the script."""
        if not self.readComplete:
            self._read()
        return self.msg.get_payload().split('\n')

def find_patches(queuedir, logger, verify_sigs):
    patches=[]
    patches_re=re.compile('^patch\.\d+$')
    for f in os.listdir(queuedir):
        if patches_re.match(f):
            fname=os.path.join(queuedir, f)
            patches.append((Script(fname, logger, verify_sigs), 
                            os.stat(fname)[stat.ST_MTIME]))
    def sortpatches(a, b):
        return cmp(a[1],b[1])
    patches.sort(sortpatches)
    return [patch[0] for patch in patches]
 
def read_email(logger, file=None):
    """Read an email and check it has the required structure for commands."""
    if not file:
        file = sys.stdin
    msg = email.message_from_file(file)
    sender = msg['From']
    logger.info("recieved email from %s", sender)
    subject = msg['Subject']
    if not sender:
        raise PQMException(None, "No From specified")
    if (not subject) or subject=='':
        raise PQMException(sender, "No Subject specified")
    text = None
    sig = None
    if msg.is_multipart():
        parts = msg.get_payload()
        if not len(parts) == 2:
            raise PQMException(sender, 
                        "Multipart message must have exactly two parts")
        if not parts[0].get_content_type() == 'text/plain':
            raise PQMException(sender,
                        "First part of multipart message must be text/plain")
        if not parts[1].get_content_type() == 'application/pgp-signature':
            raise PQMException(sender, 
                        "Second part of multipart message must be"
                        " application/pgp-signature")
        return (sender, 
                subject,
                parts[0].get_payload(),
                parts[1].get_payload())
    else:
        return (sender, subject, msg.get_payload(), None)

class PQMException(Exception):
    """PWM specific exceptions derive from this class."""
    def __init__(self, sender, msg):
        self.sender = sender
        self.msg = msg

    def __str__(self):
        return `self.msg`

def verify_sig(sender, msg, sig, check_replay, logger):
    """Verify the GPG signature on a message."""
    verifier = GPGSigVerifier([keyring], gpgv=gpgv_path)
    try:
        tmp_msgpath = os.path.join(pqm_subdir,'tmp-msg')
        open(tmp_msgpath, 'w').write(msg)
        if sig:
            tmp_sigpath = os.path.join(pqm_subdir,'tmp-sig')
            open(tmp_sigpath, 'w').write(sig)
        else:
            tmp_sigpath = None
        output = verifier.verify(tmp_msgpath, tmp_sigpath)
        os.unlink(tmp_msgpath)
        if sig:
            os.unlink(tmp_sigpath)
    except GPGSigVerificationFailure, e:
        raise PQMException(sender, "Failed to verify signature: %s" % e._value)
    gpgre = re.compile('^\[GNUPG:\] (SIG_ID.+)$')
    sigid = None
    for line in output:
        match = gpgre.match(line)
        if match:
            sigid = match.group(1)
            break
    if not sigid:        
        raise PQMException(sender, "Couldn't determine signature timestamp")
    if check_replay and used_transactions.has_key(sigid):
        logger.error("Replay attack detected, aborting")
        raise PQMException(sender, "Replay attack detected, aborting")
    gpg_key_re = re.compile('^\[GNUPG:\] GOODSIG ([0-9A-F]+) .*<([^>]*)>.*$')
    sig_from = None
    for line in output:
        match = gpg_key_re.match(line)
        if match:
            sig_from_mail = match.group (2)
            sig_from_key = match.group (1)
            break
    logger.info ("garh %s : %s", sig_from_key, sig_from_mail)
    return sigid, sig_from_mail

class GPGSigVerificationFailure(Exception):
    """GPG Verification failed."""
    def __init__(self, value, output):
        self._value = value
        self._output = output
        
    def __str__(self):
        return `self._value`

    def getOutput(self):
        return self._output

class GPGSigVerifier:
    """A class to verify GPG signatures."""
    def __init__(self, keyrings, gpgv=None):
        self._keyrings = keyrings
        if gpgv is None:
            gpgv = '/usr/bin/gpgv'
        self._gpgv = gpgv

    def verify(self, filename, sigfilename=None):
        """Verify the signature on tjhe file filename, detached signatures on
        sigfilename.
        """
        (stdin, stdout) = os.pipe()
        pid = os.fork()
        if pid == 0:
            os.close(stdin)
            os.dup2(stdout, 1)
            os.dup2(stdout, 2)
            args = [self._gpgv]
            args.append('--status-fd=2')
            for keyring in self._keyrings:
                args.append('--keyring')
                args.append(keyring)
            if sigfilename:
                args.append(sigfilename)
            args.append(filename)
            os.execvp(self._gpgv, args)
            os.exit(1)
        os.close(stdout)
        output = os.fdopen(stdin).readlines()
        (pid, status) = os.waitpid(pid, 0)
        if not (status is None 
                or (os.WIFEXITED(status) and os.WEXITSTATUS(status) == 0)):
            if os.WIFEXITED(status):
                msg = ("gpgv exited with error code %d" % 
                        os.WEXITSTATUS(status))
            elif os.WIFSTOPPED(status):
                msg = ("gpgv stopped unexpectedly with signal %d" %
                        os.WSTOPSIG(status))
            elif os.WIFSIGNALED(status):
                msg = "gpgv died with signal %d" % os.WTERMSIG(status)
            raise GPGSigVerificationFailure(msg, output)
        return output


#arch-tag: dc99ede3-0c64-434d-ac84-305c06455a8d
