/* Set aside the deleted and renamed files and directories.  There are
   versions for exact and inexact patching.

  Copyright (C) 2003 Walter Landry
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 dated June, 1991.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA */

#include "boost/filesystem/operations.hpp"
#include "boost/filesystem/fstream.hpp"
#include "boost/date_time/posix_time/posix_time.hpp"
#include "Command_Info.hpp"
#include <list>
#include <functional>
#include <sstream>
#include "Moved_Path.hpp"
#include "Inventory.hpp"
#include "file_attributes.hpp"
#include "check_symlink_hierarchy.hpp"
#include "unsafe_move_inventory_id.hpp"
#include "unsafe_delete_inventory_id.hpp"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
namespace fs=boost::filesystem;
using fs::path;


/* A version for exact patching that doesn't do all of the
   checking. */

void set_aside_paths(const path &deleted_path, const path &source_path,
                     list<Moved_Path> &moved)
{
  /* We need to get a list of all of the directories that are moving,
     and move the bottom ones first.  Then, when moving things back
     for the rename, we move the ones that will be topmost first.  */

  /* Remove all of the non-existent moves from the move list. */  
  moved.remove_if(bind2nd(ptr_fun(Moved_Path_initial_eq_string),""));

  /* Sort based on the name that the file or directory has now.  Sort
     in reverse order, so that the bottom-most files appear before top
     directories. */
  
  moved.sort(not2(ptr_fun(Moved_Path_initial_cmp)));

  /* Now move them, starting from the bottom.  We save the new name in
     the original list.  */

  int renames(0), deletes(0);

  for(list<Moved_Path>::iterator i=moved.begin(); i!=moved.end(); ++i)
    {
      const path old_source=i->initial;
      /* Make sure we aren't moving things from symlinks. */
      check_symlink_hierarchy(source_path,source_path/old_source);

      if(i->destination.empty())
        {
          file_attributes f;
          f.file_path=i->initial;
          f.inventory_id=i->inventory_id;
          f.link=false;

          stringstream deleted;
          deleted << ",,delete-" << deletes;
          i->destination=deleted_path/i->initial;
          i->initial=path(deleted.str());
          ++deletes;

          unsafe_delete_inventory_id(source_path,f,i->is_directory);
        }
      else
        {
          stringstream renamed;
          renamed << ",,rename-" << renames;
          i->initial=path(renamed.str());
          ++renames;
          unsafe_move_inventory_id(source_path,old_source,
                                   deleted_path/i->initial,i->inventory_id);
        }
      if(Command_Info::verbosity>=verbose)
        cout << "Renaming: " << old_source.native_file_string()
             << "\t" << (deleted_path/i->initial).native_file_string() << endl;
      
      rename(source_path/old_source,source_path/deleted_path/i->initial);
    }
}

/* A more generic version that checks for conflicts. */

void set_aside_paths(const path &deleted_path, const path &source_path,
                     list<Moved_Path> &moved,
                     list<Moved_Path> &missing_moves,
                     Inventory &source_inventory)
{
  /* We need to get a list of all of the directories that are moving,
     and move the bottom ones first.  Then, when moving things back
     for the rename, we move the ones that will be topmost first.  */
  
  for(list<Moved_Path>::iterator i=moved.begin(); i!=moved.end(); ++i)
    {
      /* Find the path with the same id. */
      inventory_types inv_type=is_control(i->initial) ? control : source;
      int m=i->is_directory ? arx_dir : arx_file;

      Inventory::iterator
        j(find_if(source_inventory(inv_type,m).begin(),
                  source_inventory(inv_type,m).end(),
                  bind2nd(ptr_fun(inventory_id_eq_string),
                          i->inventory_id)));

      if(j==source_inventory(inv_type,m).end())
        {
          /* Handle non-existent moves.  Ignore any missing deleted files. */
          if(!i->destination.empty())
            missing_moves.push_back(*i);
          i->initial="";
        }
      else
        {
          i->initial=j->file_path.string();

          /* Rewrite the source inventory so that we know where to
             apply patches. */
          j->file_path=i->destination;
        }
    }

  /* Write out all of the missing moves. */

  if(!missing_moves.empty())
    {
      fs::ofstream missing(source_path/(",,missing-renames--"
                                   + to_iso_extended_string
                                   (ptime(second_clock::universal_time()))));
      for(list<Moved_Path>::iterator i=missing_moves.begin();
          i!=missing_moves.end(); ++i)
        {
          missing << i->initial.string() << "\t"
                  << i->destination.string() << "\t"
                  << i->inventory_id << endl;
        }
    }

  /* Do the actual setting aside. */
  set_aside_paths(deleted_path,source_path,moved);
}

