// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "config.h"
#include "signal.h"
#include <sigc++/functors/mem_fun.h>
#include <sigc++/adaptors/bind.h>
#include <assert.h>
#include <algorithm>

#include "stacklist.h"
#include "stack.h"
#include "playerlist.h"
#include "xmlhelper.h"
#include "GameMap.h"
#include "stacktile.h"

Glib::ustring Stacklist::d_tag = "stacklist";

//#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<endl<<flush;}
#define debug(x)

Vector<int> Stacklist::getPosition(guint32 id)
{
    for (Playerlist::iterator pit = Playerlist::getInstance()->begin();
        pit != Playerlist::getInstance()->end(); pit++)
    {
        Stacklist* mylist = (*pit)->getStacklist();
        for (const_iterator it = mylist->begin(); it !=mylist->end(); it++)
            for (Stack::const_iterator sit = (*it)->begin(); sit != (*it)->end(); sit++)
                if ((*sit)->getId() == id)
                    return (*it)->getPos();
    }

    return Vector<int>(-1,-1);
}

//search all player's stacklists to find this stack
bool Stacklist::deleteStack(Stack* s)
{
    for (Playerlist::iterator pit = Playerlist::getInstance()->begin();
        pit != Playerlist::getInstance()->end(); pit++)
    {
        Stacklist* mylist = (*pit)->getStacklist();
        for (const_iterator it = mylist->begin(); it != mylist->end(); it++)
            if ((*it) == s)
                return mylist->flRemove(s);
    }
    return false;
}

bool Stacklist::deleteStack(guint32 id)
{
    for (Playerlist::iterator pit = Playerlist::getInstance()->begin();
        pit != Playerlist::getInstance()->end(); pit++)
    {
        Stacklist* mylist = (*pit)->getStacklist();
        for (const_iterator it = mylist->begin(); it != mylist->end(); it++)
            if ((*it)->getId() == id)
                return mylist->flRemove(*it);
    }
    return false;
}

unsigned int Stacklist::getNoOfStacks()
{
    unsigned int mysize = 0;

    for (Playerlist::iterator pit = Playerlist::getInstance()->begin();
        pit != Playerlist::getInstance()->end(); pit++)
    {
        mysize += (*pit)->getStacklist()->size();
    }

    return mysize;
}

unsigned int Stacklist::getNoOfArmies()
{
    unsigned int mysize = 0;

    for (Playerlist::iterator pit = Playerlist::getInstance()->begin();
        pit != Playerlist::getInstance()->end(); pit++)
    {
        mysize += (*pit)->getStacklist()->countArmies();
    }

    return mysize;
}

unsigned int Stacklist::countArmies() const
{
    unsigned int mysize = 0;

    for (const_iterator it = begin(); it != end(); it++)
      mysize += (*it)->size();

    return mysize;
}

Stacklist::Stacklist()
    :d_activestack(0)
{
}

Stacklist::Stacklist(Stacklist *stacklist)
    :d_activestack(0)
{
    for (iterator it = stacklist->begin(); it != stacklist->end(); it++)
    {
        add(new Stack(**it));
    }
}

Stacklist::Stacklist(XML_Helper* helper)
    :d_activestack(0)
{
    helper->registerTag(Stack::d_tag, sigc::mem_fun((*this), &Stacklist::load));

    load(Stacklist::d_tag, helper);
}

Stacklist::~Stacklist()
{
  //disconnect the signals
  for (ConnectionMap::iterator it = d_connections.begin(); 
       it != d_connections.end(); it++)
    {
      std::list<sigc::connection> list = (*it).second;
      for (std::list<sigc::connection>::iterator lit = list.begin(); lit != list.end(); lit++)
	(*lit).disconnect();
    }
  for (Stacklist::iterator it = begin(); it != end(); it++)
    {
      it = flErase(it);
    }

}

Stack *Stacklist::getStackById(guint32 id) const
{
  IdMap::const_iterator it = d_id.find(id);
  if (it != d_id.end())
    return (*it).second;
  else
    return NULL;
}

Stack *Stacklist::getArmyStackById(guint32 army) const
{
  for (Stacklist::const_iterator i = begin(), e = end(); i != e; ++i)
    if ((*i)->getArmyById(army))
      return *i;
  
  return 0;
}

void Stacklist::flClear()
{
    d_activestack = 0;

    for (iterator it = begin(); it != end(); it++)
    {
        delete (*it);
    }

    clear();
}

Stacklist::iterator Stacklist::flErase(iterator object)
{
    if (d_activestack == (*object))
        d_activestack = 0;
    delete (*object);
    return erase(object);
}

bool Stacklist::flRemove(guint32 id)
{
  Stack *s = getStackById(id);
  if (s == NULL)
    return false;
  return flRemove(s);
}

bool Stacklist::flRemove(Stack* object)
{
  if (object == NULL)
    return false;
    iterator stackit = find(begin(), end(), object);
    if (stackit != end())
    {
        if (d_activestack == object)
            d_activestack = 0;
	assert (object->getId() == (*stackit)->getId());
	deletePositionFromMap(object);
        delete object;
        erase(stackit);
        return true;
    }

    return false;
}

bool Stacklist::save(XML_Helper* helper) const
{
    bool retval = true;

    retval &= helper->openTag(Stacklist::d_tag);
    if (d_activestack)
      retval &= helper->saveData("active", d_activestack->getId());
    else
      retval &= helper->saveData("active", 0);

    //save stacks
    for (const_iterator it = begin(); it != end(); it++)
        retval &= (*it)->save(helper);

    retval &= helper->closeTag();

    return retval;
}

bool Stacklist::load(Glib::ustring tag, XML_Helper* helper)
{
    static guint32 active = 0;
    
    if (tag == Stacklist::d_tag)
    {
        helper->getData(active, "active");
        return true;
    }

    if (tag == Stack::d_tag)
    {
        Stack* s = new Stack(helper);
        if (active != 0 && s->getId() == active)
	  d_activestack = s;

        add(s);
        return true;
    }

    return false;
}

bool Stacklist::addPositionToMap(Stack *stack) const
{
  snewpos.emit(stack, stack->getPos());
  return true;
}

bool Stacklist::deletePositionFromMap(Stack *stack) const
{
  Vector<int> dest = GameMap::getInstance()->findStack(stack->getId());
  if (dest == Vector<int>(-1,-1))
    dest = stack->getPos();
  soldpos.emit(stack, dest);
  return true;
}

void Stacklist::add(Stack *stack)
{
  push_back(stack);
  d_id[stack->getId()] = stack;
  if (stack->getPos() != Vector<int>(-1,-1))
    {
      bool added = addPositionToMap(stack);
      if (!added)
	assert(1 == 0);
      std::list<sigc::connection> conn;
      conn.push_back(stack->smoving.connect
	 (sigc::mem_fun (this, &Stacklist::on_stack_starts_moving)));
      conn.push_back(stack->smoved.connect
	 (sigc::mem_fun (this, &Stacklist::on_stack_stops_moving)));
      conn.push_back(stack->sdying.connect
	 (sigc::mem_fun (this, &Stacklist::on_stack_died)));
      d_connections[stack] = conn;
    }
}

void Stacklist::on_stack_died (Stack *stack)
{
  deletePositionFromMap(stack);
  ConnectionMap::iterator it = d_connections.find(stack);
  if (it != d_connections.end())
    {
      std::list<sigc::connection> list = (*it).second;
      for (std::list<sigc::connection>::iterator lit = list.begin(); lit != list.end(); lit++)
	(*lit).disconnect();
    }
  d_id.erase(d_id.find(stack->getId()));
  return;
}
void Stacklist::on_stack_starts_moving (Stack *stack)
{
  deletePositionFromMap(stack);
  return;
}
void Stacklist::on_stack_stops_moving (Stack *stack)
{
  addPositionToMap(stack);
  return;
}
void Stacklist::setActivestack(Stack* activestack)
{
  d_activestack = activestack;
}

std::list<Vector<int> > Stacklist::getPositions() const
{
  std::list<Vector<int> > points;
  for (const_iterator it = begin(); it != end(); it++)
    {
      if (std::find(points.begin(), points.end(), (*it)->getPos()) == 
          points.end())
        points.push_back((*it)->getPos());
    }
  return points;
}

bool Stacklist::hasGeneral() const
{
  for (const_iterator it = begin(); it != end(); it++)
    {
      if ((*it)->hasGeneral())
        return true;
    }
  return false;
}
// End of file
