//  Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef ARMYSET_H
#define ARMYSET_H

#include <gtkmm.h>
#include <string>
#include <map>
#include <vector>
#include <sigc++/trackable.h>

#include "xmlhelper.h"
#include "armyproto.h"
#include "defs.h"
#include "File.h"
#include "set.h"

using namespace std;

class Armyset: public std::list<ArmyProto *>, public sigc::trackable, public Set
{
    public:

	//! The xml tag of this object in an armyset configuration file.
	static Glib::ustring d_tag; 
	static Glib::ustring file_extension; 

	//! Default constructor.
	/**
	 * Make a new Armyset.
	 *
	 * @param id    The unique Id of this Armyset among all other Armyset
	 *              objects.  Must be more than 0.  
	 * @param name  The name of the Armyset.  Analagous to Armyset::d_name.
	 */
	Armyset(guint32 id, Glib::ustring name);
	//! Loading constructor.
	/**
	 * Load armyset XML entities from armyset configuration files.
	 */
        Armyset(XML_Helper* helper, Glib::ustring directory);

	static Armyset *create(Glib::ustring filename, bool &unsupported);

	//! Destructor.
        ~Armyset();

	/**
	 * @param helper  An opened armyset configuration file.
	 */
	//! Save the Armyset to an Armyset configuration file.
	bool save(XML_Helper* helper) const;
        
        bool save(Glib::ustring filename, Glib::ustring extension) const;

	//! Returns the size of this armyset.
        /** 
         * @return The number of Army prototype objects in the armyset or 0 
	 *         on error (an armyset should never have a size of 0).
         */
        guint32 getSize() const {return size();}

	//! Get the tile size of the Armyset.
	/**
	 * The width and height of the Army graphic images as they appear
	 * on the screen.
	 * Analagous to the armyset.d_tilesize XML entity in the armyset 
	 * configuration file.
	 */
        guint32 getTileSize() const {return d_tilesize;}

        void setTileSize(guint32 tile_size) {d_tilesize = tile_size;}

	//! Get the unique identifier for this armyset.
	/**
	 * Analagous to the armyset.d_id XML entity in the armyset 
	 * configuration file.
	 */
        guint32 getId() const {return d_id;}

	//! Set the unique identifier for this armyset.
        void setId(guint32 id) {d_id = id;}

	//! Returns the name of the armyset.
        /** 
	 * Analagous to the armyset.d_name XML entity in the armyset 
	 * configuration file.
	 *
         * @return The name or an empty string on error.
         */
        Glib::ustring getName() const {return d_name.c_str();}

	//! Set the name of the armyset.
	/**
	 * @note This method is only used in the armyset editor.
	 */
        void setName(Glib::ustring name) {d_name = name;}

	//! Get the copyright holders for this armyset.
	Glib::ustring getCopyright() const {return d_copyright;};

	//! Set the copyright holders on the armyset.
	void setCopyright(Glib::ustring copy) {d_copyright = copy;};

	//! Get the license of this armyset.
	Glib::ustring getLicense() const {return d_license;};

        //! Returns the description of the armyset.
        Glib::ustring getInfo() const {return d_info.c_str();}

	//! Set the license for this armyset.
	void setLicense(Glib::ustring license) {d_license = license;};

	//! Set the description of the armyset.
	/**
	 * @note This method is only used in the armyset editor.
	 */
        void setInfo(Glib::ustring info) {d_info = info;}

	//! Get the base name of the armyset.
	/**
	 * This value does not contain a path (e.g. no slashes).  It is the
	 * name of an armyset directory inside army/.
	 *
	 * @return The basename of the file that the Armyset is held in.
	 */
        Glib::ustring getBaseName() const {return d_basename;}

	//! Set the base name of the file that the armyset is in.
        void setBaseName(Glib::ustring bname) {d_basename = bname;}

	//! Get the image of the raft
	PixMask* getRaftPic() const {return d_raft;}

        Glib::ustring getRaftImageName() const {return d_raft_name;}

	//! Set the image of the raft
	void setRaftImage(PixMask* raft) {d_raft = raft;};

	//! Find an army with a type in this armyset.
	/**
	 * Scan the Army prototype objects in this Armyset and return it.
	 *
	 * @note This is only used for the editor.  Most callers should use 
	 * Armysetlist::getArmy instead.
	 *
	 * @param army_type  The army type id of the Army prototype object
	 *                   to search for in this Armyset.
	 *
	 * @return The Army with the given army type id, or NULL if none
	 *         could be found.
	 */
	ArmyProto * lookupArmyByType(guint32 army_type) const;

	ArmyProto * lookupArmyByName(Glib::ustring name) const;

	//! can this armyset be used within the game?
	bool validate();
	//! get filenames in this armyset, excepting the configuration file.
	void getFilenames(std::list<Glib::ustring> &files);
        //! Delete the armyset's temporary directory.
        void clean_tmp_dir() const;

	void instantiateImages(bool &broken);
	void uninstantiateImages();
	void loadRaftPic(Glib::ustring image_filename, bool &broken);

	Glib::ustring getConfigurationFile() const;
	static std::list<Glib::ustring> scanUserCollection();
	static std::list<Glib::ustring> scanSystemCollection();

        Glib::ustring getFileFromConfigurationFile(Glib::ustring file);
        bool replaceFileInConfigurationFile(Glib::ustring file, Glib::ustring new_file);
        //! Load the armyset again.
        void reload(bool &broken);
        guint32 calculate_preferred_tile_size() const;

    private:

        //! Callback function for the army tag (see XML_Helper)
        bool loadArmyProto(Glib::ustring tag, XML_Helper* helper);
        
	//! The unique Id of this armyset.
	/**
	 * This Id is unique among all other armysets.
	 * It is analgous to armyset.d_id in the armyset configuration files.
	 */
        guint32 d_id;

	//! The name of the Armyset.
	/**
	 * This value appears in game configuration dialogs.
	 * It is analgous to armyset.d_name in the armyset configuration files.
	 */
        Glib::ustring d_name;

	//! The armyset has these copyright holders.
	Glib::ustring d_copyright;

	//! The license of the armyset.
	Glib::ustring d_license;

	//! The basename of the Armyset.
	/**
	 * This is the base name of the file that the Armyset files are
	 * residing in.  It does not contain a path (e.g. no slashes).
	 * Armyset files sit in the army/ directory.
	 */
        Glib::ustring d_basename;

	//! The description of the Armyset.
	/**
	 * Equates to the armyset.d_info XML entity in the armyset
	 * configuration file.
	 * This value is not used.
	 */
        Glib::ustring d_info;

	//! The size of each army tile as rendered in the game.
	/**
	 * The tile size represents the height and width in pixels of the 
	 * army picture.
	 * The actual army picture holds the unshaded image, and then the
	 * mask portion of the image to it's right.
	 * The tilesize is the height of the army image file, and it is also
	 * precisely equal to half of the image's width.
	 */
	guint32 d_tilesize;

	//! The picture of a raft.
	PixMask* d_raft;

	//! The name of the file that holds the picture of the raft
	Glib::ustring d_raft_name;

};

#endif // ARMYSET_H

