// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "config.h"
#include <iostream>
#include <iomanip>
#include <fstream>
#include <stdlib.h>
#include <errno.h>
#include <sigc++/functors/mem_fun.h>
#include <string.h>

#include "ucompose.hpp"
#include "GameScenario.h"
#include "playerlist.h"
#include "File.h"
#include "armysetlist.h"
#include "stacklist.h"
#include "stack.h"
#include "GameMap.h"
#include "player.h"
#include "Configuration.h"
#include "real_player.h"
#include "ai_dummy.h"
#include "counter.h"
#include "army.h"
#include "player.h"
#include "history.h"
#include "xmlhelper.h"
#include "tarhelper.h"
#include "stacktile.h"
#include "GraphicsCache.h"
#include "raftlist.h"
#include "reinforcement-point-list.h"
#include "soundset.h"
#include "snd.h"

Glib::ustring GameScenario::d_tag = "scenario";
Glib::ustring GameScenario::d_top_tag = "game";

#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<endl<<flush;}
//#define debug(x)

GameScenario::GameScenario (Glib::ustring name, Glib::ustring comment, GameScenario::PlayMode playmode):d_name (name), d_comment (comment), d_copyright (""), d_license (""),
d_playmode (playmode), loaded_game_filename ("")
{
  Armysetlist::getInstance ();

  if (fl_counter == 0)
    fl_counter = new FL_Counter ();

  setNewRandomId ();
}

//savegame has an absolute path
GameScenario::GameScenario (Glib::ustring savegame, bool & broken):d_playmode (GameScenario::HOTSEAT), loaded_game_filename ("")
{
  Tar_Helper
  t (savegame, std::ios::in, broken);
  if (broken == false)
    {
      loaded_game_filename = savegame;
      loadArmysets (&t);
      loadSoundset (&t);
      std::list < Glib::ustring > ext;
      ext.push_back (MAP_EXT);
      ext.push_back (SAVE_EXT);
      Glib::ustring filename = t.getFirstFile (ext, broken);
      XML_Helper helper (filename, std::ios::in);
      broken = loadWithHelper (helper);
      File::erase (filename);
      filename = t.getFile (d_map_image_name + ".png", broken);
      if (!broken)
        d_map_image = PixMask::create(filename, broken);
      File::erase (filename);
      filename = t.getFile (d_wall_image_name + ".png", broken);
      if (!broken)
        d_wall_image = PixMask::create(filename, broken);
      File::erase (filename);
      helper.close ();
      t.Close ();
      if (broken)
	cleanup ();
    }
  else
    {
      t.Close ();
      cleanup ();
    }
}

bool GameScenario::loadSoundset (Tar_Helper *t)
{
  bool broken = false;
  Glib::ustring music_file = t->getFile("music.xml", broken);

  if (broken == false)
    {
      XML_Helper helper(music_file, std::ios::in);
      Soundset *soundset = new Soundset(&helper);
      if (!helper.parseXML())
        {
          std::cerr<< "Error loading music descriptions\n";
          return false;
        }
      //now we extract each piece.
      std::map<Glib::ustring,MusicItem*> pieces = soundset->getPieces();
      std::map<Glib::ustring,MusicItem*> new_pieces;
      for (std::map<Glib::ustring,MusicItem*>::iterator i = pieces.begin();
           i != pieces.end(); i++)
        {
          Glib::ustring sound_file = t->getFile((*i).second->file, broken);
          if (broken)
            break;
          MusicItem *new_piece = new MusicItem();
          *new_piece = *(*i).second;
          new_piece->file = sound_file;
          new_pieces[(*i).first] = new_piece;
        }
      soundset->setPieces(new_pieces);
      if (!broken)
        Snd::getInstance()->loadSoundset(soundset);
    }
  return broken;
}

bool
GameScenario::loadArmysets (Tar_Helper * t)
{
  bool broken = false;
  std::list < Glib::ustring > armysets;
  armysets = t->getFilenamesWithExtension (Armyset::file_extension);
  for (std::list < Glib::ustring >::iterator it = armysets.begin ();
       it != armysets.end (); it++)
    {
      Armyset *armyset = Armysetlist::getInstance ()->import (t, *it, broken);
      if (armyset && !broken)
	Armysetlist::getInstance ()->getArmyset (armyset->getId ())->
	  instantiateImages (broken);
    }
  return !broken;
}

GameScenario::GameScenario (XML_Helper & helper, bool & broken):d_playmode (GameScenario::HOTSEAT),
loaded_game_filename
("")
{
  broken = loadWithHelper (helper);
}

bool
GameScenario::loadWithHelper (XML_Helper & helper)
{
  Armysetlist::getInstance ();

  bool broken = false;

  helper.registerTag (d_top_tag, sigc::mem_fun (this, &GameScenario::load));
  helper.registerTag (d_tag, sigc::mem_fun (this, &GameScenario::load));
  helper.registerTag (Playerlist::d_tag,
		      sigc::mem_fun (this, &GameScenario::load));
  helper.registerTag (Raftlist::d_tag,
		      sigc::mem_fun (this, &GameScenario::load));
  helper.registerTag (ReinforcementPointlist::d_tag,
		      sigc::mem_fun (this, &GameScenario::load));
  helper.registerTag (GameMap::d_tag,
		      sigc::mem_fun (this, &GameScenario::load));
  helper.registerTag (FL_Counter::d_tag,
		      sigc::mem_fun (this, &GameScenario::load));

  if (!helper.parseXML ())
    broken = true;

  if (!broken)
    {
      GameMap::getInstance ()->updateStackPositions ();
    }

  return broken;
}

GameScenario::~GameScenario ()
{
  if (d_map_image)
    delete d_map_image;
  if (d_wall_image)
    delete d_wall_image;
  cleanup ();
  clean_tmp_dir ();
}

Glib::ustring GameScenario::getName () const
{
  return d_name;
}

Glib::ustring GameScenario::getComment () const
{
  return d_comment;
}

bool
GameScenario::saveGame (Glib::ustring filename, Glib::ustring extension) const
{
  bool retval = true;
  string goodfilename = File::add_ext_if_necessary (filename, extension);

  std::string tmpfile = "army.XXXX";
  int fd = Glib::file_open_tmp (tmpfile, "army.XXXX");
  close (fd);
  XML_Helper helper (tmpfile, std::ios::out);
  retval &= saveWithHelper (helper);
  helper.close ();

  if (retval == false)
    return false;

  bool broken = false;
  Tar_Helper t (goodfilename, std::ios::out, broken);
  if (broken == true)
    return false;

  t.saveFile (tmpfile, File::get_basename (goodfilename, true));
  File::erase (tmpfile);
  std::list < Glib::ustring > files;


  Playerlist *plist = Playerlist::getInstance ();
  std::list < guint32 > armysets;
  for (Playerlist::iterator it = plist->begin (); it != plist->end (); it++)
    {
      guint32 armyset = (*it)->getArmyset ();
      if (std::find (armysets.begin (), armysets.end (), armyset) ==
	  armysets.end ())
	armysets.push_back (armyset);
    }
  for (std::list < guint32 >::iterator it = armysets.begin ();
       it != armysets.end (); it++)
    {
      files.clear ();
      Armyset *as = Armysetlist::getInstance ()->getArmyset (*it);
      t.saveFile (as->getConfigurationFile ());
      as->getFilenames (files);
      for (std::list < Glib::ustring >::iterator i = files.begin ();
	   i != files.end (); i++)
	t.saveFile (as->getFile (*i));
    }

  Glib::ustring map_image_file = "/tmp/" + d_map_image_name + ".png";
  d_map_image->to_pixbuf()->save(map_image_file, "png");
  sync();
  t.saveFile(map_image_file);
  Glib::ustring wall_image_file = "/tmp/" + d_wall_image_name + ".png";
  d_wall_image->to_pixbuf()->save(wall_image_file, "png");
  sync();
  t.saveFile(wall_image_file);
  File::erase(map_image_file);
  File::erase(wall_image_file);

  Soundset *soundset = Snd::getInstance()->getSoundset();
  XML_Helper soundhelper (tmpfile + "-music.xml", std::ios::out);
  retval &= soundset->save(&soundhelper);
  soundhelper.close ();
  t.saveFile (tmpfile +"-music.xml", "music.xml");
  File::erase(tmpfile +"-music.xml");
      
  files.clear();
  soundset->getFilenames (files);
  for (std::list < Glib::ustring >::iterator i = files.begin (); 
       i != files.end (); i++)
    t.saveFile (*i);
  t.Close ();
  return true;

}

bool
GameScenario::saveWithHelper (XML_Helper & helper) const
{
  bool retval = true;

  retval &= helper.begin (ARMY_SAVEGAME_VERSION);
  retval &= helper.openTag (d_top_tag);

  retval &= fl_counter->save (&helper);
  retval &= Playerlist::getInstance ()->save (&helper);
  retval &= GameMap::getInstance ()->save (&helper);
  retval &= Raftlist::getInstance()->save(&helper);

  //save the private GameScenario data last due to dependencies
  retval &= helper.openTag (GameScenario::d_tag);
  retval &= helper.saveData ("id", d_id);
  retval &= helper.saveData ("name", d_name);
  retval &= helper.saveData ("comment", d_comment);
  retval &= helper.saveData ("map_image", d_map_image_name);
  retval &= helper.saveData ("wall_image", d_wall_image_name);
  retval &= helper.saveData ("copyright", d_copyright);
  retval &= helper.saveData ("license", d_license);
  retval &= helper.saveData ("turn", s_round);
  Glib::ustring playmode_str =
    playModeToString (GameScenario::PlayMode (d_playmode));
  retval &= helper.saveData ("playmode", playmode_str);

  retval &= helper.closeTag ();

  retval &= helper.closeTag ();

  return retval;
}

bool
GameScenario::load (Glib::ustring tag, XML_Helper * helper)
{
  if (tag == d_top_tag)
    {
      if (helper->getVersion () != ARMY_SAVEGAME_VERSION)
	{
          std::cerr << "savefile has wrong version, we want ";
	  std::cerr << ARMY_SAVEGAME_VERSION << ",\n";
          std::cerr << "savefile offers " << helper->getVersion () << ".\n";
	  return false;
	}
      return true;
    }
  if (tag == GameScenario::d_tag)
    {
      helper->getData (d_id, "id");
      helper->getData (d_name, "name");
      helper->getData (d_comment, "comment");
      helper->getData (d_map_image_name, "map_image"); //load the image after
      helper->getData (d_wall_image_name, "wall_image"); //load the image after
      helper->getData (d_copyright, "copyright");
      helper->getData (d_license, "license");
      helper->getData (s_round, "turn");
      Glib::ustring playmode_str;
      helper->getData (playmode_str, "playmode");
      d_playmode = GameScenario::playModeFromString (playmode_str);

      return true;
    }

  if (tag == FL_Counter::d_tag)
    {
      fl_counter = new FL_Counter (helper);
      return true;
    }

  if (tag == Playerlist::d_tag)
    {
      Playerlist::getInstance (helper);
      return true;
    }

  if (tag == Raftlist::d_tag)
    {
      Raftlist::getInstance (helper);
      return true;
    }

  if (tag == ReinforcementPointlist::d_tag)
    {
      ReinforcementPointlist::getInstance (helper);
      return true;
    }

  if (tag == GameMap::d_tag)
    {
      GameMap::getInstance (helper);
      return true;
    }

  return false;
}

void
GameScenario::nextRound ()
{
  s_round++;
}

Glib::ustring GameScenario::playModeToString (const GameScenario::PlayMode mode)
{
  switch (mode)
    {
    case GameScenario::HOTSEAT:
      return "GameScenario::HOTSEAT";
      break;
    }
  return "GameScenario::HOTSEAT";
}

GameScenario::PlayMode GameScenario::playModeFromString (const Glib::ustring str)
{
  if (str.size () > 0 && isdigit (str.c_str ()[0]))
    return GameScenario::PlayMode (atoi (str.c_str ()));
  if (str == "GameScenario::HOTSEAT")
    return GameScenario::HOTSEAT;
  return GameScenario::HOTSEAT;
}

void
GameScenario::setNewRandomId ()
{
  d_id = generate_guid ();
}

bool
GameScenario::validate (std::list < Glib::ustring > &errors,
			std::list < Glib::ustring > &warnings)
{
  Glib::ustring s;
  Playerlist *pl = Playerlist::getInstance ();
  guint32 num = pl->countPlayersAlive ();
  if (num < 2)
    errors.push_back ("There must be at least 2 players in the scenario.");

  if (errors.size () == 0)
    return true;
  return false;
}

void
GameScenario::initialize (GameParameters g)
{
  Playerlist::getInstance ()->clearAllActions ();
  int n = rand() % Playerlist::getInstance()->size();
  for (int i = 0; i < n; i++)
    Playerlist::getInstance()->nextPlayer();
  nextRound ();
  GameMap::getInstance ()->updateStackPositions ();
}

void
GameScenario::clean_tmp_dir () const
{
  if (loaded_game_filename != "")
    Tar_Helper::clean_tmp_dir (loaded_game_filename);
}

Glib::ustring GameScenario::generate_guid ()
{
  Glib::TimeVal now;
  now.assign_current_time ();
  double
    n = now.as_double ();
  n -= (int) n;
  n *= 100000000;
  Glib::Rand r (n);
  char
    buf[40];
  //this is a very poor guid generator.
  snprintf (buf, sizeof (buf),
	    "{%08X-%04X-%04X-%02X%02X-%02X%02X%02X%02X%02X%02X}",
	    r.get_int (), rand () % 4096, r.get_int_range (0, 4096),
	    r.get_int_range (0, 256), r.get_int_range (0, 256),
	    r.get_int_range (0, 256) % 256, r.get_int_range (0, 256),
	    r.get_int_range (0, 256), r.get_int_range (0, 256),
	    r.get_int_range (0, 256), r.get_int_range (0, 256));

  return Glib::ustring (buf);
}

void
GameScenario::cleanup ()
{
  Playerlist::deleteInstance ();
  GameMap::deleteInstance ();
  if (fl_counter)
    {
      delete fl_counter;
      fl_counter = 0;
    }
  GameScenarioOptions::s_round = 0;
}
