// Copyright (C) 2011, 2014 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef GAMEMAP_H
#define GAMEMAP_H

#include <sigc++/trackable.h>

#include <vector>

#include <gtkmm.h>
#include "vector.h"
#include "rectangle.h"
#include "maptile.h"
#include "stack.h"

class XML_Helper;
class Armyset;
class ArmyProto;
class Raft;

class GameMap: public sigc::trackable
{
    public:
	//! The xml tag of this object in a saved-game file.
	static Glib::ustring d_tag; 

        /** Singleton function to get the GameMap instance
          * 
          * @return singleton instance
          */
        static GameMap* getInstance();

        /** Creates a new singleton instance from a savegame file
          * 
          * @param helper           see XML_Helper for an explanation
          *
          * \note This function deletes an existing instance!
          */
        static GameMap* getInstance(XML_Helper* helper);

        //! Explicitely deletes the singleton instance
        static void deleteInstance();

        //! Set the width of the game map
        static void setWidth(int width){s_width = width;}
 
        //! Set the height of the game map
        static void setHeight(int height){s_height = height;}

        
        //! Returns the width of the map
        static int getWidth() {return s_width;}
 
        //! Returns the height of the map
        static int getHeight() {return s_height;}

	static Vector<int> get_dim() { return Vector<int>(s_width, s_height); }
	
	static Rectangle get_boundary()
	    { return Rectangle(0, 0, s_width, s_height); }

	//! Sets the tile object at position (x, y)
	void setTile(int x, int y, Maptile *tile);
        
        //! Alternative setting
        void setTile(Vector<int> p, Maptile *t) {return setTile(p.x, p.y, t);}

        static Raft* getRaft(Vector<int> pos);
	static Stack* getStack(Vector<int> pos);
	static StackTile* getStacks(Vector<int> pos);
	static bool canAddArmy(Vector<int> pos);
	static bool canAddArmies(Vector<int> dest, guint32 stackSize);
	static Stack* getFriendlyStack(Vector<int> pos);
	static std::list<Stack*> getFriendlyStacks(Vector<int> pos, Player *player = NULL);
	static Stack* getEnemyStack(Vector<int> pos);
	static std::list<Stack*> getEnemyStacks(std::list<Vector<int> > posns);
	static std::list<Stack*> getEnemyStacks(Vector<int> pos, Player *player = NULL);
	static std::list<Stack*> getNearbyFriendlyStacks(Vector<int> pos, int dist);
	static std::list<Stack*> getNearbyEnemyStacks(Vector<int> pos, int dist);

        static std::list<Vector<int> > getNearbyPoints(Vector<int> pos, int dist);
	static guint32 countArmyUnits(Vector<int> pos);

        static std::list<Vector<int> > getPointsInLine(Vector<int> s, Vector<int> d);
        static Vector<int> getReinforcementPoint();
        static bool crosses_over_enemy_combatant(Stack *src, Stack *dest);
        static bool crosses_over_water(Stack *src, Stack *dest);
        static bool crosses_over_brick(Stack *src, Stack *dest);

        //! Get the tile object at position (x,y)
        Maptile* getTile(int x, int y) const;

        //! Alternative access
        Maptile* getTile(Vector<int> p) const {return getTile(p.x, p.y);}

	//! Add an army to this location on the map
        Stack* addArmy(Vector<int> pos, Army *a);
        Stack* addArmyAtPos(Vector<int> pos, Army *a);
        void addArmies(const ArmyProto *a, guint32 num_allies, Vector<int> pos);

	//! go find the player's stack, the slow way.
	Vector<int> findStack(guint32 id);

        /** Save the contents of the map
          * 
          * @param helper           see XML_Helper for more information
          * @return true if saving went well, false otherwise
          */
        bool save(XML_Helper* helper) const;

	//! figure out where a non-flying unit can't go
        void calculateBlockedAvenues();
	void calculateBlockedAvenue(int i, int j);
	void updateStackPositions();
        void clearStackPositions();

	bool canPutStack(guint32 size, Player *p, Vector<int> to);
	bool moveStack(Stack *stack, Vector<int> to);

	bool putStack(Stack *s);
	void removeStack(Stack *s);

        static Vector<int> getCenterOfMap();
        std::list<Vector<int> > getBricks() const;

    protected:
        GameMap();

        //! Load the map using the given XML_Helper
        GameMap(XML_Helper* helper);

        ~GameMap();
        
    private:
        //! Callback for item loading used during loading.
        bool isBlockedAvenue(int x, int y, int destx, int desty);

	static std::list<Stack*> getNearbyStacks(Vector<int> pos, int dist, bool friendly);

	static bool offmap(int x, int y);

        // Data
        static GameMap* s_instance;
        static int s_width;
        static int s_height;

        Maptile** d_map;
};

#endif

// End of file
