// Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "fight.h"
#include <algorithm>
#include <iomanip>
#include <assert.h>
#include <stdlib.h>     // for random numbers
#include <math.h>       // for has_hit()
#include <sstream>
#include "army.h"
#include "stacklist.h"
#include "player.h"
#include "playerlist.h"
#include "GameMap.h"
#include "stack.h"

//#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<endl<<flush;}
#define debug(x)

// Helper class; the single units participating in the fight are saved with
// additional information. This should be a struct, but I don't know how to
// forward the declaration properly.
//! A particpant in a Fight.
class Fighter
{
    public:
        Fighter(Army* a, Vector<int> p);
        
        Army* army;
        Vector<int> pos;       // location on the map (needed to calculate boni)
	//! needed for sorting
        //bool operator() ( const Fighter* f1, const Fighter* f2 );
        int died;

};

Fighter::Fighter(Army* a, Vector<int> p)
    :army(a), pos(p)
{
}


void Fight::loadArmies(std::list<Stack*> stacks, std::vector<Army*> &armies)
{
  std::list<Stack*>::iterator it;
  if (stacks.empty())
    return;
  for (it = stacks.begin(); it != stacks.end(); it++)
    for (Stack::iterator sit = (*it)->begin(); sit != (*it)->end(); sit++)
      armies.push_back((*sit));
  return;
}
Fight::Fight(Stack* attacker, Stack* defender)
    : d_turn(0), d_result(DRAW)
{
    std::list<Stack*>::iterator it;
    
    //setup fighters
    if (defender)
    d_defenders.push_back(defender);
    if (attacker)
    d_attackers.push_back(attacker);
    it = d_defenders.begin();
    std::vector<Army*> def;
    loadArmies (d_defenders, def);
    for (std::vector<Army*>::iterator ait = def.begin(); ait != def.end(); ait++)
      {
	Fighter* f = new Fighter((*ait), (*it)->getPos());
	d_def_close.push_back(f);
      }

    it = d_attackers.begin();
    std::vector<Army*> att;
    loadArmies (d_attackers, att);
    for (std::vector<Army*>::iterator ait = att.begin(); ait != att.end(); ait++)
      {
	Fighter* f = new Fighter((*ait), (*it)->getPos());
	d_att_close.push_back(f);
      }

}

Fight::Fight(std::list<Stack*> attackers, std::list<Stack*> defenders,
             std::list<FightItem> history)
{
  d_attackers = attackers;
  d_defenders = defenders;
  d_actions = history;

}

Fight::~Fight()
{
      
  d_attackers.clear();
  d_defenders.clear();

  // clear all fighter items in all lists
  while (!d_att_close.empty())
    {
      delete (*d_att_close.begin());
      d_att_close.erase(d_att_close.begin());
    }

  while (!d_def_close.empty())
    {
      delete (*d_def_close.begin());
      d_def_close.erase(d_def_close.begin());
    }
}

void Fight::battle(int die_value, double water_penalty)
{
  // first, fight until the fight is over
  for (d_turn = 0; doRound(die_value, water_penalty); d_turn++);

  // Now we have to set the fight result.

  // First, look if the attacker died; the attacking stack is the first
  // one in the list
  bool survivor = false;
  if (d_att_close.empty() == false)
    survivor = true;

  if (!survivor)
    d_result = DEFENDER_WON;
  else
    {
      // Now look if the defender died; also the first in the list
      survivor = false;
      if (d_def_close.empty() == false)
        survivor = true;

      if (!survivor)
	d_result = ATTACKER_WON;
    }

}

Army *findArmyById(const std::list<Stack *> &l, guint32 id)
{
  for (std::list<Stack *>::const_iterator i = l.begin(), end = l.end();
       i != end; ++i) {
    Army *a = (*i)->getArmyById(id);
    if (a)
      return a;
  }

  return 0;
}

bool Fight::doRound(int die_value, double attacking_over_water_penalty)
{
  if (MAX_ROUNDS && d_turn >= MAX_ROUNDS)
    return false;

  debug ("Fight round #" <<d_turn);

  //fight the first one in attackers with the first one in defenders
  std::list<Fighter*>::iterator ffit = d_att_close.begin();
  std::list<Fighter*>::iterator efit = d_def_close.begin();

  //have the attacker and defender try to hit each other
  fightArmies(*ffit, *efit, die_value, attacking_over_water_penalty);

  if (*efit && (*efit)->died)
    remove((*efit));

  if (*ffit && (*ffit)->died)
    remove((*ffit));

  if (d_def_close.empty() || d_att_close.empty())
    return false;

  return true;
}

void Fight::fightArmies(Fighter* attacker, Fighter* defender, int die_value, double attacking_over_water_penalty)
{
  if (!attacker || !defender)
    return;

  Army *a = attacker->army;
  Army *d = defender->army;

  guint sides = 6;

  FightItem item;
  item.turn = d_turn;
  item.attacker_roll = die_value - 1;
  item.defender_roll = rand() % sides;
  double attacker_power = getPower(a, item.attacker_roll + 1) - 
    attacking_over_water_penalty;
  if (attacker_power < 0)
    attacker_power = 0;
  double defender_power = getPower(d, item.defender_roll + 1);
  std::ostringstream astream;
  astream << std::setprecision(2) << attacker_power;
  item.attacker_power = atof(astream.str().c_str());
  std::ostringstream dstream;
  dstream << std::setprecision(2) << defender_power;
  item.defender_power = atof(dstream.str().c_str());

  item.defender_id = d->getId();
  item.attacker_id = a->getId();
  d_actions.push_back(item);
  defender->died = 0;
  attacker->died = 0;
  if (item.attacker_power > item.defender_power)
    defender->died = 1;
  if (item.defender_power > item.attacker_power)
    attacker->died = 1;
}

void Fight::remove(Fighter* f)
{
  std::list<Fighter*>::iterator it;

  // is the fighter in the attacker lists?
  for (it = d_att_close.begin(); it != d_att_close.end(); it++)
    if ((*it) == f)
      {
	d_att_close.erase(it);
	delete f;
	return;
      }

  // or in the defender lists?
  for (it = d_def_close.begin(); it != d_def_close.end(); it++)
    if ((*it) == f)
      {
	d_def_close.erase(it);
	delete f;
	return;
      }

  // if the fighter wa sin no list, we are rather careful and don't do anything
  debug("Fight: fighter without list!")
}
