// Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef ACTION_H
#define ACTION_H

#include <gtkmm.h>
#include <string>
#include "vector.h"
#include <sigc++/trackable.h>

#include "fight.h"
#include "army.h"
#include "player.h"
#include "Ownable.h"
#include "defs.h"

class Stack;
class XML_Helper;

//! A temporary record of an event during gameplay.
/** 
 * The purpose of the action classes is to keep track of what a player has
 * done. This information can be sent over the network, so that a networked 
 * player then just has to decode and repeat the remote player's actions so
 * that the game state is synchronised.
 * 
 * Each action item is derived from the abstract Action class. It must
 * contain three functions; A loading constructor (which takes an 
 * XML_Helper parameter), a save function which saves the data to the
 * XML file, and a fillData function which takes some parameters and 
 * stores the pertinent data about what happened.
 *
 * Each Player has an Actionlist to which these actions belong.
 */

class Action
{
    public:
	//! The xml tag of this object in a saved-game file.
	static std::string d_tag; 

	//! An Action can be one of the following kinds.
        enum Type {
	        /** A stack has moved. */
                STACK_MOVE = 1,
		/** A stack is fighting a city or another stack. */
                STACK_FIGHT = 3,
		/** A player has surrendered. */
		RESIGN = 23,
                END_TURN = 30,
                PLAYER_RENAME = 33,
		INIT_TURN = 35,
                KILL_PLAYER = 42,
                STACK_SELECT = 47,
                STACK_DESELECT = 48,
                REINFORCEMENTS = 49,
                DIE_SELECTED = 50

        };
	static std::string actionTypeToString(Action::Type type);
	static Action::Type actionTypeFromString(std::string str);

	//! Default constructor.
        Action(Type type);

	//! Copy constructor (shallow).
	Action(const Action &action);

	//! Loading constructor.
        Action(XML_Helper *helper);

	//! Destructor.
        virtual ~Action();

        //! Returns debug information. Needs to be overwritten by derivatives.
        virtual std::string dump() const = 0;

        //! Save function. See XML_Helper for information about saving.
        bool save(XML_Helper* helper) const;
	bool saveContents(XML_Helper* helper) const;
        
        /** 
	 * static load function (see XML_Helper)
         * 
         * Whenever an action item is loaded, this function is called. It
         * examines the stored id and calls the constructor of the appropriate
         * action class.
         *
         * @param helper       the XML_Helper instance for the savegame
         */
	//! Load the action from an opened saved-game file.
        static Action* handle_load(XML_Helper* helper);

        //! Make a new action from an existing one.
        static Action* copy(const Action* a);

        //! Returns the Action::Type for this action.
        Type getType() const {return d_type;}

    protected:
        virtual bool doSave(XML_Helper* helper) const = 0;
        
        Type d_type;
};

//-----------------------------------------------------------------------------

//! A temporary record of a Stack moving.
/**
 * The purpose of the Action_Move class is to record when a stack has
 * moved to a new position on the map.
 */
class Action_Move : public Action
{
    public:
	//! Make a new move action.
        Action_Move();
	//! Copy constructor
        Action_Move(const Action_Move &action);
	//! Load a new move action from an opened saved-game file.
        Action_Move(XML_Helper* helper);
	//! Destroy a move action.
        ~Action_Move();

	//! Return some debug information about this action.
        std::string dump() const;

	//! Save this move action to an opened saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

	//! Populate the move action with the stack and it's new position.
        bool fillData(Stack* s, Vector<int> dest);
    
	guint32 getStackId() const {return d_stack;};
	Vector<int> getEndingPosition() const {return d_dest;};
	Vector<int> getPositionDelta() const {return d_delta;};

        private:
        guint32 d_stack;
        Vector<int> d_dest;
	Vector<int> d_delta;
};

//-----------------------------------------------------------------------------

//! A temporary record of a fight between opposing Stack objects.
/**
 * The purpose of the Action_Fight class is to record the results of a
 * fight between two Players.
 */
class Action_Fight : public Action, public sigc::trackable
{
    public:
	//! Make a new fight action.
        Action_Fight();
	//! Copy constructor
	Action_Fight(const Action_Fight &action);
	//! Load a new fight action from an opened saved-game file.
        Action_Fight(XML_Helper* helper);
	//! Destroy a fight action.
        ~Action_Fight();

	//! Return some debug information about this action.
        std::string dump() const;

	//! Save this fight action to an opened saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

	/**
	 * Populate the action with the Fight.  Please note that the
	 * Fight must have already been faught.
	 */
	//! Fill the action with pertinent data.
        bool fillData(const Fight* f);


	std::list<FightItem> getBattleHistory() const {return d_history;};
	std::list<guint32> getAttackerStackIds() const {return d_attackers;};
	std::list<guint32> getDefenderStackIds() const {return d_defenders;};

        private:
        
        std::list<FightItem> d_history;
        std::list<guint32> d_attackers;
        std::list<guint32> d_defenders;

        bool is_army_id_in_stacks(guint32 id, std::list<guint32> stack_ids) const;
        bool stack_ids_to_stacks(std::list<guint32> stack_ids, std::list<Stack*> &stacks, guint32 &stack_id) const;

        bool loadItem(std::string tag, XML_Helper* helper);

};

//-----------------------------------------------------------------------------

class Action_EndTurn: public Action
{
    public:
	//! Make a new end turn action.
        Action_EndTurn();
	//! Copy constructor
	Action_EndTurn(const Action_EndTurn &action);
	//! Load a new end turn action from an opened saved-game file.
        Action_EndTurn(XML_Helper* helper);
	//! Destroy a end turn action.
        ~Action_EndTurn();

	//! Return some debug information about this action.
        std::string dump() const;

	//! Save this action to an opened saved-game file.
        virtual bool doSave(XML_Helper* helper) const;
};

//-----------------------------------------------------------------------------

class Action_RenamePlayer: public Action
{
    public:
	//! Make a new rename player action
        Action_RenamePlayer();
	//! Copy constructor
	Action_RenamePlayer(const Action_RenamePlayer &action);
	//! Load a new rename player action from an opened saved-game file.
        Action_RenamePlayer(XML_Helper* helper);
	//! Destroy a rename player action.
        ~Action_RenamePlayer();

	//! Return some debug information about this action.
        std::string dump() const;

	//! Save this city occupied action to an opened saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

	//! Populate the action.
        bool fillData(std::string name);
    
	std::string getName() const {return d_name;};

        private:
	std::string d_name;
};

//-----------------------------------------------------------------------------

class Action_InitTurn: public Action
{
    public:
	//! Make a new initialize turn action.
        Action_InitTurn();
	//! Copy constructor
	Action_InitTurn(const Action_InitTurn &action);
	//! Load a new initialize turn action from an opened saved-game file.
        Action_InitTurn(XML_Helper* helper);
	//! Destroy a initialize turn action.
        ~Action_InitTurn();

        bool fillData(int die1, int die2, int die3, int die4);
	//! Return some debug information about this action.
        std::string dump() const;

        std::vector<guint32> getDice() const {return d_dice;}
	//! Save this action to an opened saved-game file.
        virtual bool doSave(XML_Helper* helper) const;
    private:
        std::vector<guint32> d_dice;
};

//-----------------------------------------------------------------------------
//! A temporary record of a player dying.
/**
 * The purpose of the Action_Kill class is to record when a player has been
 * vanquished by foes.
 */
class Action_Kill: public Action
{
    public:
	//! Make a kill action.
        Action_Kill();
	//! Copy constructor
	Action_Kill(const Action_Kill &action);
	//! Load a new kill action from a saved-game file.
        Action_Kill(XML_Helper* helper);
	//! Destroy a kill action.
        ~Action_Kill();

	//! Return some debug information about this action.
        std::string dump() const;

	//! Save this kill action to a saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

	//! The Action_Kill doesn't take any info to fill.
        bool fillData();
};

//-----------------------------------------------------------------------------

//! A temporary record of a player selecting a stack to work with.
/**
 * The purpose of the Action_SelectStack class is to record when a player
 * grabs a stack to work with.  Only one stack can be selected at a time.
 */
class Action_SelectStack: public Action
{
    public:
	//! Make a select stack action.
        Action_SelectStack();
	//! Copy constructor
	Action_SelectStack (const Action_SelectStack &action);
	//! Load a new select stack action from a saved-game file.
        Action_SelectStack (XML_Helper* helper);
	//! Destroy a select stack action.
        ~Action_SelectStack();

	//! Return some debug information about this action.
        std::string dump() const;

	//! Save this select stack action to a saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

	//! Supply the stack that is being selected
        bool fillData(Stack *s);
	
        guint32 getStackId() const {return d_stack_id;};
    private:
        guint32 d_stack_id;
};

//-----------------------------------------------------------------------------

//! A temporary record of a player deselecting a stack.
/**
 * The purpose of the Action_DeselectStack class is to record when a player
 * removes focus from any and all stacks.
 */
class Action_DeselectStack: public Action
{
    public:
	//! Make a deselect stack action.
        Action_DeselectStack();
	//! Copy constructor
	Action_DeselectStack (const Action_DeselectStack &action);
	//! Load a new deselect stack action from a saved-game file.
        Action_DeselectStack (XML_Helper* helper);
	//! Destroy a deselect stack action.
        ~Action_DeselectStack();

	//! Return some debug information about this action.
        std::string dump() const;

	//! Save this deslect stack action to a saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

	//! The Action_DeselectStack class doesn't take any info to fill.
        bool fillData();
};
//-----------------------------------------------------------------------------

//! A temporary record of a Player surrendering.
/**
 * The purpose of the Action_Resign class is to record when a Player has
 * resigned from the game.  Because these actions are held in a Player's
 * Actionlist, we do not have to store the player's Id.
 */
class Action_Resign: public Action
{
    public:
	//! Make a new player resignation action.
        Action_Resign();
	//! Copy constructor
	Action_Resign(const Action_Resign &action);
	//! Load a new player resignation action from an opened saved-game file.
        Action_Resign(XML_Helper* helper);
	//! Destroy a player resignation action.
        ~Action_Resign();

	//! Return some debug information about this action.
        std::string dump() const;

	//! Save this player resignation action to an opened saved-game file.
        virtual bool doSave(XML_Helper* helper) const;

	//! This method doesn't need to be called for Action_Resign actions.
        bool fillData();
};

//-----------------------------------------------------------------------------

class Action_Reinforcements: public Action, public sigc::trackable
{
    public:
        Action_Reinforcements();
	Action_Reinforcements(const Action_Reinforcements &action);
        Action_Reinforcements(XML_Helper* helper);
        ~Action_Reinforcements();
        std::string dump() const;
        virtual bool doSave(XML_Helper* helper) const;
        bool fillData(std::list<Stack*> stacks);
	std::list<Stack*> getStacks() const {return d_stacks;};

        private:
        
        std::list<Stack*> d_stacks;

        bool loadStack(std::string tag, XML_Helper* helper);

};

//-----------------------------------------------------------------------------
class Action_DieSelected: public Action
{
    public:
        Action_DieSelected();
	Action_DieSelected (const Action_DieSelected &action);
        Action_DieSelected (XML_Helper* helper);
        ~Action_DieSelected();
        std::string dump() const;
        virtual bool doSave(XML_Helper* helper) const;
        bool fillData(guint32 die_index);
	
        guint32 getDieIndex() const {return d_index;}
    private:
        guint32 d_index;
};

#endif //ACTION_H
