// Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include <stdlib.h>
#include <sstream>
#include <sigc++/functors/mem_fun.h>

#include "action.h"
#include "stack.h"
#include "army.h"
#include "armysetlist.h"
#include "playerlist.h"
#include "player.h"
#include "stacklist.h" //remove me

std::string Action::d_tag = "action";
using namespace std;

#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<< x << endl<<flush;}
//#define debug(x)

Action::Action(Type type)
    :d_type(type)
{
}

Action::Action(const Action &action)
:d_type(action.d_type)
{

}
Action::Action(XML_Helper *helper)
{
  std::string type_str;
  helper->getData(type_str, "type");
  d_type = actionTypeFromString(type_str);
}

Action::~Action()
{
}

bool Action::save(XML_Helper* helper) const
{
    bool retval = true;

    retval &= helper->openTag(Action::d_tag);
    retval &= saveContents(helper);
    retval &= helper->closeTag();

    return retval;
}

bool Action::saveContents(XML_Helper* helper) const
{
    bool retval = true;

    std::string type_str = actionTypeToString(Action::Type(d_type));
    retval &= helper->saveData("type", type_str);
    retval &= doSave(helper);

    return retval;
}

Action* Action::handle_load(XML_Helper* helper)
{
  std::string type_str;

  helper->getData(type_str, "type");
  Action::Type t = actionTypeFromString(type_str);

  switch (t)
    {
      case STACK_MOVE:
          return (new Action_Move(helper));
      case STACK_FIGHT:
          return (new Action_Fight(helper));
      case RESIGN:
          return (new Action_Resign(helper));
      case END_TURN:
          return (new Action_EndTurn(helper));
      case INIT_TURN:
          return (new Action_InitTurn(helper));
      case KILL_PLAYER:
          return (new Action_Kill(helper));
      case PLAYER_RENAME:
          return (new Action_RenamePlayer(helper));
      case STACK_SELECT:
          return (new Action_SelectStack(helper));
      case STACK_DESELECT:
          return (new Action_DeselectStack(helper));
      case REINFORCEMENTS:
          return (new Action_Reinforcements(helper));
      case DIE_SELECTED:
          return (new Action_DieSelected(helper));
    }

  return 0;
}


Action* Action::copy(const Action* a)
{
    switch(a->getType())
    {
        case STACK_MOVE:
            return (new Action_Move(*dynamic_cast<const Action_Move*>(a)));
        case STACK_FIGHT:
            return (new Action_Fight(*dynamic_cast<const Action_Fight*>(a)));
        case RESIGN:
            return (new Action_Resign(*dynamic_cast<const Action_Resign*>(a)));
        case END_TURN:
            return 
              (new Action_EndTurn
                (*dynamic_cast<const Action_EndTurn*>(a)));
        case PLAYER_RENAME:
            return 
              (new Action_RenamePlayer
                (*dynamic_cast<const Action_RenamePlayer*>(a)));
        case INIT_TURN:
            return 
              (new Action_InitTurn
                (*dynamic_cast<const Action_InitTurn*>(a)));
        case KILL_PLAYER:
            return (new Action_Kill (*dynamic_cast<const Action_Kill*>(a)));
        case STACK_SELECT:
            return (new Action_SelectStack
                    (*dynamic_cast<const Action_SelectStack*>(a)));
        case STACK_DESELECT:
            return (new Action_DeselectStack
                    (*dynamic_cast<const Action_DeselectStack*>(a)));
        case REINFORCEMENTS:
            return (new Action_Reinforcements
                    (*dynamic_cast<const Action_Reinforcements*>(a)));
        case DIE_SELECTED:
            return (new Action_DieSelected
                    (*dynamic_cast<const Action_DieSelected*>(a)));
    }

    return 0;
}

//-----------------------------------------------------------------------------
//Action_Move_Step

Action_Move::Action_Move()
    :Action(Action::STACK_MOVE), d_stack(0)
{
    d_dest.x = d_dest.y = 0;
    d_delta.x = d_delta.y = 0;
}

Action_Move::Action_Move (const Action_Move &action)
:Action(action), d_stack(action.d_stack), d_dest(action.d_dest), 
    d_delta(action.d_delta)
{
}

Action_Move::Action_Move(XML_Helper* helper)
    :Action(helper)
{
    helper->getData(d_stack, "stack");

    int i;
    helper->getData(i, "x");
    d_dest.x = i;
    helper->getData(i, "y");
    d_dest.y = i;
    helper->getData(i, "delta_x");
    d_delta.x = i;
    helper->getData(i, "delta_y");
    d_delta.y = i;
}

Action_Move::~Action_Move()
{
}

std::string Action_Move::dump() const
{
    std::stringstream s;

    s <<"Stack " <<d_stack <<" moved to (";
    s <<d_dest.x <<"," <<d_dest.y <<")\n";
    
    return s.str();
}

bool Action_Move::doSave(XML_Helper* helper) const
{
    bool retval = true;

    retval &= helper->saveData("stack", d_stack);
    retval &= helper->saveData("x", d_dest.x);
    retval &= helper->saveData("y", d_dest.y);
    retval &= helper->saveData("delta_x", d_delta.x);
    retval &= helper->saveData("delta_y", d_delta.y);

    return retval;
}

bool Action_Move::fillData(Stack* s, Vector<int> dest)
{
    d_stack = s->getId();
    d_dest = dest;
    d_delta = dest - s->getPos();
    return true;
}

//-----------------------------------------------------------------------------
//Action_Fight

Action_Fight::Action_Fight()
    :Action(Action::STACK_FIGHT)
{
}

Action_Fight::Action_Fight(const Action_Fight &action)
: Action(action), d_history(action.d_history), d_attackers(action.d_attackers), 
    d_defenders(action.d_defenders)
{
}

Action_Fight::Action_Fight(XML_Helper* helper)
    :Action(helper)
{
    std::string s;
    std::istringstream si;
    int ival = -1;

    helper->registerTag("item", sigc::mem_fun(this, &Action_Fight::loadItem));
    // get attacking and defending stacks
    helper->getData(s, "attackers");
    si.str(s);
    while (si.eof() == false)
      {
        ival = -1;
        si >> ival;
        if (ival != -1)
          d_attackers.push_back((guint32)ival);
      }
    si.clear();

    helper->getData(s, "defenders");
    si.str(s);
    while (si.eof() == false)
      {
        ival = -1;
        si >> ival;
        if (ival != -1)
          d_defenders.push_back((guint32) ival);
      }
}

bool Action_Fight::loadItem(std::string tag, XML_Helper* helper)
{
    FightItem item;
    
    helper->getData(item.turn, "turn");
    helper->getData(item.attacker_id, "attacker_id");
    helper->getData(item.defender_id, "defender_id");
    helper->getData(item.attacker_roll, "attacker_roll");
    helper->getData(item.defender_roll, "defender_roll");
    helper->getData(item.attacker_power, "attacker_power");
    helper->getData(item.defender_power, "defender_power");

    d_history.push_back(item);

    return true;
}

Action_Fight::~Action_Fight()
{
}

std::string Action_Fight::dump() const
{
    std::stringstream s;
    std::list<guint32>::const_iterator uit;

    s << "Battle fought.\n Attacking stacks: ";
    for (uit = d_attackers.begin(); uit != d_attackers.end(); uit++)
        s << (*uit) <<" ";

    s << "\n Defending stacks: ";
    for (uit = d_defenders.begin(); uit != d_defenders.end(); uit++)
        s << (*uit) <<" ";

    return s.str();
}

bool Action_Fight::doSave(XML_Helper* helper) const
{
    std::stringstream si;
    std::list<guint32>::const_iterator uit;
    bool retval = true;
    

    // save the stack's ids
    for (uit = d_attackers.begin(); uit != d_attackers.end(); uit++)
        si << (*uit) << " ";
    retval &= helper->saveData("attackers", si.str());

    for (uit = d_defenders.begin(), si.str(""); uit != d_defenders.end(); uit++)
        si << (*uit) << " ";
    retval &= helper->saveData("defenders", si.str());

    // save what happened
    for (std::list<FightItem>::const_iterator fit = d_history.begin(); 
            fit != d_history.end(); fit++)
    {
        retval &= helper->openTag("item");
        retval &= helper->saveData("turn", (*fit).turn);
        retval &= helper->saveData("attacker_id", (*fit).attacker_id);
        retval &= helper->saveData("defender_id", (*fit).defender_id);
        retval &= helper->saveData("attacker_roll", (*fit).attacker_roll);
        retval &= helper->saveData("defender_roll", (*fit).defender_roll);
        retval &= helper->saveData("attacker_power", (*fit).attacker_power);
        retval &= helper->saveData("defender_power", (*fit).defender_power);
        retval &= helper->closeTag();
    }
    return retval;
}

bool Action_Fight::fillData(const Fight* f)
{
    std::list<Stack*> list = f->getAttackers();
    std::list<Stack*>::const_iterator it;

    for (it = list.begin(); it != list.end(); it++)
        d_attackers.push_back((*it)->getId());
        
    list = f->getDefenders();

    for (it = list.begin(); it != list.end(); it++)
        d_defenders.push_back((*it)->getId());
    
    d_history = f->getCourseOfEvents();

    return true;
}

bool Action_Fight::stack_ids_to_stacks(std::list<guint32> stack_ids, std::list<Stack*> &stacks, guint32 &stack_id) const
{
  for (std::list<guint32>::iterator i = stack_ids.begin(); i != stack_ids.end(); i++)
    {
      bool found = false;
      for (Playerlist::iterator j = Playerlist::getInstance()->begin(), jend = Playerlist::getInstance()->end();
           j != jend; ++j) 
        {
          Stack *s = (*j)->getStacklist()->getStackById(*i);
          if (s)
            {
              found = true;
              stacks.push_back(s);
              break;
            }
        }
      if (found == false)
        {
          stack_id = *i;
          return false;
        }
    }
  return true;
}

bool Action_Fight::is_army_id_in_stacks(guint32 id, std::list<guint32> stack_ids) const
{
  std::list<Stack*> stacks;
  guint32 stack_id = 0;
  bool success = stack_ids_to_stacks(stack_ids, stacks, stack_id);
  if (!success)
    return false;
  bool found = false;
  for (std::list<Stack*>::iterator i = stacks.begin(); i != stacks.end(); i++)
    {
      if ((*i)->getArmyById(id))
        {
          found = true;
          break;
        }
    }
  if (found)
    return true;
  else
    return false;
}


//-----------------------------------------------------------------------------
//Action_Resign

Action_Resign::Action_Resign()
:Action(Action::RESIGN)
{
}

Action_Resign::Action_Resign(const Action_Resign &action)
: Action(action)
{
}

Action_Resign::Action_Resign(XML_Helper* helper)
:Action(helper)
{
}

Action_Resign::~Action_Resign()
{
}

std::string Action_Resign::dump() const
{
  std::stringstream s;
  s << "this player resigns\n";

  return s.str();
}

bool Action_Resign::doSave(XML_Helper* helper) const
{
  bool retval = true;

  return retval;
}

bool Action_Resign::fillData()
{
  return true;
}

//-----------------------------------------------------------------------------
//Action_EndTurn

Action_EndTurn::Action_EndTurn()
:Action(Action::END_TURN)
{
}

Action_EndTurn::Action_EndTurn(const Action_EndTurn &action)
: Action(action)
{
}

Action_EndTurn::Action_EndTurn(XML_Helper* helper)
:Action(helper)
{
}

Action_EndTurn::~Action_EndTurn()
{
}

std::string Action_EndTurn::dump() const
{
  return "ending turn\n";
}

bool Action_EndTurn::doSave(XML_Helper* helper) const
{
  bool retval = true;

  return retval;
}

//-----------------------------------------------------------------------------
//Action_RenamePlayer

Action_RenamePlayer::Action_RenamePlayer()
  :Action(Action::PLAYER_RENAME), d_name("")
{
}

Action_RenamePlayer::Action_RenamePlayer(const Action_RenamePlayer &action)
:Action(action), d_name(action.d_name)
{
}

Action_RenamePlayer::Action_RenamePlayer(XML_Helper* helper)
  :Action(helper)
{
    helper->getData(d_name, "name");
}

Action_RenamePlayer::~Action_RenamePlayer()
{
}

std::string Action_RenamePlayer::dump() const
{
    std::stringstream s;

    s << "Player changes name to " << d_name <<".\n";

    return s.str();
}

bool Action_RenamePlayer::doSave(XML_Helper* helper) const
{
    bool retval = true;

    retval &= helper->saveData("name", d_name);

    return retval;
}

bool Action_RenamePlayer::fillData(std::string name)
{
  d_name = name;
  return true;
}

//-----------------------------------------------------------------------------
//Action_InitTurn

Action_InitTurn::Action_InitTurn()
:Action(Action::INIT_TURN)
{
}

Action_InitTurn::Action_InitTurn(const Action_InitTurn &action)
: Action(action), d_dice(action.d_dice)
{
}

Action_InitTurn::Action_InitTurn(XML_Helper* helper)
:Action(helper)
{
  guint32 die;
  helper->getData(die, "die1");
  d_dice.push_back(die);
  helper->getData(die, "die2");
  d_dice.push_back(die);
  helper->getData(die, "die3");
  d_dice.push_back(die);
  helper->getData(die, "die4");
  d_dice.push_back(die);
}

Action_InitTurn::~Action_InitTurn()
{
}

bool Action_InitTurn::fillData(int die1, int die2, int die3, int die4)
{
  d_dice.push_back(die1);
  d_dice.push_back(die2);
  d_dice.push_back(die3);
  d_dice.push_back(die4);
  return true;
}

std::string Action_InitTurn::dump() const
{
  std::stringstream ss;
  ss <<"player starts turn with dice:";
  for (std::vector<guint32>::const_iterator i = d_dice.begin(); 
       i != d_dice.end(); i++)
    ss << " " << *i;
  ss << "\n";

  return ss.str();
}

bool Action_InitTurn::doSave(XML_Helper* helper) const
{
  bool retval = true;
  helper->saveData("die1", d_dice[0]);
  helper->saveData("die2", d_dice[1]);
  helper->saveData("die3", d_dice[2]);
  helper->saveData("die4", d_dice[3]);
  return retval;
}

//-----------------------------------------------------------------------------
// Action_Kill

Action_Kill::Action_Kill()
:Action(Action::KILL_PLAYER)
{
}

Action_Kill::Action_Kill(const Action_Kill &action)
: Action(action)
{
}

Action_Kill::Action_Kill(XML_Helper* helper)
:Action(helper)
{
}

Action_Kill::~Action_Kill()
{
}

std::string Action_Kill::dump() const
{
  std::stringstream ss;
  ss <<"player is vanquished.";
  ss << "\n";

  return ss.str();
}

bool Action_Kill::doSave(XML_Helper* helper) const
{
  bool retval = true;
  return retval;
}

bool Action_Kill::fillData()
{
  return true;
}

//-----------------------------------------------------------------------------
// Action_SelectStack

Action_SelectStack::Action_SelectStack()
:Action(Action::STACK_SELECT), d_stack_id(0)
{
}

Action_SelectStack::Action_SelectStack(const Action_SelectStack &action)
: Action(action), d_stack_id(action.d_stack_id)
{
}

Action_SelectStack::Action_SelectStack(XML_Helper* helper)
:Action(helper)
{
  helper->getData(d_stack_id, "stack_id");
}

Action_SelectStack::~Action_SelectStack()
{
}

std::string Action_SelectStack::dump() const
{
  std::stringstream ss;

  ss <<"Stack " <<d_stack_id <<" is selected. " << std::endl;

  return ss.str();
}

bool Action_SelectStack::doSave(XML_Helper* helper) const
{
  bool retval = true;
  retval &= helper->saveData("stack_id", d_stack_id);
  return retval;
}

bool Action_SelectStack::fillData(Stack *s)
{
  d_stack_id = s->getId();
  return true;
}

//-----------------------------------------------------------------------------
// Action_DeselectStack

Action_DeselectStack::Action_DeselectStack()
:Action(Action::STACK_DESELECT)
{
}

Action_DeselectStack::Action_DeselectStack(const Action_DeselectStack &action)
: Action(action)
{
}

Action_DeselectStack::Action_DeselectStack(XML_Helper* helper)
:Action(helper)
{
}

Action_DeselectStack::~Action_DeselectStack()
{
}

std::string Action_DeselectStack::dump() const
{
  std::stringstream ss;

  ss <<"deselecting stack. " << std::endl;

  return ss.str();
}

bool Action_DeselectStack::doSave(XML_Helper* helper) const
{
  return true;
}

bool Action_DeselectStack::fillData()
{
  return true;
}

//-----------------------------------------------------------------------------
//Action_Reinforcements

Action_Reinforcements::Action_Reinforcements()
    :Action(Action::REINFORCEMENTS)
{
}

Action_Reinforcements::Action_Reinforcements(const Action_Reinforcements &action)
: Action(action), d_stacks(action.d_stacks)
{
}

Action_Reinforcements::Action_Reinforcements(XML_Helper* helper)
    :Action(helper)
{
    helper->registerTag(Stack::d_tag, 
                        sigc::mem_fun(this, &Action_Reinforcements::loadStack));
}

bool Action_Reinforcements::loadStack(std::string tag, XML_Helper* helper)
{
  if (tag == Stack::d_tag)
    {
      d_stacks.push_back(new Stack(helper));
      return true;
    }
  return false;
}

Action_Reinforcements::~Action_Reinforcements()
{
}

std::string Action_Reinforcements::dump() const
{
    std::stringstream s;
    std::list<guint32>::const_iterator uit;

    s << "Reinforcements showed up.  Stacks:";
    for (std::list<Stack*>::const_iterator i = d_stacks.begin(); i != d_stacks.end();
         i++)
        s << " " << (*i)->getId();
    s << "\n";

    return s.str();
}

bool Action_Reinforcements::doSave(XML_Helper* helper) const
{
  for (std::list<Stack*>::const_iterator i = d_stacks.begin(); 
       i != d_stacks.end(); i++)
    (*i)->save(helper);
  return true;
}

bool Action_Reinforcements::fillData(std::list<Stack*> stacks)
{
  d_stacks = stacks;
  return true;
}

//-----------------------------------------------------------------------------
// Action_DieSelected

Action_DieSelected::Action_DieSelected()
:Action(Action::DIE_SELECTED), d_index(0)
{
}

Action_DieSelected::Action_DieSelected(const Action_DieSelected &action)
: Action(action), d_index(action.d_index)
{
}

Action_DieSelected::Action_DieSelected(XML_Helper* helper)
:Action(helper)
{
  helper->getData(d_index, "index");
}

Action_DieSelected::~Action_DieSelected()
{
}

std::string Action_DieSelected::dump() const
{
  std::stringstream ss;

  ss <<"die " <<d_index <<" is selected. " << std::endl;

  return ss.str();
}

bool Action_DieSelected::doSave(XML_Helper* helper) const
{
  bool retval = true;
  retval &= helper->saveData("index", d_index);
  return retval;
}

bool Action_DieSelected::fillData(guint idx)
{
  d_index = idx;
  return true;
}

std::string Action::actionTypeToString(Action::Type type)
{
  switch (type)
    {
    case Action::STACK_MOVE:
      return "Action::STACK_MOVE";
    case Action::STACK_FIGHT:
      return "Action::STACK_FIGHT";
    case Action::RESIGN:
      return "Action::RESIGN";
    case Action::END_TURN:
      return "Action::END_TURN";
    case Action::PLAYER_RENAME:
      return "Action::PLAYER_RENAME";
    case Action::INIT_TURN:
      return "Action::INIT_TURN";
    case Action::KILL_PLAYER:
      return "Action::KILL_PLAYER";
    case Action::STACK_SELECT:
      return "Action::STACK_SELECT";
    case Action::STACK_DESELECT:
      return "Action::STACK_DESELECT";
    case Action::REINFORCEMENTS:
      return "Action::REINFORCEMENTS";
    case Action::DIE_SELECTED:
      return "Action::DIE_SELECTED";
    }
      
  return "Action::MOVE";
}

Action::Type Action::actionTypeFromString(std::string str)
{
  if (str.size() > 0 && isdigit(str.c_str()[0]))
    return Action::Type(atoi(str.c_str()));
  if (str == "Action::STACK_MOVE")
    return Action::STACK_MOVE;
  else if (str == "Action::STACK_FIGHT")
    return Action::STACK_FIGHT;
  else if (str == "Action::RESIGN")
    return Action::RESIGN;
  else if (str == "Action::END_TURN")
    return Action::END_TURN;
  else if (str == "Action::PLAYER_RENAME")
    return Action::PLAYER_RENAME;
  else if (str == "Action::INIT_TURN")
    return Action::INIT_TURN;
  else if (str == "Action::STACK_SELECT")
    return Action::STACK_SELECT;
  else if (str == "Action::STACK_DESELECT")
    return Action::STACK_DESELECT;
  else if (str == "Action::REINFORCEMENTS")
    return Action::REINFORCEMENTS;
  else if (str == "Action::DIE_SELECTED")
    return Action::DIE_SELECTED;
  return Action::STACK_MOVE;
}
