// Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#ifndef STACKLIST_H
#define STACKLIST_H

#include <list>
#include <map>
#include <vector>
#include <string>
#include <sigc++/trackable.h>
#include <gtkmm.h>
#include "vector.h"
#include <sstream>
#include <sigc++/signal.h>

class Stack;
class XML_Helper;
class Player;

class Stacklist : public std::list<Stack*>, public sigc::trackable
{
    public:

	//! The xml tag of this object in a saved-game file.
	static std::string d_tag; 

	//! Default constructor.
        Stacklist();

	//! Copy constructor.
        Stacklist(Stacklist *stacklist);

	//! Loading constructor.
        Stacklist(XML_Helper* helper);

	//! Destructor.
        ~Stacklist();

	// Set Methods

	//!Set the stack the player is currently working with.
        /** 
	 * Sets the currently selected stack. The purpose of this method is 
         * to designate a stack to be the one the player is currently touching.
         * It is important to use this method because several functions
	 * expect that there is an active stack.
         *
         * @param activestack      The stack currently selected by the player.
         */
        void setActivestack(Stack* activestack);


	// Get Methods

        //! Returns the currently selected stack.
        Stack* getActivestack() const {return d_activestack;}

        bool hasGeneral() const;

	// Methods that operate on class data and do not modify the class.

        //! Returns the total number of armies in the list.
        unsigned int countArmies() const;

        //! Save the data to an opened saved-game file.
        bool save(XML_Helper* helper) const;

	//! Find a stack in this stacklist given this id.
        Stack *getStackById(guint32 id) const;

	//! Find the stack in this stacklist that contains an army with this id.
        Stack *getArmyStackById(guint32 army) const;

        //! Erase all stacks from the list, and their contents too.
        void flClear();

	//! Add a stack to the list.
	void add(Stack *stack);

        /** 
	 * Erase a Stack from the list, and free the contents of the Stack.
	 *
	 * @param it   The place in the Stacklist to erase.
	 *
	 * @return The place in the list that was erased.
         */
	//! Erase a stack from the list.
        iterator flErase(iterator object);

        /** 
	 * Scan the list of stacks for a particular stack.  If it is found,
	 * remove it from the list of stacks and free it's contents.
	 *
	 * @param stack  The stack in the Stacklist to remove.
	 *
	 * @return Whether or not the stack was found and deleted.
         */
        //! Erase a stack from the list.
        bool flRemove(Stack* stack);

	//! Erase a stack from the list, given the stack id.
        bool flRemove(guint32 id);

	// Signals

	//! The stack in the stacklist is arriving on a new tile.
	sigc::signal<void, Stack*, Vector<int> > snewpos;

	//! The stack in the stacklist is leaving an old tile.
	sigc::signal<void, Stack*, Vector<int> > soldpos;


	// Static Methods

        //! Searches through the all players Stacklists and deletes the stack.
        static bool deleteStack(Stack* stack);

	//! Searches for the stack id in all lists, and deletes it.
	static bool deleteStack(guint32 id);

        //! Returns the total number of stacks owned by all players.
        static unsigned int getNoOfStacks();

        //! Returns the total number of armies owned by all players.
        static unsigned int getNoOfArmies();

        //! Return position of an Army with the given id in this stacklist.
	/**
	 * Scan through all stacks in the list, and then through each Army 
	 * unit of every Stack for an Army unit with a particular Id.
	 *
	 * @param id     The Id of the Army unit that we're looking for.
	 *
	 * @return The position of the Army unit.  If no Army unit could be
	 *         found with the given Id, the position of (-1,-1) is
	 *         returned.
	 */
        static Vector<int> getPosition(guint32 id);


        std::list<Vector<int> > getPositions() const;

    private:

        //! Callback for loading this object from an opened saved-game file.
        bool load(std::string tag, XML_Helper* helper);

	//! Callback for when a stack initiates movement after being stopped.
	void on_stack_starts_moving (Stack *s);

	//! Callback for when a stack halts.
	void on_stack_stops_moving (Stack *s);

	//! Callback for when a stack has been killed, or disbanded.
	void on_stack_died (Stack *stack);

	//! Notify the game map that a stack is leaving it's tile.
	bool deletePositionFromMap(Stack *stack) const;

	//! Notify the game map that a stack is arriving on a tile.
	bool addPositionToMap(Stack *s) const;

	// DATA

	//! A pointer to the currently selected Stack.
        Stack* d_activestack;

	typedef std::map<Stack *, std::list<sigc::connection> > ConnectionMap;
	// A set of signal connections for each stack in our list.
	/**
	 * We can use this map to lookup the outstanding signals connections
	 * that a given stack has.  This is so that we can destroy them
	 * when the stack is deleted.
	 */
	ConnectionMap d_connections;
	
	typedef std::map<guint32, Stack*> IdMap;
	//! A map to quickly lookup the stack by it's unique id.
	IdMap d_id;

};

#endif // STACKLIST_H

// End of file
