// Copyright (C) 2011 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "config.h"

#include <algorithm>
#include <vector>
#include <assert.h>
#include <gtkmm.h>
#include <sigc++/functors/mem_fun.h>
#include <sigc++/adaptors/bind.h>

#include "game.h"

#include "ucompose.hpp"
#include "rectangle.h"
#include "sound.h"
#include "GraphicsCache.h"
#include "GameScenario.h"
#include "NextTurnHotseat.h"

#include "army.h"
#include "fight.h"
#include "stacklist.h"
#include "GameMap.h"
#include "playerlist.h"
#include "Configuration.h"
#include "File.h"
#include "action.h"
#include "game-parameters.h"
#include "GameMap.h"
#include "history.h"
#include "stacktile.h"

#include "GameScenarioOptions.h"

Game *Game::current_game = 0;

//#define debug(x) {cerr<<__FILE__<<": "<<__LINE__<<": "<<x<<flush<<endl;}
#define debug(x)
void Game::addPlayer(Player *p)
{

  //disconnect prior players' connections
  for (std::list<sigc::connection>::iterator it = 
       connections[p->getId()].begin(); 
       it != connections[p->getId()].end(); it++) 
    (*it).disconnect();
  connections[p->getId()].clear();

  //now setup the connections that are specific for human players
  if (p->getType() == Player::HUMAN)
    {

      connections[p->getId()].push_back
	(p->smovingStack.connect
	 (sigc::mem_fun(this, &Game::on_stack_starts_moving)));
      connections[p->getId()].push_back
	(p->smovingRaft.connect
	 (sigc::mem_fun(this, &Game::on_raft_starts_moving)));
      connections[p->getId()].push_back
	(p->sstoppingStack.connect
	 (sigc::mem_fun(this, &Game::on_stack_stopped)));
      connections[p->getId()].push_back
	(p->shaltedStack.connect
	 (sigc::mem_fun(this, &Game::on_stack_halted)));
    }
      
      
  //now do all of the common connections
      
  connections[p->getId()].push_back
    (p->getStacklist()->snewpos.connect
     (sigc::mem_fun(this, &Game::stack_arrives_on_tile)));
  connections[p->getId()].push_back
    (p->getStacklist()->soldpos.connect
     (sigc::mem_fun(this, &Game::stack_leaves_tile)));
  connections[p->getId()].push_back
    (p->aborted_turn.connect (sigc::mem_fun
	   (game_stopped, &sigc::signal<void>::emit)));

  connections[p->getId()].push_back
    (p->fight_started.connect (sigc::mem_fun(*this, &Game::on_fight_started)));
  if (p->getType() == Player::HUMAN && p == Playerlist::getActiveplayer())
    unlock_inputs();
}

void Game::on_raft_starts_moving(Stack *stack, std::list<Vector<int> > points)
{
  if (Playerlist::getActiveplayer()->getType() == Player::HUMAN)
    lock_inputs();
  raft_moves.emit(stack, points);
}

void Game::on_stack_starts_moving(Stack *stack, std::list<Vector<int> > points, Vector<int> blocked, Vector<int> stopped_short)
{
  if (Playerlist::getActiveplayer()->getType() == Player::HUMAN)
    lock_inputs();
  stack_moves.emit(stack, points, blocked, stopped_short);
}

void Game::on_stack_stopped()
{
  if (Playerlist::getActiveplayer()->getType() == Player::HUMAN)
    unlock_inputs();
  if (Playerlist::getActiveplayer()->count_dice_used_this_turn() < 4)
    maybe_reinforce();
}

void Game::on_stack_halted(Stack *stack)
{
  if (Playerlist::getActiveplayer()->getType() == Player::HUMAN)
    unlock_inputs();
  if (stack == NULL)
    return;
}

Game::Game(GameScenario* gameScenario, NextTurn *nextTurn)
    : d_gameScenario(gameScenario), d_nextTurn(nextTurn)
{
    current_game = this;
    input_locked = false;

    // connect player callbacks
    Playerlist* pl = Playerlist::getInstance();
    for (Playerlist::iterator i = pl->begin(); i != pl->end(); ++i)
    {
	Player *p = *i;
	addPlayer(p);
    }
    if (gameScenario->getPlayMode() == GameScenario::HOTSEAT)
      pl->splayerDead.connect(sigc::mem_fun(this, &Game::on_player_died));

    d_nextTurn->splayerStart.connect(
	sigc::mem_fun(this, &Game::init_turn_for_player));
    d_nextTurn->snextRound.connect(
	sigc::mem_fun(d_gameScenario, &GameScenario::nextRound));
    d_nextTurn->snextRound.connect(
	sigc::mem_fun(this, &Game::nextRound));
    d_nextTurn->supdating.connect(
	sigc::mem_fun(this, &Game::redraw));
            
}

Game::~Game()
{
  for (unsigned int i = 0; i < MAX_PLAYERS + 1; i++)
    {
      for (std::list<sigc::connection>::iterator it = connections[i].begin(); 
	   it != connections[i].end(); it++) 
	(*it).disconnect();
      connections[i].clear();
    }
    delete d_gameScenario;
    delete d_nextTurn;
}

GameScenario *Game::getScenario()
{
  return current_game->d_gameScenario;
}

void Game::end_turn()
{
  //only human players hit this.
    unselect_active_stack();
    lock_inputs();

    d_nextTurn->endTurn();
}


void Game::redraw()
{
}

void Game::deselect_selected_stack()
{
  Player *player = Playerlist::getActiveplayer();
  player->setActivestack(0);
  unselect_active_stack();
}

void Game::stackUpdate(Stack* s)
{
  if (!s)
    s = Playerlist::getActiveplayer()->getActivestack();

  if (!s)
    redraw();
}

void Game::on_stack_selected(Stack* s)
{
}

void Game::lock_inputs()
{
  input_locked = true;
}

void Game::unlock_inputs()
{
  input_locked = false;
}

void Game::startGame()
{
  debug ("start_game()");
      
  lock_inputs();

    d_nextTurn->start();
}

void Game::loadGame()
{
  Player *player = Playerlist::getActiveplayer();
  if (!player)
    {
      Playerlist::getInstance()->nextPlayer();
      player = Playerlist::getActiveplayer();
    }

  if (player->getType() == Player::HUMAN && (d_gameScenario->getPlayMode() == GameScenario::HOTSEAT))
    {
      //human players want access to the controls and an info box
      unlock_inputs();
      player->setActivestack(0);
      game_loaded.emit(player);
      if (player->getType() == Player::HUMAN)
	d_nextTurn->setContinuingTurn();
    }
  else
    lock_inputs();

  d_nextTurn->start();
}

void Game::stopGame()
{
  d_nextTurn->stop();
}

bool Game::saveGame(std::string file)
{
  return d_gameScenario->saveGame(file);
}

void Game::init_turn_for_player(Player* p)
{
  next_player_turn.emit(p, d_gameScenario->getRound());

  if (p->getType() == Player::HUMAN)
    {
      unlock_inputs();

      redraw();

    }
  else
    {
      //SDL_Delay(250);
    }
}

void Game::on_player_died(Player *player)
{
  const Playerlist* pl = Playerlist::getInstance();
  if (pl->getNoOfPlayers() <= 1)
    game_over.emit(pl->getFirstLiving());
  else
    player_died.emit(player);
}

void Game::on_fight_started(Fight &fight)
{
  FightItem item = fight.getCourseOfEvents().front();
  fight_started.emit(fight.getResult(), item.defender_roll + 1,
                     item.attacker_power, item.defender_power);
  
  if (Playerlist::getActiveplayer()->count_dice_used_this_turn() < 4)
    maybe_reinforce();
}

void Game::select_active_stack()
{
}

void Game::unselect_active_stack()
{
}

void Game::nextRound()
{
}
    
void Game::stack_arrives_on_tile(Stack *stack, Vector<int> tile)
{
  StackTile *stile = GameMap::getInstance()->getTile(tile)->getStacks();
  stile->arriving(stack);
}

void Game::stack_leaves_tile(Stack *stack, Vector<int> tile)
{
  StackTile *stile = GameMap::getInstance()->getTile(tile)->getStacks();
  bool left = stile->leaving(stack);
  if (left == false)
    {
      if (stack == NULL)
	{
	  printf("stack is %p\n", stack);
	  printf("WTFFF!!!!!!!!!!!!!!!!!!!!\n");
	  return;
	}
    }
}

void Game::maybe_reinforce()
{
  //if at least 10 armies have died.
  //and if there's a pool of dead guys larger than 2
  //decrement the pool of dead guys when used as reinforcements
  //increment the pool of dead guys when we kill a bad guy
  //one pool of dead guys per player.
  //and don't reinforce if we already did that this turn.
  //
  //we only get 3 chances at this per turn
  //
  Player *active = Playerlist::getActiveplayer();
  if (active->count_reinforcements_this_turn() > 0)
    return;

  std::list<guint32> my_dead_armies;
  active->getArmyTypesKilled(my_dead_armies, active);
  Playerlist *pl = Playerlist::getInstance();
  std::list<guint32> other_dead_armies;
  for (Playerlist::iterator i = pl->begin(); i != pl->end(); i++)
    {
      if (*i == active)
        continue;
      (*i)->getArmyTypesKilled(other_dead_armies, active);
    }
  if ((my_dead_armies.size() + other_dead_armies.size()) < 10)
    return;
  std::list<guint32> reinforcements = active->getArmyTypesReinforced();
  for (std::list<guint32>::iterator i = reinforcements.begin(); i != reinforcements.end(); i++)
    my_dead_armies.remove(*i);

  if (my_dead_armies.size() > 2 && (rand() % 13) == 0)
    {
      std::vector<guint32> possible_reinforcements;
      for (std::list<guint32>::iterator i = my_dead_armies.begin(); i != my_dead_armies.end(); i++)
        possible_reinforcements.push_back(*i);

      std::list<Army*> armies;
      for (int i = 0; i < 3; i++)
        {
          ArmyProto *proto = Armysetlist::getInstance()->getArmy(active->getArmyset(), possible_reinforcements[rand() % possible_reinforcements.size()]);
          armies.push_back(new Army(*proto, active));
        }
      active->reinforce(armies);
    }

}
