;;   Emacs-Lisp major mode to support argile programming language
;;   Copyright (C) 2009,2010,2011 the Argile authors
;;
;;   This software is free software: you can redistribute it and/or modify
;;   it under the terms of the GNU General Public License as published by
;;   the Free Software Foundation, either version 3 of the License, or
;;   (at your option) any later version.
;;
;;   This software is distributed in the hope that it will be useful,
;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;   GNU General Public License for more details.
;;
;;   You should have received a copy of the GNU General Public License
;;   along with this software.  If not, see <http://www.gnu.org/licenses/>.
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; To use this file, add in your ~/.emacs :
;; (load "/path/to/argile.el")
;; (argile-mode-be-auto)
;; (prefer-coding-system 'utf-8-unix) ;; recommended
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;; MAJOR MODE FOR ARGILE ;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; Sets up keymap, syntax table, abbrev table
;; and some local variables ...
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(provide 'argile-mode)
(provide 'argile-mode-map)
(provide 'argile-mode-abbrev-table)
(provide 'argile-mode-syntax-table)
(provide 'argile-mode-be-auto)
(provide 'argile-indent-line)
(provide 'argile-comment-region)
(provide 'argile-speedbar-add-ext)

(require 'speedbar)

;;;;;;;;;;;;
;; Keymap ;;
;;;;;;;;;;;;
(defvar argile-mode-map nil
  "Keymap used in argile mode.")
(unless argile-mode-map
  (setq argile-mode-map (make-sparse-keymap))
  (define-key argile-mode-map "\C-c\C-c" 'argile-comment-region))

;;;;;;;;;;;;;;;;;;
;; Abbrev table ;;
;;;;;;;;;;;;;;;;;;
(defvar argile-mode-abbrev-table nil
  "Abbrev table used in argile mode.")
(define-abbrev-table 'argile-mode-abbrev-table ())

;;;;;;;;;;;;;;;;;;
;; Syntax table ;;
;;;;;;;;;;;;;;;;;;
(defvar argile-mode-syntax-table nil
  "Syntax table used in argile mode.")
(unless argile-mode-syntax-table
  (setq argile-mode-syntax-table (make-syntax-table))
  (let ((i 0))
    (while (<= i 127)
      (modify-syntax-entry i ". "  argile-mode-syntax-table)
      (setq i (1+ i)))
    ; 0-9
    (setq i ?0)
    (while (<= i ?9)
      (modify-syntax-entry i "_ "  argile-mode-syntax-table)
      (setq i (1+ i)))
    ; a-z
    (setq i ?a)
    (while (<= i ?z)
      (modify-syntax-entry i "w "  argile-mode-syntax-table)
      (setq i (1+ i)))
    ; A-Z
    (setq i ?A)
    (while (<= i ?Z)
      (modify-syntax-entry i "w "  argile-mode-syntax-table)
      (setq i (1+ i))))
  (modify-syntax-entry ?\; "_ "    argile-mode-syntax-table)
  (modify-syntax-entry ?.  "_ "    argile-mode-syntax-table)
  (modify-syntax-entry ?_  "w "    argile-mode-syntax-table)
  (modify-syntax-entry ?\\ "\\ "   argile-mode-syntax-table)
  (modify-syntax-entry ?\" "\" "   argile-mode-syntax-table)
  (modify-syntax-entry ?\( "()1n"  argile-mode-syntax-table)
  (modify-syntax-entry ?:  "_ 23n" argile-mode-syntax-table)
  (modify-syntax-entry ?\) ")(4n"  argile-mode-syntax-table)
;;(modify-syntax-entry ?\[ "(]"    argile-mode-syntax-table)
;;(modify-syntax-entry ?\] ")["    argile-mode-syntax-table)
  (modify-syntax-entry ?\{ "(}"    argile-mode-syntax-table)
  (modify-syntax-entry ?\} "){"    argile-mode-syntax-table)
  (modify-syntax-entry ?\n "  "    argile-mode-syntax-table)
  (modify-syntax-entry ?\t "  "    argile-mode-syntax-table)
  (modify-syntax-entry ?   "  "    argile-mode-syntax-table))

;;;;;;;;;;;;;;;;;;;;;;;;
;; Rainbow Highlights ;;
;;;;;;;;;;;;;;;;;;;;;;;;
(defvar argile-font-lock-comment-face 'argile-font-lock-comment-face
  "Comment face used in argile mode.")
(make-face 'argile-font-lock-comment-face)
(set-face-foreground 'argile-font-lock-comment-face "gray55")
(set-face-italic-p 'argile-font-lock-comment-face t)

(defvar argile-font-lock-string-face 'argile-font-lock-string-face
  "String face used in argile mode.")
(make-face 'argile-font-lock-string-face)
(set-face-foreground 'argile-font-lock-string-face "steel blue")
;(set-face-background 'argile-font-lock-string-face "gray77")
(set-face-italic-p 'argile-font-lock-string-face t)

(defvar argile-font-lock-int-face 'argile-font-lock-int-face
  "Integer face used in argile mode.")
(make-face 'argile-font-lock-int-face)
(set-face-foreground 'argile-font-lock-int-face "blue")

(defvar argile-font-lock-float-face 'argile-font-lock-float-face
  "Float face used in argile mode.")
(make-face 'argile-font-lock-float-face)
(set-face-foreground 'argile-font-lock-float-face "lime green")

(defvar argile-font-lock-op-face 'argile-font-lock-op-face
  "Operator face used in argile mode.")
(make-face 'argile-font-lock-op-face)
(set-face-foreground 'argile-font-lock-op-face "firebrick")

(defvar argile-font-lock-paren-face 'argile-font-lock-paren-face
  "Parenthesis face used in argile mode.")
(make-face 'argile-font-lock-paren-face)
(set-face-bold-p 'argile-font-lock-paren-face t)
(set-face-foreground 'argile-font-lock-paren-face "orange")

(defvar argile-font-lock-brace-face 'argile-font-lock-brace-face
  "Brace face used in argile mode.")
(make-face 'argile-font-lock-brace-face)
(set-face-foreground 'argile-font-lock-brace-face "firebrick")
(set-face-bold-p 'argile-font-lock-brace-face t)

(defvar argile-font-lock-syntax-face 'argile-font-lock-syntax-face
  "Syntax delimiters face used in argile mode.")
(copy-face 'argile-font-lock-brace-face 'argile-font-lock-syntax-face)

(defvar argile-font-lock-keywords nil
  "Highlight table used in argile mode.")
(setq argile-font-lock-keywords
      (list (list "\\(\\B\\|-\\)[0-9]+\\.[0-9]+\\([eE][+-]?[0-9]+\\)?" 0
		  argile-font-lock-float-face)
	    (list "\\([^a-zA-Z_0-9-]\\|^\\)\\(-?0[xX][0-9a-fA-F]+\\)" 2
		  argile-font-lock-int-face)
	    (list "\\([^a-zA-Z_0-9-]\\|^\\)\\(-?0[bB][01]+\\)" 2
		  argile-font-lock-int-face)
	    (list "\\([^a-zA-Z_0-9-]\\|^\\)\\(-?[0-9]+\\)" 2
		  argile-font-lock-int-face)
	    (list "[:;.]" 0 argile-font-lock-syntax-face)
	    (list "\\s.+" 0 argile-font-lock-op-face)
	    (list "[][]+" 0 argile-font-lock-op-face)
	    (list "[{}]+" 0 argile-font-lock-brace-face)
	    (list "[()]+" 0 argile-font-lock-paren-face)))

;;;;;;;;;;;;;;;;;
;; Indentation ;;
;;;;;;;;;;;;;;;;;
(defun argile-open-paren ()
  "Check if the current line has non-closed parenthesis."
  (save-excursion
    (beginning-of-line)
    (let ((count 0)
	  (open nil))
       (while (not (looking-at "$"))
	 (cond
	  ((looking-at "\\s(")
	   (setq count (1+ count))
	   (setq open t))
	  ((looking-at "\\s)")
	   (setq count (1- count))
	   (setq open nil)))
	 (forward-char))
       (or (> count 0) open))))

(defun argile-indent-to (column)
  "Indent line to a column number."
  (when (< column (current-indentation))
    (beginning-of-line)
    (when (re-search-forward "^[ \t]+" nil t)
      (replace-match "" nil nil)))
  (indent-to column))

(defun argile-indent-line ()
  "Indent line in argile mode."
  (interactive)
  (save-excursion
    (while (and (looking-at "[ \t]") (not (bolp)))
      (backward-char))
    (if (looking-at "[ \t]")
	(while (and (looking-at "[ \t]") (not (eolp)))
	  (forward-char))
      (let ((col 0))
	(beginning-of-line)
	(when (looking-at "^[ \t]*\\(\\s)\\|:)\\)")
	  (setq col -2))
	(re-search-forward "[^ \t]" nil t)
	(backward-char)
	(while (looking-at "[ \t]")
	  (forward-char))
	(argile-indent-to
	 (save-excursion
	   (beginning-of-line)
	   (if (not (re-search-backward "^[^\n]" nil t))
	       0
	     (setq col (+ col (current-indentation)))
	     (when (argile-open-paren)
	       (setq col (+ 2 col)))
	     col))))))
  (when (looking-at "[ \t]+")
    (goto-char (match-end 0))))

;;;;;;;;;;;;;;
;; Comments ;;
;;;;;;;;;;;;;;
(defun argile-comment-region (beg end)
  "Comment a region."
  (interactive "r")
  (let ((decorator (if (< (count-lines beg end) 1)
		       " " "\n")))
    ;; insert comment end
    (goto-char end)
    (end-of-line)
    (insert decorator comment-end)
    ;; insert comment start
    (goto-char beg)
    (beginning-of-line)
    (when (and (looking-at "[ \t]+") (not (string-equal decorator "\n")))
      (goto-char (match-end 0)))
    (insert comment-start decorator)))

;;;;;;;;;;;;;;;
;; Auto-mode ;;
;;;;;;;;;;;;;;;
(defun argile-speedbar-add-ext ()
  "Add .arg files to speedbar"
  (speedbar-add-supported-extension ".[aA]+[rR]+[gG]+[lL]*"))
(defun argile-mode-be-auto ()
  "Set argile mode when openning a .arg file"
  (interactive)
  (setq auto-mode-alist
	(cons '("\\.[aA]+[rR]+[gG]+[lL]*$" . argile-mode) auto-mode-alist))
  (argile-speedbar-add-ext))

;;;;;;;;;;;
;; Hooks ;;
;;;;;;;;;;;
(defvar argile-mode-hook nil
  "*Hook called by argile-mode.")

;;;;;;;;;;;;;;;;;;;;
;; Setup function ;;
;;;;;;;;;;;;;;;;;;;;
(defun argile-mode ()
  "Major mode for editing argile code in Emacs.
\\{argile-mode-map}"
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'argile-mode)
  (setq mode-name ": argile :")

  ; maps
  (use-local-map argile-mode-map)
  (set-syntax-table argile-mode-syntax-table)
  (setq local-abbrev-table argile-mode-abbrev-table)

  ; vars
  (set (make-local-variable 'indent-line-function) 'argile-indent-line)
  (set (make-local-variable 'font-lock-defaults)
       (list 'argile-font-lock-keywords nil nil))
  (set (make-local-variable 'font-lock-string-face)
       'argile-font-lock-string-face)
  (set (make-local-variable 'font-lock-comment-face)
       'argile-font-lock-comment-face)
  (set (make-local-variable 'comment-start) "(:")
  (set (make-local-variable 'comment-end) ":)")

  ; msg
  (message "argile mode loaded")

  ; speedbar
  (argile-speedbar-add-ext)

  ; hooks
  (run-hooks 'argile-mode-hook))
