#!/bin/bash
#############################################################################
# arc-make.sh -- Argile compiler dependency resolving script
#
#   Copyright (C) 2009,2010,2011 the Argile authors
#
#   This software is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation, either version 3 of the License, or
#   (at your option) any later version.
#
#   This software is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this software.  If not, see <http://www.gnu.org/licenses/>.
#
#############################################################################
#
# This script takes a list of argile source files, checks dependencies
# and only rebuilds what is needed.
#
# We use this script instead of Makefile rules, because:
#
# If we update foo.arg, foo.argl will be regenerated,
# its timestamp will therefore be updated, and all
# other *.argl that depends on foo.arg (or foo.argl) will be rebuilt,
# even if the modification is local and does not affect interface
# (e.g. prototypes of functions, types, inclusions, macros, etc.).
#
# Instead, we compare the content of the new foo.argl with the old foo.argl;
# if foo.argl is unchanged, we could avoid the update of its timestamp,
# but then ARC would legitimately warns about foo.argl beeing ignored
# because it is older than foo.arg, and since foo.argl contains C-identifiers
# of exported definitions, wrong identifiers could be used (i.e. GCC fails).
#
# So, we recompile the files that depend on foo.argl only if its content
# changed, but update its timestamp wether it changed or not.
#
# Also, for external dependencies (files not listed in arguments list),
# we keep MD5 checksum of their interfaces (.argl) to determine wether
# a rebuild is needed or not.
#
# Note: We need bash because we use arrays.
#

if [ $# -eq 0 ]; then
    echo "usage: "$(basename "$0")" FILE_1.arg FILE_2.arg ... FILE_n.arg"
    exit 1
fi

P=$(basename "$0"):

[ -z "$ARC" ] && ARC="arc $ARCFLAGS"

function echo_run() {
    echo "$P $*"
    "$@"
}

md5dir=arg_ext_dep_md5/
[ -d $md5dir ] || mkdir $md5dir

mods=()
deps=()
redo=()

echo -n "$P # Checking argile dependencies "
interns='('$(echo "$*"|sed -re 's/(\.arg)? /\\.arg|/g;s/(\.arg)?$/\\.arg/g')')$'
i=0; for mod in $*; do
    echo -n '*'
    #mod=$(echo $mod | sed -e 's/\.arg$//') # basename would remove directory
    mod=${mod%.arg}
    [ ! -e $mod.arg ] && echo && echo "$P # $mod.arg does not exist" && exit 1
    if [ '(' ! -e $mod.p ')' -o $mod.arg -nt $mod.p ]; then
	#echo "$P # Checking $mod.arg"
	err=0
	echo;echo_run $ARC -p $mod.p $mod.arg || err=1
	[ $err = 1 ] && echo "$P # Aborting due to error." && exit 1
    fi
    deps[$i]="$(cut -d: -f2- $mod.p)"
    mods[$i]=$mod
    redo[$i]=0
    if [ '(' ! -e $mod.c ')' -o $mod.arg -nt $mod.c ]; then
	redo[$i]=1
    else # if [ -e $mod.c -a $mod.c -nt $mod.arg ]; then
	# is there any updated external dependency ?
	for ext in $(echo "${deps[$i]}"|tr ' ' '\n'|grep -vE "$interns"); do
	    #ext=$(echo $ext | sed -e 's/\.arg$//')
	    ext=${ext%.arg}
	    if [ -e   $ext.argl			\
		 -a   $ext.argl -nt $mod.c	\
		 -a ! $ext.argl -ot $ext.arg    ]; then
		#extmd5=$md5dir/_$(echo $ext|sed -e 's/\//_/g').argl.md5
		extmd5=$md5dir/_${ext////_}.argl.md5
		if [ -e $extmd5 ]; then
		    if [ $ext.argl -nt $extmd5 ]; then
			[ -e $extmd5.tmp ] || md5sum $ext.argl > $extmd5.tmp
			diff -q $extmd5 $extmd5.tmp >/dev/null || redo[$i]=1
		    fi
		else   # .argl changed, no md5 to check => redo
		    [ -e $extmd5.tmp ] || md5sum $ext.argl > $extmd5.tmp
		    redo[$i]=1
		fi
	    elif [ $ext.arg -nt $mod.c ]; then
		redo[$i]=1
	    fi
	    # [ ${redo[$i]} = 1 ] && break
	    # we don't break anymore to update all md5
	done
    fi
    i=$((i+1))
done
nmod=$i
echo

fpatt="$md5dir/*.md5.tmp"
files=$(echo $fpatt)
if [ "$fpatt" != "$files" ]; then
    for tmp in $files; do
	mv $tmp $md5dir/$(basename $tmp .tmp)
    done
fi

echo "$P # Checking argile rebuilds    "
max_loop=3 need_loop=1; while [ $need_loop = 1 -a $max_loop -gt 0 ] ; do
    need_loop=0
    i=0; for mod in ${mods[*]}; do
	#mod=$(echo $mod | sed -e 's/\.arg$//')
	mod=${mod%.arg}
	if [ ${redo[$i]} = 1 ]; then
	    #echo "$P # Rebuilding from $mod.arg"
	    [ -e $mod.argl ] && echo_run mv $mod.argl $mod.argl.old
	    [ -e $mod.h ] && mv $mod.h $mod.h.old
	    echo_run $ARC -o $mod.c -H $mod.h -m $mod.argl $mod.arg
	    [ $? -ne 0 ] && echo "$P # Aborting due to error." && exit 1

	    # updating .h ?
	    if [ -e $mod.h.old ]; then
		oln=$(wc -l $mod.h.old|cut -d ' ' -f 1)
		head -n $((oln-1)) $mod.h.old | tail -n $((oln-3)) | grep -v ^#line > $mod.h.old.filter
		hln=$(wc -l $mod.h|cut -d ' ' -f 1)
		head -n $((hln-1)) $mod.h | tail -n $((hln-3)) | grep -v ^#line > $mod.h.filter
		if diff -q $mod.h.old.filter $mod.h.filter >/dev/null; then
		    #echo "$P # Keeping old $mod.h"
		    mv $mod.h.old $mod.h
		else
		    #echo "$P # Updating new $mod.h"
		    rm $mod.h.old
		fi
		rm $mod.h.old.filter $mod.h.filter
	    else
		#echo "$P # Creating new $mod.h"
		: # avoid syntax error due to previous line being commented
	    fi
	    redo[$i]=0

	    # .argl changed ?
	    changed=0
	    if [ -e $mod.argl.old ]; then
		changed=1
		if diff -q $mod.argl.old $mod.argl >/dev/null; then
		    #echo "$P # Unchanged $mod.argl"
		    changed=0
		fi
		echo_run rm $mod.argl.old
	    fi
	    if [ $changed = 1 ]; then
		j=0; for m in ${mods[*]}; do
		    #m=$(echo $m | sed -e 's/\.arg$//')
		    m=${m%.arg}
		    if [ $j != $i ]; then
			for d in ${deps[$j]}; do
			    if [ $d -ef $mod.arg ]; then
				#echo "$P # $m needs rebuild"
				redo[$j]=1
				[ $j -lt $i ] && need_loop=1
				break
			    fi
			done
		    fi
		    j=$((j+1))
		done
	    fi
	fi
	i=$((i+1))
    done
    max_loop=$((max_loop-1))
done
