/* parser.y -- grammar Parser of Argile programming language */
/*
 *   Argile programming language compiler
 *
 *   Copyright (C) 2009 the Argile authors
 *
 *   This file is part of ARC Argile compiler.
 *
 *   Argile is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Argile is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Argile.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 ******************************************************************************
 *                                                                            *
 *                             P A R S E R                                    *
 *                                                                            *
 ******************************************************************************
 */

%{
#include <stdlib.h>
#include <stdio.h>

#include "argile.h"

#define YYERROR_VERBOSE 1

#undef yylex
#define yylex argile_yylex_wrapper

extern FILE *argile_yyin;
%}

%union
{
  char               *string;
  argile_text_t      *text;
  argile_list_t      *call_list;
  argile_call_t      *call;
  argile_list_t      *match_list;
  argile_match_t     *match;
  argile_cons_t      *cons;
  argile_list_t      *syn_list;
  argile_syntax_t    *syntax;
  argile_syn_param_t *synparm;
  argile_list_t      *synparm_list;
  argile_syn_list_t  *synlst;
  argile_list_t      *enm_list;
}

%token <string> T_WORD      "word"
%token <string> T_OP        "operator"
%token <string> T_SYN_OP    "syntax operator"
%token <string> T_PAR_OP    "parameter operator"
%token <string> T_INT_HEX   "hexadecimal integer"
%token <string> T_INT_DEC   "decimal integer"
%token <string> T_INT_OCT   "octal integer"
%token <string> T_INT_BIN   "binary integer"
%token <string> T_REAL      "real number"
%token <text>   T_TEXT      "text string"
%token T_SYN_BEG            "syntax beginning"
%token T_SYN_END            "end of syntax"
%token T_CODE_BEG           "code beginning"
%token T_CODE_END           "end of code"
%token T_SUB_BEG            "subcall beginning"
%token T_SUB_END            "end of subcall"
%token T_CALL_END           "end of call"
%token T_PAR_BEG            "parameter beginning"
%token T_PAR_INIT           "parameter initializer"
%token T_PAR_SEP            "parameter separator"
%token T_PAR_END            "end of parameter"
%token T_OPT_BEG            "option beginning"
%token T_OPT_END            "end of option"
%token T_ENM_BEG            "enumeration beginning"
%token T_ENM_SEP            "enumeration separator"
%token T_ENM_END            "end of enumeration"
%token T_LST_BEG            "syntax list beginning"
%token T_LST_SEP            "syntax list separator"
%token T_LST_BND            "syntax list bound separator"
%token T_LST_END            "end of syntax list"
%token T_END                "end of input"

%type <call>         subcall
%type <call_list>    calls
%type <cons>         code syntax
%type <enm_list>     synenm enmlist
%type <match>        callelem parmcallelem parmval
%type <match_list>   call parmcall
%type <synlst>       synlst lstbounds
%type <synparm>      synparm parmcallinit
%type <synparm_list> parmcallinits
%type <syntax>       synelem
%type <syn_list>     synelemlist synopt synparms

%{
 void _set_loc (YYLTYPE *yyloc, argile_loc_t *loc);
%}

%%

argile		: callends
		  {
		    argile_code_append (&argile.main, NULL, NULL);
		  }
		| calls
		  {
		    argile_code_append (&argile.main, NULL,
					argile_list_start ($1));
		  }
		| callends calls
		  {
		    argile_code_append (&argile.main, NULL,
					argile_list_start ($2));
		  }
		| error
		  {
		    argile_die ("parse error at %d:%d-%d:%d",
				@1.first_line, @1.first_column,
				@1.last_line, @1.last_column);
		    YYABORT;
		  }
		;

callend		: T_CALL_END
		| T_END
		;

callends	: callend
		| callends callend
		;

calls		: call callends
                  {
		    argile_call_t *call;
		    call = argile_call_new (NULL, argile_list_start ($1));
		    argile_add_leak (call);
		    argile.numcalls++;
		    _set_loc (&@1, &call->loc);
		    $$ = argile_list_new (call, argile_call_del);
		    argile_add_leak ($$);
		  }
		| calls call callends
                  {
		    argile_call_t *call;
		    call = argile_call_new (NULL, argile_list_start ($2));
		    argile_add_leak (call);
		    argile.numcalls++;
		    _set_loc (&@2, &call->loc);
		    $$ = argile_list_new (call, argile_call_del);
		    argile_add_leak ($$);
		    argile_list_link ($1, $$);
		  }

call		: callelem
                  {
		    $$ = argile_list_new ($1, argile_match_del);
		    argile_add_leak ($$);
		  }
		| call callelem
                  {
		    argile_list_t *lst;

		    lst = argile_list_new ($2, argile_match_del);
		    argile_add_leak (lst);
		    argile_list_link ($1, lst);
		    $$ = lst;
		  }
		;

callelem	: T_WORD
                  {
		    $$ = argile_match_new (ARGILE_MATCH_WORD, $1);
		    argile_add_leak ($$);
		  }
		| T_OP
                  {
		    $$ = argile_match_new (ARGILE_MATCH_OP, $1);
		    argile_add_leak ($$);
		  }
		| T_INT_HEX
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_HEX,
							    $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_INT_DEC
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_DEC,
							    $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_INT_OCT
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_OCT,
							    $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_INT_BIN
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_BIN,
							    $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_REAL
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_REAL,
							    $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_TEXT
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_TEXT,
							    $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| subcall
                  {
		    $$ = argile_match_new (ARGILE_MATCH_SUBCALL, $1);
		    argile_add_leak ($$);
		  }
		| code
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS, $1);
		    argile_add_leak ($$);
		  }
		| syntax
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS, $1);
		    argile_add_leak ($$);
		  }
		;

subcall		: T_SUB_BEG T_SUB_END
                  {
		    $$ = argile_call_new (NULL, NULL);
		    $$->loc.start.row = @1.first_line;
		    $$->loc.start.col = @1.first_column;
		    $$->loc.end.row = @2.last_line;
		    $$->loc.end.col = @2.last_column;
		    argile_add_leak ($$);
		  }
		| T_SUB_BEG call T_SUB_END
                  {
		    $$ = argile_call_new (NULL, argile_list_start ($2));
		    _set_loc (&@2, &$$->loc);
		    argile_add_leak ($$);
		  }
		;

code_start	: T_CODE_BEG
		| T_CODE_BEG callends
		;

code		: code_start T_CODE_END
                  {
		    $$ = argile_cons_new (ARGILE_CONS_CODE,
					  argile_code_new (NULL, NULL));
		    argile_add_leak ($$);
		    argile_add_leak ($$->cvalue.code);
		    if ($$->cvalue.code->input)
		      argile_add_leak ($$->cvalue.code->input);
		  }
                | code_start call T_CODE_END
                  {
		    argile_list_t *calls;
		    argile_call_t *call;
		    call = argile_call_new (NULL, argile_list_start ($2));
		    argile.numcalls++;
		    argile_add_leak (call);
		    _set_loc (&@2, &call->loc);
		    calls = argile_list_new (call, argile_call_del);
		    $$ = argile_cons_new (ARGILE_CONS_CODE,
					  argile_code_new (NULL, calls));
		    argile_add_leak ($$);
		    argile_add_leak ($$->cvalue.code);
		    argile_add_leak ($$->cvalue.code->calls);
		    if ($$->cvalue.code->input)
		      argile_add_leak ($$->cvalue.code->input);
		  }
                | code_start calls T_CODE_END
                  {
		    $$ = argile_cons_new (ARGILE_CONS_CODE,
					  argile_code_new (NULL,
							   argile_list_start ($2)));
		    argile_add_leak ($$);
		    argile_add_leak ($$->cvalue.code);
		    if ($$->cvalue.code->input)
		      argile_add_leak ($$->cvalue.code->input);
		  }
                | code_start calls call T_CODE_END
                  {
		    argile_list_t *lst;
		    argile_call_t *call = argile_call_new (NULL,
							   argile_list_start ($3));
		    argile.numcalls++;
		    argile_add_leak (call);
		    _set_loc (&@3, &call->loc);
		    lst = argile_list_new (call, argile_call_del);
		    argile_add_leak (lst);
		    argile_list_link ($2, lst);
		    $$ = argile_cons_new (ARGILE_CONS_CODE,
					  argile_code_new (NULL,
							   argile_list_start ($2)));
		    argile_add_leak ($$);
		    argile_add_leak ($$->cvalue.code);
		    if ($$->cvalue.code->input)
		      argile_add_leak ($$->cvalue.code->input);
		  }
		;

syntax		: T_SYN_BEG T_SYN_END
		  {
		    $$ = argile_cons_new (ARGILE_CONS_SYNTAX, NULL);
		    argile_add_leak ($$);
		  }
		| T_SYN_BEG synelemlist T_SYN_END
		  {
		    $$ = argile_cons_new (ARGILE_CONS_SYNTAX,
					  argile_list_start ($2));
		    argile_add_leak ($$);
		  }
		;

synelemlist	: synelem
                  {
		    $$ = argile_list_new ($1, argile_syntax_del);
		    argile_add_leak ($$);
		  }
                | synparms
		  {
		    $$ = $1;
		  }
                | synelemlist synparms
		  {
		    argile_list_link ($1, argile_list_start ($2));
		    $$ = $2;
		  }
		| synelemlist synelem
                  {
		    argile_list_t *lst;

		    if ($2)
		      {
			lst = argile_list_new ($2, argile_syntax_del);
			argile_add_leak (lst);
			argile_list_link ($1, lst);
			$$ = lst;
		      }
		    else
		      $$ = $1;
		  }
		;

synelem		: T_WORD
                  {
		    $$ = argile_syntax_new (ARGILE_SYN_WORD, $1);
		    argile_add_leak ($$);
		  }
		| T_SYN_OP
                  {
		    $$ = argile_syntax_new (ARGILE_SYN_OP, $1);
		    argile_add_leak ($$);
		  }
		| synparm
                  {
		    $$ = argile_syntax_new (ARGILE_SYN_PARAM, $1);
		    argile_add_leak ($$);
		  }
		| synopt
                  {
		    $$ = argile_syntax_new (ARGILE_SYN_OPTION, $1);
		    argile_add_leak ($$);
		  }
		| synenm
                  {
		    argile_list_t *lst = $1;
		    if (argile_list_count (lst) < 2)
		      {
			argile_die ("length of syntax enumeration < 2");
			YYERROR;
		      }
		    $$ = argile_syntax_new (ARGILE_SYN_ENUM, lst);
		    argile_add_leak ($$);
		  }
		| synlst
                  {
		    $$ = argile_syntax_new (ARGILE_SYN_LIST, $1);
		    argile_add_leak ($$);
		  }
		;

synparm		: T_PAR_BEG parmcallinit T_PAR_END
                  {
		    $$ = $2;
		  }
		;

synparms        : T_PAR_BEG parmcallinits T_PAR_END
                  {
		    argile_list_t *lst, *params = NULL;
		    for (lst = $2; lst; lst = lst->prev)
		      {
			argile_rm_leak (lst);
			argile_list_prepend (&params,
					     argile_list_new (argile_syntax_new (ARGILE_SYN_PARAM,
										 lst->data.u_anything),
							      argile_syntax_del));
			argile_add_leak (params);
			argile_add_leak (params->data.u_anything);
		      }
		    argile_list_del ($2);
		    $$ = argile_list_end (params);
		  }
                ;

parmcallinit    : parmcall T_PAR_INIT parmval
                  {
		    $$ = argile_syn_param_new (argile_list_start ($1), $3);
		    argile_add_leak ($$);
		  }
                | parmcall
		  {
		    $$ = argile_syn_param_new (argile_list_start ($1), NULL);
		    argile_add_leak ($$);
		  }
                ;

parmcallinits   : parmcallinit T_PAR_SEP parmcallinit
                  {
		    argile_list_t *first = argile_list_new ($1, NULL);
		    argile_add_leak (first);
		    $$ = argile_list_new ($3, NULL);
		    argile_add_leak ($$);
		    argile_list_link (first, $$);
		  }
                | parmcallinits T_PAR_SEP parmcallinit
                  {
		    $$ = argile_list_new ($3, NULL);
		    argile_add_leak ($$);
		    argile_list_link ($1, $$);
		  }
                ;

parmcall	: parmcallelem
                  {
		    $$ = argile_list_new ($1, argile_match_del);
		    argile_add_leak ($$);
		  }
		| parmcall parmcallelem
                  {
		    argile_list_t *lst;

		    lst = argile_list_new ($2, argile_match_del);
		    argile_add_leak (lst);
		    argile_list_link ($1, lst);
		    $$ = lst;
		  }
		;

parmcallelem	: T_WORD
                  {
		    $$ = argile_match_new (ARGILE_MATCH_WORD, $1);
		    argile_add_leak ($$);
		  }
		| T_PAR_OP
                  {
		    $$ = argile_match_new (ARGILE_MATCH_OP, $1);
		    argile_add_leak ($$);
		  }
		| parmval
		  {
		    $$ = $1;
		  }
		;

parmval		: T_INT_HEX
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_HEX, $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_INT_DEC
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_DEC, $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_INT_OCT
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_OCT, $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_INT_BIN
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_BIN, $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_REAL
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_REAL, $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| T_TEXT
                  {
		    $$ = argile_match_new (ARGILE_MATCH_CONS,
					   argile_cons_new (ARGILE_CONS_TEXT, $1));
		    argile_add_leak ($$);
		    argile_add_leak ($$->value.cons);
		  }
		| subcall
		  {
		    $$ = argile_match_new (ARGILE_MATCH_SUBCALL, $1);
		    argile_add_leak ($$);
		  }
		;

synopt		: T_OPT_BEG synelemlist T_OPT_END
                  {
		    $$ = argile_list_start ($2);
		  }
		| T_LST_BEG synelemlist T_LST_END
		  {
		    $$ = argile_list_start ($2);
		  }
		;

synenm		: T_ENM_BEG enmlist T_ENM_END
                  {
		    $$ = argile_list_start ($2);
		  }
		;

enmlist		: synelemlist
                  {
		    $$ = argile_list_new (argile_list_start ($1), argile_list_del);
		    argile_add_leak ($$);
		  }
		| enmlist T_ENM_SEP  synelemlist
                  {
		    argile_list_t *lst;

		    lst = argile_list_new (argile_list_start ($3), argile_list_del);
		    argile_add_leak (lst);
		    argile_list_link ($1, lst);
		    $$ = lst;
		  }
		;

synlst		: T_LST_BEG T_LST_SEP T_LST_END
		  {
		    argile_list_t *ls;
		    argile_syntax_t *s;
		    argile_syn_param_t *sp;

		    sp = argile_syn_param_new (NULL, NULL);
		    sp->type = ARGILE_TYPE_ANYTHING;
		    sp->pseudo = 1;
		    sp->comp = 1;
		    s = argile_syntax_new (ARGILE_SYN_PARAM, sp);
		    ls = argile_list_new (s, argile_syntax_del);
		    $$ = argile_syn_list_new (ls, 0, 0);
		    $$->pseudoempty = 1;
		    argile_add_leak (sp);
		    argile_add_leak (s);
		    argile_add_leak (ls);
		    argile_add_leak ($$);
		  }
		| T_LST_BEG synelemlist T_LST_SEP T_LST_END
                  {
		    $$ = argile_syn_list_new (argile_list_start ($2), 0, 0);
		    argile_add_leak ($$);
		  }
		| T_LST_BEG synelemlist T_LST_SEP lstbounds T_LST_END
                  {
		    argile_syn_list_t *lst;

		    lst = $4;
		    if (lst)
		      lst->sub = argile_list_start ($2);
		    $$ = lst;
		  }
		;

lstbounds	: T_INT_DEC T_LST_BND T_INT_DEC
                  {
		    int min, max;

		    min = atoi ($1);
		    max = atoi ($3);
		    argile_rm_leak ($1);
		    argile_rm_leak ($3);
		    argile_dbg_free ($1);
		    argile_dbg_free ($3);
		    if (!max)
		      {
			argile_die ("null syntax list maximum counter");
			YYERROR;
		      }
		    if (max < min)
		      {
			argile_die ("inverted syntax list min,max counters");
			YYERROR;
		      }
		    $$ = argile_syn_list_new (NULL, min, max);
		    argile_add_leak ($$);
		  }
		| T_INT_DEC T_LST_BND
                  {
		    int min;

		    min = atoi ($1);
		    argile_rm_leak ($1);
		    argile_dbg_free ($1);
		    $$ = argile_syn_list_new (NULL, min, 0);
		    argile_add_leak ($$);
		  }
		| T_INT_DEC
                  {
		    int num;

		    num = atoi ($1);
		    argile_rm_leak ($1);
		    argile_dbg_free ($1);
		    if (!num)
		      {
			argile_die ("null syntax list exact counter");
			YYERROR;
		      }
		    $$ = argile_syn_list_new (NULL, num, num);
		    argile_add_leak ($$);
		  }
		;

%%

int
yyerror (char *msg)
{
  argile_die ("%s", msg);
  return 0;
}

void
_set_loc (YYLTYPE *yyloc, argile_loc_t *loc)
{
  loc->start.row = yyloc->first_line;
  loc->start.col = yyloc->first_column;
  loc->end.row   = yyloc->last_line;
  loc->end.col   = yyloc->last_column;
}
