/* Arg_parser - POSIX/GNU command line argument parser. (C++ version)
   Copyright (C) 2006-2022 Antonio Diaz Diaz.

   This program is free software: you have unlimited permission to
   copy, distribute, and modify it.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/
/*
   Exit status: 0 for a normal exit, 1 for environmental problems
   (file not found, invalid flags, I/O errors, etc), 2 to indicate a
   corrupt or invalid input file, 3 for an internal consistency error
   (e.g., bug) which caused arg_parser to panic.
*/

#include <climits>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>

#include "arg_parser.h"


namespace {

const char * const program_name = "arg_parser";
const char * const program_year = "2022";
const char * invocation_name = program_name;		// default value


void show_help( const bool verbose )
  {
  std::printf( "Arg_parser - POSIX/GNU command line argument parser. (C++ version)\n"
               "\nArg_parser is an argument parser that follows POSIX and GNU conventions for\n"
               "command line arguments. There exist C++ and C versions of Arg_parser. The\n"
               "C++ version is implemented as a C++ class, while the C version is\n"
               "implemented as a single struct plus associated functions. Both are simpler,\n"
               "easier to use, and safer than 'getopt_long'.\n"
               "\nFor maximum stability, Arg_parser is self-contained. It extracts all the\n"
               "information it needs from its arguments to avoid refering to them later.\n"
               "This avoids index-out-of-bounds errors.\n"
               "\nArg_parser does not modify its arguments, nor uses any global variables. So\n"
               "you may create more than one parser in your program if you need or want to.\n"
               "\nThe C++ version of Arg_parser can also parse options from configuration\n"
               "files.\n"
               "\nTo use Arg_parser in your own programs simply copy the files 'arg_parser.h'\n"
               "and 'arg_parser.cc' in your source tree. See the file 'main.cc' for an\n"
               "example of use.\n"
               "\nUsage: %s [options]\n", invocation_name );
  std::printf( "\nOptions:\n"
               "  -h, --help                   display this help and exit\n"
               "  -V, --version                output version information and exit\n"
               "  -a, --append                 example of option with no argument\n"
               "  -b, --block=<arg>            example of option with required argument\n"
               "  -c, --casual[=<arg>]         example of option with optional argument\n"
               "  -o <arg>                     example of short only option\n"
               "      --orphan                 example of long only option\n"
               "  -q, --quiet                  quiet operation\n"
               "  -u, --uncaught               example of intentional bug\n"
               "  -v, --verbose                verbose operation\n" );
  if( verbose )
    std::printf( "  -H, --hidden                 example of hidden option (shown with -v -h)\n" );
  std::printf( "\nReport bugs to arg-parser-bug@nongnu.org\n"
               "Arg_parser home page: http://www.nongnu.org/arg-parser/arg_parser.html\n" );
  }


void show_version()
  {
  std::printf( "%s %s\n", program_name, PROGVERSION );
  std::printf( "Copyright (C) %s Antonio Diaz Diaz.\n", program_year );
  std::printf( "License 2-clause BSD.\n"
               "This is free software: you are free to change and redistribute it.\n"
               "There is NO WARRANTY, to the extent permitted by law.\n" );
  }


void show_error( const char * const msg, const int errcode = 0,
                 const bool help = false )
  {
  if( msg && msg[0] )
    std::fprintf( stderr, "%s: %s%s%s\n", program_name, msg,
                  ( errcode > 0 ) ? ": " : "",
                  ( errcode > 0 ) ? std::strerror( errcode ) : "" );
  if( help )
    std::fprintf( stderr, "Try '%s --help' for more information.\n",
                  invocation_name );
  }


void internal_error( const char * const msg )
  {
  std::fprintf( stderr, "%s: internal error: %s\n", program_name, msg );
  std::exit( 3 );
  }

} // end namespace


int main( const int argc, const char * const argv[] )
  {
  bool verbose = false;
  if( argc > 0 ) invocation_name = argv[0];

  const Arg_parser::Option options[] =
    {
    { 'a', "append",   Arg_parser::no    },
    { 'b', "block",    Arg_parser::yes   },
    { 'c', "casual",   Arg_parser::maybe },
    { 'h', "help",     Arg_parser::no    },
    { 'H', "hidden",   Arg_parser::no    },
    { 'o', 0,          Arg_parser::yes   },
    { 'q', "quiet",    Arg_parser::no    },
    { 'u', "uncaught", Arg_parser::no    },
    { 'v', "verbose",  Arg_parser::no    },
    { 'V', "version",  Arg_parser::no    },
    { 256, "orphan",   Arg_parser::no    },
    { INT_MIN, "int-min", Arg_parser::no },
    { INT_MAX, "int-max", Arg_parser::no },
    {   0, 0,          Arg_parser::no    } };

  const Arg_parser parser( argc, argv, options );
  if( parser.error().size() )				// bad option
    { show_error( parser.error().c_str(), 0, true ); return 1; }

  for( int argind = 0; argind < parser.arguments(); ++argind )
    {
    const int code = parser.code( argind );
    if( !code ) break;				// no more options
    switch( code )
      {
      case 'a': break;				// example, do nothing
      case 'b': break;				// example, do nothing
      case 'c': break;				// example, do nothing
      case 'h': show_help( verbose ); return 0;
      case 'H': break;				// example, do nothing
      case 'o': break;				// example, do nothing
      case 'q': verbose = false; break;
      // case 'u': break;			// intentionally not caught
      case 'v': verbose = true; break;
      case 'V': show_version(); return 0;
      case 256: break;				// example, do nothing
      case INT_MIN: break;			// example, do nothing
      case INT_MAX: break;			// example, do nothing
      default : internal_error( "uncaught option." );
      }
    } // end process options

  for( int argind = 0; argind < parser.arguments(); ++argind )
    {
    const int code = parser.code( argind );
    const char * const arg = parser.argument( argind ).c_str();
    if( code )	// option
      {
      std::printf( "option '%s'", parser.parsed_name( argind ).c_str() );
      if( arg[0] ) std::printf( " with argument '%s'", arg );
      if( code == INT_MIN || code == INT_MAX ) std::printf( " (code 0x%X)", code );
      }
    else	// non-option
      std::printf( "non-option argument '%s'", arg );
    std::fputc( '\n', stdout );
    }

  if( !parser.arguments() ) std::fputs( "Hello, world!\n", stdout );

  return 0;
  }
