# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman, Enno Cramer
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.006;
use strict;
use warnings;

package ArchWay::Widget::Inventory::Directory;

use Glib qw(TRUE FALSE);
use Gtk2;

use base qw(Gtk2::TreeView);

use Arch::Inventory qw(:category :type);
use Arch::Changes qw(:type);
use POSIX qw(strftime);

use constant MODEL_COL_TYPES => qw(
	Glib::String
	Glib::String
	Glib::String
	Glib::String
	Glib::String
	Glib::String
	Glib::String
	Glib::String
);

use constant MODEL_COL_TYPE     => 0;
use constant MODEL_COL_CATEGORY => 1;
use constant MODEL_COL_NAME     => 2;
use constant MODEL_COL_STATUS   => 3;
use constant MODEL_COL_SIZE     => 4;
use constant MODEL_COL_MODE     => 5;
use constant MODEL_COL_MODIFIED => 6;
use constant MODEL_COL_PATH     => 7;

my @VIEW_COLUMNS = (
	{
		title  => 'Filename',
		renderers => [
			{
				class   => 'Gtk2::CellRendererPixbuf',
				static  => {},
				dynamic => { stock_id => MODEL_COL_TYPE },
			},
			{
				class   => 'Gtk2::CellRendererText',
				static  => {},
				dynamic => { text => MODEL_COL_NAME },
			},
		],
		sort_column => MODEL_COL_NAME,
	},
	{
		title  => 'Category',
		renderer => {
			class   => 'Gtk2::CellRendererText',
			static  => {},
			dynamic => { text => MODEL_COL_CATEGORY },
		},
		sort_column => MODEL_COL_CATEGORY,
	},
	{
		title  => 'Status',
		renderer => {
			class   => 'Gtk2::CellRendererText',
			static  => {},
			dynamic => { text => MODEL_COL_STATUS },
		},
		sort_column => MODEL_COL_STATUS,
	},
	{
		title  => 'Size',
		renderer => {
			class   => 'Gtk2::CellRendererText',
			static  => { xalign => 1.0 },
			dynamic => { markup => MODEL_COL_SIZE },
		},
		sort_column => MODEL_COL_SIZE,
	},
	{
		title  => 'Mode',
		renderer => {
			class   => 'Gtk2::CellRendererText',
			static  => {},
			dynamic => { markup => MODEL_COL_MODE },
		},
	},
	{
		title  => 'Last Modified',
		renderer => {
			class   => 'Gtk2::CellRendererText',
			static  => {},
			dynamic => { markup => MODEL_COL_MODIFIED },
		},
	}
);

my %CATEGORY_NAMES = (
	TREE()         => 'Arch Tree Root',
	SOURCE()       => 'Source',
	PRECIOUS()     => 'Precious',
	BACKUP()       => 'Backup',
	JUNK()         => 'Junk',
	UNRECOGNIZED() => 'Unrecognized',
);

my %TYPE_NAMES = (
	FILE()      => 'F',
	DIRECTORY() => 'D',
	SYMLINK()   => 'L',
);

sub new ($$) {
	my $class     = shift;
	my $inventory = shift;

	my $self = bless Gtk2::TreeView->new(), $class;
	$self->set_model(Gtk2::ListStore->new(MODEL_COL_TYPES));

	$self->_init();
	$self->show($inventory) if ($inventory);

	return $self;
}

sub _init ($) {
	my $self = shift;

	$self->set_rules_hint(TRUE);
	$self->set_search_column(MODEL_COL_NAME);

	foreach my $col (@VIEW_COLUMNS) {
		my $column = Gtk2::TreeViewColumn->new;
		$column->set_title($col->{title});

		foreach my $rdr ($col->{renderer} || @{$col->{renderers}}) {
			my $renderer = Gtk2::Object->new(
				$rdr->{class},
				%{$rdr->{static}},
			);

			$column->pack_start($renderer, TRUE);
			$column->set_attributes($renderer, %{$rdr->{dynamic}});
		}

		$column->set_resizable(TRUE);
		$column->set_sort_column_id($col->{sort_column})
			if exists $col->{sort_column};

		$self->append_column($column);
	}
}

sub get ($$;@) {
	my $self = shift;
	my $iter = shift;

	return $self->get_model()->get($iter, @_);
}

sub get_selected ($;@) {
	my $self = shift;

	my $iter = $self->get_selection()->get_selected();

	return $iter ? $self->get($iter, @_) : undef;
}

sub get_selected_path ($) {
	my $self = shift;

	return $self->get_selected(MODEL_COL_PATH);
}

sub select_by_path ($$) {
	my $self = shift;
	my $path = shift;

	my $model = $self->get_model();
	my $iter  = $model->get_iter_first();
	while (defined $iter) {
		my $ipath = $model->get($iter, MODEL_COL_PATH);

		if ($ipath eq $path) {
			$self->get_selection()->select_iter($iter);
			last;
		} else {
			$iter = $model->iter_next($iter);
		}
	}
}

sub show ($$$) {
	my $self      = shift;
	my $inventory = shift;
	my $parent    = shift;

	my $model = $self->get_model();
	$model->clear();

	return unless (defined $inventory);

	list_model_append($model, '.',  $inventory);
	list_model_append($model, '..', $parent)
		if $parent;

	foreach my $name (sort keys %{$inventory->{children}}) {
		list_model_append($model, $name, $inventory->{children}->{$name});
	}

	$self->get_selection->select_iter($self->get_model->iter_nth_child(undef, 0));	
}

sub inventory_entry_status ($) {
	my $entry = shift;

	my $status;
	if (exists $entry->{changes}) {
		if (exists $entry->{changes}->{ADD()}) {
			$status = 'New';
		} elsif (exists $entry->{changes}->{REMOVE()}) {
			$status = 'Removed';
		} else {
			$status = 'Modified';
		}
	} elsif ($entry->{category} eq SOURCE) {
		$status = 'Unmodified';
	} else {
		$status = 'Not arch controlled';
	}

	return $status;
}

sub list_model_append ($$$) {
	my ($model, $name, $entry) = @_;

	my $stockid;

	if ($entry->{type} eq DIRECTORY) {
		$stockid = 'gtk-directory';
	} elsif ($entry->{type} eq SYMLINK) {
		$stockid = 'gtk-redo';
	} else {
		$stockid = 'gtk-file';
	}

	my $iter = $model->append;
	$model->set($iter,
		MODEL_COL_TYPE,     $stockid,
		MODEL_COL_CATEGORY, $CATEGORY_NAMES{$entry->{category}},
		MODEL_COL_NAME,     $name,
		MODEL_COL_STATUS,   inventory_entry_status($entry),
		MODEL_COL_PATH,     $entry->{path},
	);

	if ($entry->{stat}) {
		$model->set($iter,
			MODEL_COL_SIZE,     $entry->{stat}->[7],
			MODEL_COL_MODE,     mode_string($entry->{stat}->[2]),
			MODEL_COL_MODIFIED, strftime('%c', localtime($entry->{stat}->[9])),
		);
	} else {
		$model->set($iter,
			MODEL_COL_SIZE,     '<i>unknown</i>',
			MODEL_COL_MODE,     '<i>unknown</i>',
			MODEL_COL_MODIFIED, '<i>unknown</i>',
		);
	}
}

sub mode_string ($) {
	my $mode = shift;

	my @modes = qw(--- --x -w- -wx r-- r-x rw- rwx);
	my %types = ( 4 => 'd', 8 => '-', 10 => 'l' );

	return ($types{$mode >> 12} || '?') .
		join('', map { $modes[($mode >> 3*$_) & 7] } (2, 1, 0));
}

1;

__END__
