# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package Arch::Session;

use base 'Arch::Storage';

use Arch::Util qw(run_tla _parse_revision_details);
use Arch::TempFiles qw(temp_dir_name);
use Arch::Changeset;
use Arch::Log;
use Arch::Tree;

sub new ($%) {
	my $class = shift;
	my %init = @_;
	my $self = $class->SUPER::new(%init);
	$self->clear_cache;
	return $self;
}

sub archives ($) { 
	my $self = shift;
	$self->{archives} ||= [ run_tla("archives -n") ];
	return $self->{archives};
}
 
sub is_archive_registered ($;$) {
	my $self = shift;
	my $archive = @_? shift: $self->{archive};
	die "No working archive" unless defined $archive;

	unless ($self->{archives_presence}) {
		my $archives_hash = {};
		$archives_hash->{$_} = 1 foreach @{$self->archives};
		$self->{archives_presence} = $archives_hash;
	}
	return $self->{archives_presence}->{$archive};
}

sub categories ($;$) {
	my $self = shift;
	my $archive  = @_ > 0? shift: $self->{archive};
	die "No working archive" unless defined $archive;

	unless ($self->{categories}->{$archive}) {
		$self->{categories}->{$archive} = [ run_tla("categories", $archive) ];
	}
	return $self->{categories}->{$archive};
}

sub branches ($;$$) {
	my $self = shift;
	my $archive  = @_ > 1? shift: $self->{archive};
	my $category = @_ > 0? shift: $self->{category};
	die "No working archive" unless defined $archive;
	die "No working category" unless defined $category;

	my $full_category =
		$archive
		. '/' . $category;

	unless ($self->{branches}->{$full_category}) {
		$self->{branches}->{$full_category} = [ run_tla("branches", $full_category) ];
	}
	return $self->{branches}->{$full_category};
}

sub versions ($;$$$) {
	my $self = shift;
	my $archive  = @_ > 2? shift: $self->{archive};
	my $category = @_ > 1? shift: $self->{category};
	my $branch   = @_ > 0? shift: $self->{branch};
	$branch = "" unless defined $branch;  # support branchless revisions
	die "No working archive"  unless defined $archive;
	die "No working category" unless defined $category;
	die "No working branch"   unless defined $branch;

	my $full_branch =
		$archive
		. '/' . $category
		. ($branch ne '' ? '--' : '') . $branch;

	unless ($self->{versions}->{$full_branch}) {
		$self->{versions}->{$full_branch} = [ run_tla("versions", $full_branch) ];
		$self->{versions}->{$full_branch} = [ map { s/--/----/; $_ } grep !/--.*--/, @{$self->{versions}->{$full_branch}} ]
			if $branch eq '';
	}
	return $self->{versions}->{$full_branch};
}

sub revisions ($;$$$$) {
	my $self = shift;
	my $archive  = @_ > 3? shift: $self->{archive};
	my $category = @_ > 2? shift: $self->{category};
	my $branch   = @_ > 1? shift: $self->{branch};
	my $version  = @_ > 0? shift: $self->{version};
	die "No working archive"  unless defined $archive;
	die "No working category" unless defined $category;
	die "No working branch"   unless defined $branch;
	die "No working version"  unless defined $version;

	my $full_version =
		$archive 
		. '/' . $category
		. ($branch ne '' ? '--' : '') . $branch
		. '--' . $version;

	unless ($self->{revisions}->{$full_version}) {
		$self->{revisions}->{$full_version} = [ run_tla("revisions", $self->working_name) ];
	}
	return $self->{revisions}->{$full_version};
}

sub revision_details ($;$$$$) {
	my $self = shift;
	my $archive  = @_ > 3? shift: $self->{archive};
	my $category = @_ > 2? shift: $self->{category};
	my $branch   = @_ > 1? shift: $self->{branch};
	my $version  = @_ > 0? shift: $self->{version};
	die "No working archive"  unless defined $archive;
	die "No working category" unless defined $category;
	die "No working branch"   unless defined $branch;
	die "No working version"  unless defined $version;

	my $full_version =
		$archive 
		. '/' . $category
		. ($branch ne '' ? '--' : '') . $branch
		. '--' . $version;

	unless ($self->{revision_details}->{$full_version}) {
		my $version = $full_version;
		$version =~ s|^.*/||;

		# $ok is used to work around the tla bug with branchless version
		# $prev_line is used to track revisions with no (empty) summary
		my $ok = 0;
		my $prev_line = "";

		my @revision_lines = map { s/^        //? $_: undef }
			grep {
				$ok = /^      \Q$version\E$/ if /^      [^ ]/;
				my $end = ($prev_line =~ /^        /) && ($_ eq "");
				$prev_line = $_;
				($end || /^        /) && $ok
			}
			run_tla("abrowse --desc", $full_version);

		my $revision_details = _parse_revision_details(2, \@revision_lines);
		$self->{revision_details}->{$full_version} = $revision_details;
		$self->{revisions}->{$full_version} = [ map { $_->{name} } @$revision_details ];
	}
	return $self->{revision_details}->{$full_version};
}

sub clear_cache ($) {
	my $self = shift;

	$self->{archives} = undef;
	$self->{categories} = {};
	$self->{branches} = {};
	$self->{versions} = {};
	$self->{revisions} = {};
	$self->{revision_details} = {};
	$self->{missing_revisions} = {};
	$self->{missing_revision_details} = {};
}

# [
#   [ category1, [
#     [ branch1, [
#       [ version1, start_revision1, end_revision1 ],
#       [ version2, start_revision2, end_revision2 ],
#     ] ],
#     [ branch2, [
#       [ version3, start_revision3, end_revision3 ],
#       [ version4, start_revision4, end_revision4 ],
#     ] ],
#     ...,
#   ] ],
# ]

sub expanded_archive_info ($;$$) {
	my $self = shift;
	die "expanded_archive_info: no working archive\n" unless defined $self->{archive};
	my $archive_name = $self->working_name || shift;
	my $full_listing = shift || 0;  # currently ignored

	my $infos = [];
	my @category_infos = split(/^\b/m, join('',
		map { s/^  //; "$_\n" } grep { /^  / }
			run_tla("abrowse $archive_name")
	));

	my $error = 0;
	CATEGORY_ITEM:
	foreach (@category_infos) {
		my ($category, $branch_infos) = /^([^\s]+)\n(  .*)$/s;
		push @$infos, [ $category, [] ];
		unless (defined $category) {
			$error = 1; next CATEGORY_ITEM;
		}

		my @branch_infos = split(/^\b/m, join('',
			map { s/^  // or $error = 1; "$_\n" }
				split("\n", $branch_infos)
		));
		$error = 1 unless @branch_infos;
		foreach (@branch_infos) {
			my ($branch, $version_infos) = /^\Q$category\E(?:--([^\s]+))?\n(  .*)$/s;
			$branch = "" if defined $version_infos && !defined $branch;
			unless (defined $branch) {
				$error = 1; next CATEGORY_ITEM;
			}
			push @{$infos->[-1]->[1]}, [ $branch, [] ];

			my @version_infos = split(/^\b/m, join('',
				map { s/^  // or $error = 1; "$_\n" }
					split("\n", $version_infos)
			));
			$error = 1 unless @version_infos;
			foreach (@version_infos) {
				my ($version, $revision0, $revisionl) = /^\Q$category\E(?:--)?\Q$branch\E--([^\s]+)(?:\n  ([^\s]+)(?: \.\. ([^\s]+))?\n)?$/s;
				unless (defined $version) {
					$error = 1; next CATEGORY_ITEM;
				}
				# TODO: consider $full_listing here
				$revision0 = '' unless defined $revision0;
				$revisionl = '' unless defined $revisionl;
				push @{$infos->[-1]->[1]->[-1]->[1]}, [ $version, $revision0, $revisionl ];
			}
		}
	} continue {
		if ($error) {
			warn "Unexpected abrowse output, skipping:\n$_\n";
			pop @$infos;
			$error = 0;
		}
	}
	return $infos;
}

sub get_revision_changeset ($$;$) {
	my $self = shift;
	my $revision = shift;
	my $dir = defined $_[0]? shift: temp_dir_name("arch-changeset");
	die "get_changeset: incorrect dir ($dir)\n" unless $dir && !-d $dir;

	run_tla("get-changeset", $revision, $dir);
	return Arch::Changeset->new($revision, $dir);
}

sub get_changeset ($;$) {
	my $self = shift;
	my $dir = shift;
	my $full_revision = $self->working_name;
	return $self->get_revision_changeset($full_revision, $dir);
}

sub get_revision_log ($$) {
	my $self = shift;
	my $revision = shift || die "get_revision_log: No revision given\n";
	my $message = run_tla("cat-archive-log", $revision);
	die "Can't get log of $revision from archive.\n"
		. "Unexisting revision or system problems.\n"
		unless $message;
	return Arch::Log->new($message);
}

sub get_log ($) {
	my $self = shift;
	die "get_log: no working revision\n" unless defined $self->{revision};
	return $self->get_revision_log($self->working_name);
}

sub get_tree ($;$$) {
	my $self = shift;
	my $revision = shift || $self->working_name || die "get_tree: no c/b/r\n";
	my $dir = shift || temp_dir_name("arch-tree");
	die "get_tree: incorrect dir ($dir)\n" unless $dir && !-d $dir;

	run_tla("get --silent --no-pristine", $revision, $dir);
	die "Can't get revision $revision from archive.\n"
		. "Unexisting revision or system problems.\n"
		unless -d $dir;
	return Arch::Tree->new($dir);
}

sub init_tree ($$;$) {
	my $self = shift;
	my $version = shift || $self->working_name || die "init_tree: no version\n";
	my $dir = shift || ".";

	run_tla("init-tree", "-d", $dir, $version);
	return undef unless $? == 0;
	return Arch::Tree->new($dir);
}

sub my_id ($;$) {
	my $self = shift;
	my $userid = shift;

	if (defined $userid) {
		return 0 unless $userid =~ /<.+\@.*>/;
		run_tla("my-id", $userid);
		return !$?;
	} else {
		($userid) = run_tla("my-id");
		return $userid;
	}
}

1;
