/*  Begin sun.cpp  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "sun.h"

/*  local includes  */
#include "include/constant.h"

/*  graphic lib  */
extern "C"
{
#include <GL/gl.h>
#include <GL/glut.h>
}

/*  C lib  */
extern "C"
{
#include <math.h>
}


/****************  public functions  ****************/


Sun::Sun(GLenum light,       /*  light to which the sun is connected       */
	 float distance,     /*  distance from the origin, in meters       */
	 float day_length,   /*  in seconds                                */
	 float latitude,     /*  latitude of the observer, in degrees      */
	 float day_of_year)  /*  [0, 2pi[, in radians, 0 is march equinox  */ 
{
  /*  light parameters  */
  GLfloat light_diffuse[] = { 1.0, 0.9, 0.8, 1.0 };
  GLfloat light_specular[] = { 1.0, 0.9, 0.8, 1.0 };
  GLfloat light_ambient[] = { 0.3, 0.3, 0.3, 1.0 };

  this->distance = distance;
  this->radius_set(distance);
  this->angular_speed = Constant_pi / day_length;
  this->position_base_set(latitude, day_of_year);

  /*  sets this->position  */
  this->position[3] = 0.0;  /*  directionnal light  */
  this->set_position(0.0);  /*  set at time 0  */

  /*  sets light parameters  */
  glLightfv(light, GL_DIFFUSE, light_diffuse);
  glLightfv(light, GL_SPECULAR, light_specular);
  glLightfv(light, GL_AMBIENT, light_ambient);

  glEnable(light);

  this->light = light;
}


Sun::~Sun(void)
{
  delete[] this->position;
}


/****  set  ****/

/*  "time" in seconds  */
void
Sun::set_position(float time)
{
  /*  simulates sun race  */

  float phi;
  float cos_phi, sin_phi;

  phi = time * this->angular_speed;
  /*  skips the night  */
  if (cosf(phi) < 0.0)
    {
      phi -= Constant_pi;
    }
  cos_phi = cosf(phi);
  sin_phi = sinf(phi);

  /*  position is normalized  */
  /*  west  */
  this->position[0] = this->position_base_x_mult * sin_phi;
  /*  up  */
  this->position[1] =
    this->position_base_y_mult * cos_phi + this->position_base_y_add;
  /*  north  */
  this->position[2] =
    this->position_base_z_mult * cos_phi + this->position_base_z_add;

  glLightfv(this->light, GL_POSITION, this->position);

  this->position[0] *= this->distance;
  this->position[1] *= this->distance;
  this->position[2] *= this->distance;
}


/****  get  ****/

/*  fills three elements vector "position"  */
void
Sun::get_position(float *position) const
{
  int i;

  for (i = 0; i < 3; i++)
    {
      position[i] = this->position[i];
    }
}


float
Sun::get_radius(void) const
{
  return this->radius;
}


/****************  protected functions  ****************/


void
Sun::radius_set(float distance)
{
  this->radius =
    tanf(Constant_sun_angular_diameter / 2.0 * Constant_deg_to_rad) * distance;
}


void
Sun::position_base_set(float latitude, float day_of_year)
{
  float alpha, psi, theta;
  float cos_alpha, sin_alpha;
  float cos_psi, sin_psi, tan_psi;
  float cos_theta, sin_theta;
  float sec_phi;

  alpha = Constant_ecliptic_obliquity * Constant_deg_to_rad;
  psi = day_of_year;
  theta = latitude * Constant_deg_to_rad;

  cos_alpha = cosf(alpha);
  sin_alpha = sinf(alpha);
  cos_psi = cosf(psi);
  sin_psi = sinf(psi);
  tan_psi = tanf(psi);
  cos_theta = cosf(theta);
  sin_theta = sinf(theta);
  sec_phi = sqrtf(1 + powf(tan_psi * cos_alpha, 2));

  /*  west  */
  this->position_base_x_mult = cos_psi * sec_phi;
  /*  up  */
  this->position_base_y_mult = cos_theta * cos_psi * sec_phi;
  this->position_base_y_add = sin_alpha * sin_theta * sin_psi;
  /*  north  */
  this->position_base_z_mult = - sin_theta * cos_psi * sec_phi;
  this->position_base_z_add = sin_alpha * cos_theta * sin_psi;
}


/*  End sun.cp  */
