/*  Emacs mode: -*- C++ -*-  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


#ifndef SCENE_H
#define SCENE_H


/****************  includes  ****************/


/*  C++ lib  */
#include <string>


/****************  namespaces  ****************/


using namespace std;


/****************  abstract classes  ****************/


class Scene
{
 public:

  Scene(int points_x_index,     /*  number of points (must be powers of two)  */
	int points_z_index,
	int size_x_index,               /*  surface size, meters  */
	int size_z_index,
	int depth_index,                /*  depth of water, meters  */
	int displacement_factor_index,  /*  horizontal displacement factor  */
	int surface_alpha_index,        /*  OpenGL alpha value (opacity)  */
	bool is_wired,                  /*  is surface drawn wired?  */
	bool is_normals_draw,           /*  are normals drawn?  */
	bool is_tiled,                  /*  is surface tiled?  */
	bool is_stone_draw);            /*  is stone drawn?  */

  virtual ~Scene(void) = 0;


  /****  set  ****/

  /*  time  */
  void set_time(float time);
  /*  points  */
  void points_decrease(void);
  void points_increase(void);
  void points_x_decrease(void);
  void points_x_increase(void);
  void points_z_decrease(void);
  void points_z_increase(void);
  /*  size  */
  void size_decrease(void);
  void size_increase(void);
  void size_x_decrease(void);
  void size_x_increase(void);
  void size_z_decrease(void);
  void size_z_increase(void);
  /*  depth  */
  void depth_decrease(void);
  void depth_increase(void);
  /*  displacement_factor  */
  void displacement_factor_decrease(void);
  void displacement_factor_increase(void);
  /*  surface_alpha  */
  void surface_alpha_decrease(void);
  void surface_alpha_increase(void);
  /*  booleans  */
  void is_wired_toggle(void);
  void is_normals_draw_toggle(void);
  void is_tiled_toggle(void);
  void is_stone_draw_toggle(void);


  /****  get  ****/

  int get_points_x(void) const;
  int get_points_z(void) const;
  float get_size_x(void) const;
  float get_size_z(void) const;
  float get_resolution_x(void) const;
  float get_resolution_z(void) const;
  float get_depth(void) const;
  float get_displacement_factor(void) const;
  float get_surface_alpha(void) const;
  bool get_is_wired(void) const;
  bool get_is_normals_draw(void) const;
  bool get_is_tiled(void) const;
  bool get_is_stone_draw(void) const;
  float get_whole_size_x(void) const;
  float get_whole_size_z(void) const;
  /*  returns surface height at given index  */
  float get_surface_height(int index_x, int index_z) const;
  float ***get_field_surface(void) const;
  float ***get_field_normals(void) const;
  void get_string(string *str) const;


 protected:

  /*  values arrays  */
  const int * const points_x_array;
  const int * const points_z_array;
  const float * const size_x_array;
  const float * const size_z_array;
  const float * const depth_array;
  const float * const displacement_factor_array;
  const float * const surface_alpha_array;
  
  /*  arrays sizes  */
  const int points_x_array_size;
  const int points_z_array_size;
  const int size_x_array_size;
  const int size_z_array_size;
  const int depth_array_size;
  const int displacement_factor_array_size;
  const int surface_alpha_array_size;

  /*  arrays indexes  */
  int points_x_index;
  int points_z_index;
  int size_x_index;
  int size_z_index;
  int depth_index;
  int displacement_factor_index;
  int surface_alpha_index;

  bool is_wired;         /*  is surface drawn wired?  */
  bool is_normals_draw;  /*  are normals drawn?  */
  bool is_tiled;         /*  is surface tiled?  */
  bool is_stone_draw;    /*  is stone drawn?  */

  float whole_size_x;  /*  whole scene size, altered by "tiled", in meters  */
  float whole_size_z;
  /*  surface fields  */
  float ***field_surface;
  float ***field_normals;
  class Aqua_Surface *surface;

  bool index_decrease(int &index);
  bool index_increase(int &index, int array_size);

  /****  global changes  ****/
  void global_change_points(bool is_points_changed,
			    bool is_wired,
			    bool is_normals_draw);
  void global_change_size(bool is_size_changed,
			  bool tiled,
			  float size_x,
			  float size_z,
			  bool is_wired,
			  bool is_normals_draw);
  void global_change_depth(bool is_depth_changed,
			   float depth,
			   class Aqua_Surface *surface,
			   bool is_wired,
			   bool is_normals_draw);
  void global_change_displacement_factor(bool is_displacement_factor_changed,
					 float displacement_factor,
					 class Aqua_Surface *surface,
					 bool is_wired,
					 bool is_normals_draw);

  /****  others  ****/

  void fields_compute(bool is_wired, bool is_normals_draw);
  void whole_size_set(bool is_tiled, float size_x, float size_z);
  virtual void surface_recreate(void) = 0;

  /*  copy constructor to be defined  */
  Scene(const class Scene &);
};


#endif  /*  SCENE_H  */
