/*  Begin draw.cpp  */

/*  OpenGL drawing functions  */

/*
  Copyright (C) 2003  Jocelyn Frchot

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/****************  includes  ****************/


#include "draw.h"

#include "draw_surface.h"
#include "scene.h"
#include "sun.h"

/*  graphic lib  */
extern "C"
{
#include <GL/gl.h>
#include <GL/glut.h>
}

/*  C++ lib  */
#include <string>


/****************  namespaces  ****************/


using namespace std;


/****************  static functions prototypes  ****************/


namespace
{
  /****  floor  ****/
  void draw_floor(float size_x, float size_z, float depth);
  /****  stone  ****/
  void draw_stone(float coord_x,
		  float coord_z,
		  float depth,
		  int height,
		  float width);
  void draw_square_centered(float half_width);
}


/****************  functions  ****************/


void
draw(class Scene *scene)
{
  float ***surface;
  float ***normals;


  surface = scene->get_field_surface();
  normals = scene->get_field_normals();

  /*  floor  */
  if (scene->get_depth() != 0)
    {
      draw_floor(scene->get_whole_size_x(),
		 scene->get_whole_size_z(),
		 scene->get_depth());
    }

  /*  stone  */
  if (scene->get_is_stone_draw())
    {
      float stone_x, stone_z;
      float stone_depth;
      int stone_height;
      float stone_width;

      stone_x = scene->get_whole_size_x() / 2.0;
      stone_z = scene->get_whole_size_z() - scene->get_size_z() / 2.0;
      stone_height = 2;
      stone_width = 1;
      if (scene->get_depth() == 0)
	{
	  /*  no floor  */
	  stone_depth = 10;
	}
      else
	{
	  stone_depth = scene->get_depth();
	}
      draw_stone(stone_x, stone_z, stone_depth, stone_height, stone_width);
    }

  /*  surface */
  /*  drawn  last because of transparency  */
  if (scene->get_is_wired())
    {
      draw_surface_wire(scene->get_points_x(),
			scene->get_points_z(),
			scene->get_size_x(),
			scene->get_size_z(),
			scene->get_resolution_x(),
			scene->get_resolution_z(),
			scene->get_is_normals_draw(),
			scene->get_is_tiled(),
			surface,
			normals);
    }
  else
    {
      draw_surface_color(scene->get_points_x(),
			 scene->get_points_z(),
			 scene->get_size_x(),
			 scene->get_size_z(),
			 scene->get_surface_alpha(),
			 scene->get_is_tiled(),
			 surface,
			 normals);
    }
}


void
draw(class Sun *sun)
{
  float radius;

  radius = sun->get_radius();

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  {
    glDisable(GL_LIGHTING);
    glDisable(GL_CULL_FACE);
    /*  Sun color. Alpha value is for the corona.  */
    glColor4f(1.0, 0.9, 0.0, 0.3);
    /*  sun  */
    glutSolidSphere(radius, 20, 20);
    /*  corona  */
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_BLEND);
    glutSolidSphere(radius * 1.5, 20, 20);
  }
  glPopAttrib();
}


/*  draws "string" at given screen position  */
void
draw_string(const string *string, int position_x, int position_y)
{
  const char *str;
  int line;
  int i;

  str = string->data();
  line = 0;
  i = 0;

  glPushAttrib(GL_ALL_ATTRIB_BITS);
  {
    glDisable(GL_LIGHTING);
    glDisable(GL_DEPTH_TEST);
    glShadeModel(GL_FLAT);
    glColor3f(1.0, 1.0, 0.0);

    /*  must be called after "glColor"  */
    glRasterPos2i(position_x, position_y);

    while (str[i] != '\0')
      {
	if (str[i] == '\n')
	  {
	    line++;
	    glRasterPos2i(position_x, position_y + 12 * line);
	    /*  doesn't print this character  */
	    i++;
	    continue;
	  }
	glutBitmapCharacter(GLUT_BITMAP_HELVETICA_12, str[i]);
	i++;
      }
  }
  glPopAttrib();
}


/****************  static functions  ****************/


namespace
{


/****  floor  ****/

void
draw_floor(float size_x, float size_z, float depth)
{
  GLfloat mat_diffuse[] = { 0.5, 0.8, 0.8, 1.0 };
  GLfloat mat_ambient[] = { 0.0, 0.7, 0.5, 1.0 };

  glPushMatrix();
  {
    glTranslatef(0.0, -depth, 0.0);

    glPushAttrib(GL_ALL_ATTRIB_BITS);
    {
      glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
      glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);

      glBegin(GL_QUADS);
      {
	glNormal3f(0.0, 1.0, 0.0);
	glVertex3f(0.0, 0.0, 0.0);
	glVertex3f(size_x, 0.0, 0.0);
	glVertex3f(size_x, 0.0, size_z);
	glVertex3f(0.0, 0.0, size_z);
      }
      glEnd();
    }
    glPopAttrib();
  }
  glPopMatrix();
}


/****  stone  ****/

void
draw_stone(float coord_x, float coord_z, float depth, int height, float width)
{
  /*  draws a graduated stone  */

  GLfloat mat_diffuse[] = { 0.4, 0.4, 0.4, 1.0 };
  GLfloat mat_ambient[] = { 0.0, 0.7, 0.5, 1.0 };

  float half_width;

  int i;


  half_width = width / 2.0 + 0.01;

  glPushMatrix();
  {
    glTranslatef(coord_x, 0.0, coord_z);

    /*  stone  */
    glPushMatrix();
    {
      glTranslatef(0.0, (-depth + height) / 2.0, 0.0);
      glScalef(1.0, (depth + height) / width, 1.0);

      glPushAttrib(GL_ALL_ATTRIB_BITS);
      {
	glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
	glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);

	glDisable(GL_CULL_FACE);

	glutSolidCube(width);
      }
      glPopAttrib();
    }
    glPopMatrix();

    /*  graduations  */
    glPushAttrib(GL_ALL_ATTRIB_BITS);
    {
      glDisable(GL_LIGHTING);
      glShadeModel(GL_FLAT);

      /*  zero  */
      glColor3f(0.6, 0.0, 0.0);
      draw_square_centered(half_width);

      glColor3f(0.4, 0.0, 0.0);
      /*  above  */
      glPushMatrix();
      {
	for (i = 1; i < height; i++)
	  {
	    glTranslatef(0.0, 1.0, 0.0);
	    draw_square_centered(half_width);
	  }
      }
      glPopMatrix();
      /*  below  */
      glPushMatrix();
      {
	for (i = -1; i > -depth; i--)
	  {
	    glTranslatef(0.0, -1.0, 0.0);
	    draw_square_centered(half_width);
	  }
      }
      glPopMatrix();
    }
    glPopAttrib();
  }
  glPopMatrix();
}


void
draw_square_centered(float half_width)
{
  glBegin(GL_LINE_LOOP);
  {
    glVertex3f(-half_width, 0.0, half_width);
    glVertex3f(half_width, 0.0, half_width);
    glVertex3f(half_width, 0.0, -half_width);
    glVertex3f(-half_width, 0.0, -half_width);
  }
  glEnd();
}


}  /*  namespace  */


/*  End draw.cpp  */
