/*
 * keystore.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <iostream>
#include <set>

#include <boost/filesystem/fstream.hpp>
#include <boost/filesystem/operations.hpp>

#include "keystore.h"


namespace apso {

/**
 * Constructor.
 */
Keystore::Keystore() {
	latest = 0;
	_encrypted = false;
	_already_read = false;
}

/**
 * Checks if the keystore is encrypted.
 */
bool
Keystore::encrypted() {
	return _encrypted;
}

/**
 * Checks if the keystore has already been read.
 */
bool
Keystore::already_read() {
	return _already_read;
}

/**
 * Checks if a key is present.
 *
 * @param id A string with the key ID.
 */
bool
Keystore::key_present(std::string id) {
	unsigned i;
	for (i=0; i<list.size(); i++)
	if (list[i]->get_id().compare(id) == 0)
		return true;
	return false;
}

/**
 * Sets the encrypted flag for this keystore.
 */
void
Keystore::set_encrypted(bool value) {
	_encrypted = value;
}

/**
 * Sets the current key.
 *
 * @param id A string with the key ID.
 */
void
Keystore::set_current(std::string id) {
        unsigned i;
        for (i=0; i<list.size(); i++)
                if (list[i]->get_id().compare(id) == 0) {
                        latest = i;
                        return;
                }
}

/**
 * Adds a key to the key list.
 *
 * @param The key to be added.
 */
void
Keystore::add(Key& k) {
	boost::shared_ptr<Key> key (new Key);
	*key = k;
	list.push_back(key);
}

/** 
 * Gets a key from the list.
 *
 * @param index The index of the key
 * @return The key wanted
 */
Key
Keystore::get(const long index) {
	if ( (index < 0) || ((unsigned)index >= list.size()) )
		throw std::runtime_error ("Key index out of range");
	return * list.at(index);
}

/** 
 * Gets the latest key being used.
 * 
 */
Key
Keystore::get() {
	if (list.empty()) {
		std::cout << "There were no keys in the keystore, so I added one\n";
		Key_ptr k (new Key());
		add(*k);
		latest = 0;
	}
		//throw std::runtime_error ("There are no keys in the keystore");
	return * list.at(latest);
}

/**
 *  Gets the key to a revision.
 */
Key
Keystore::get(const std::string& revision) {
	// FIXME: check first if the key is there!
	return * list.at(keymap[revision]);
}

/**
 * Gets the key list.
 *
 * FIXME: this method should not exist!
 */
std::vector<Key_ptr>
Keystore::get_key_list() {
	return list;
}


/**
 * Maps a revision onto a key, given the revision ID and the key index.
 *
 * @param revision A string with the revision ID
 * @param index The index of the key.
 */
void
Keystore::set(const std::string& revision, const long index) {
	keymap[revision] = index;
}

/**
 * Prints the keymap to an ostream.
 *
 * @param The ostream to which the map is printed.
 */
void
Keystore::showmap(std::ostream& out) {
	std::map<std::string,int>::iterator i;
	for (i=keymap.begin(); i != keymap.end(); i++)
		out << i->first << " --> " << i->second << "\n";
}

/**
 * Maps a revision onto a key, given the revision ID and the key ID.
 *
 * @param revision A string with the revision ID
 * @param id The key ID.
 */
void
Keystore::set (const std::string& revision, const std::string id) {
	unsigned i;
	for (i=0; i<list.size(); i++)
		if (list[i]->get_id().compare(id) == 0) {
			keymap[revision] = i;
			return;
		}
}

/**
 * Maps a revision ID onto the latest key.
 *
 * @param revision A string with the revision ID
 */
void
Keystore::set (const std::string& revision) {
	keymap[revision] = latest;
}


/** 
 * Gets the size of the list.
 *
 * @return The size of the key list.
 */
long
Keystore::get_size() const {
	return list.size();
}

/**
 * Creates new key and append it to the list.
 *
 * The new key will be initialized with random bits, and
 * its ID will be empty.
 */
void
Keystore::newkey() {
	boost::shared_ptr<Key> k (new Key);
	list.push_back(k);
}

/**
 * Reads a keystore from a directory.
 *
 * Will read the whole key mapping and the keys from a directory.
 * The file "keymap" in that directory contains the keymap, while the
 * other files are keys.
 *
 * The filenames should be the key IDs.
 *
 * @param in The path of a DIRECTORY where the keys and key mapping are.
 */
void
Keystore::read(const Path& dir) {
	std::cout << "[ Keystore read start ]\n";
	// FIXME: sanity checks could be better
	if (boost::filesystem::exists(dir)) {
		if (!boost::filesystem::is_directory(dir))
			throw std::runtime_error (dir.string()+" is not a directory");
	} else
		throw std::runtime_error (dir.string()+" does not exist");

	Path keyspath (dir);
	Path keymappath (dir);
	keyspath   /= "keys";
	keymappath = keymappath.branch_path();
	keymappath = keymappath.branch_path();
	keymappath /= "keymap";
	if ( (!boost::filesystem::exists(keyspath)) ||
	     (!boost::filesystem::exists(keymappath)))
		throw std::runtime_error (keyspath.string() + " AND " +
					  keymappath.string() + " must exist");
	
	
	std::string id;
	std::string rev;
	std::string data;
	Key_ptr new_k;
	std::set<std::string> ids;
	unsigned size;
	boost::filesystem::directory_iterator end_itr;
	std::cout << "Before reading revisions, " << get_size() << " keys\n";

        for ( boost::filesystem::directory_iterator itr( keymappath ); itr != end_itr; ++itr ) {
                if ( boost::filesystem::is_directory(*itr) )
			continue;
		rev = itr->leaf();
		// Read one more revision from the mapping:
		boost::filesystem::ifstream in_stream (*itr);
		in_stream >> id;
		// Update the mapping:
		set(rev, id);
		in_stream.close();
	}
	std::cout << "After reading revisions, " << get_size() << " keys\n";
	for ( boost::filesystem::directory_iterator itr( keyspath ); itr != end_itr; ++itr ) {
		if ( boost::filesystem::is_directory(*itr) )
			continue;
		if (itr->string().find("_key") == itr->string().size())
			continue;
		id = itr->leaf();
		if (ids.count(id) == 0 &&
		    !key_present(id)) {
			// Open the file which has the same name as the key ID, and
                        // read the content into the key:
                        Path key_path (keyspath);
                        key_path /= id;
                        boost::filesystem::ifstream key_in (key_path);
                        key_in >> data;
                        size = data.size();
                        key_in.close();
                        ids.insert(id);
                        // Create new key object:
                        new_k = Key_ptr(new Key);
                        bdata b (data);
                        new_k->set_value(b);
                        new_k->set_id(id);
                        // Add it:
                        add(*new_k);
			std::cout << "ADDING KEY " << new_k->get_id() << "\n";
		}
	}
        Path path_latest (dir);
        path_latest /= "current_key";
        boost::filesystem::ifstream file_latest (path_latest);
        file_latest >> id;
        file_latest.close();
	set_current(id);

	_already_read = true;
	std::cout << "[ Keystore read end ]\n";
}




/**
 * Saves a keymap in a directory.
 *
 * Will save the whole key mapping into a directory.
 *
 * The direcotry given is the WC directory.
 *
 * @param in The path of a DIRECTORY where the key mapping will be saved.
 */
void
Keystore::store_keymap(const Path& dir) {
	std::cout << "[ Keystore store_keymap start ]\n";
	std::map<std::string,int>::const_iterator i;
	
	// Keymap file:
	Path out (dir);

	if (boost::filesystem::exists(dir)) {
		if (!boost::filesystem::is_directory(dir))
			throw std::runtime_error ("Keystore path should be a directory, and this isn't one");
	} else {
		std::cout << "Keystore not there, creating it. " << dir.string();
		boost::filesystem::create_directory(dir);
	}

	Path p (dir);
	p = p.branch_path();
	p = p.branch_path();
	p /= "keymap";
	boost::filesystem::create_directory (p);

	std::set<std::string> ids;
	for (i=keymap.begin(); i!=keymap.end(); i++) {
		// Write it to the keymap:
		Path id_path (dir);
		id_path /= "keymap";
		id_path /= i->first;
		boost::filesystem::ofstream out_stream (id_path);
		Key_ptr k = list[i->second];
		out_stream << k->get_id();
		out_stream << "\n";
		out_stream.close();
	}
	Path path_latest (dir);
	path_latest /= "current_key";
	std::cout << "Saving latest (" << get().get_id() << ") key to " << path_latest.string() << "\n";
	boost::filesystem::ofstream file_latest (path_latest);
	file_latest << get().get_id();
	file_latest.close();
	std::cout << "[ Keystore store_keymap end ]\n";
}

/**
 * Saves a user's keys in a directory.
 *
 * Will save the the keys into a directory.
 *
 * The direcotry given is the user's directory.
 *
 * users/johndoe/...
 *
 * This method accepts "users/johndoe" as the parameter.
 *
 * The filenames of the key files will be the key IDs.
 *
 * @param in The path of a DIRECTORY where the keys will be saved.
 */
void
Keystore::store_keys(const Path& dir) {
        std::cout << "[ Keystore store start ]\n";
        std::vector<Key_ptr>::iterator k;

        Path out (dir);

        if (boost::filesystem::exists(dir)) {
                if (!boost::filesystem::is_directory(dir))
                        throw std::runtime_error ("Keys path should be a directory, and this isn't one");
        } else {
                std::cout << "Keys not there, creating it. " << dir.string();
                boost::filesystem::create_directory(dir);
        }

        Path p (dir);
        p /= "keys";
        boost::filesystem::create_directory (p);

        std::set<std::string> ids;
	std::cout << "\nOK, I HAVE " << get_size() << "keys!\n";
        for (k=list.begin(); k!=list.end(); k++) {
                // If we didn't save this key yet...
		std::cout << "Checking key " << (*k)->get_id() << "\n";
                if (ids.count((*k)->get_id()) == 0) {
                        // Then save it on a file that has the key ID as name:
                        Path key_path (dir);
                        key_path /= "keys";
                        key_path /= (*k)->get_id();
			(*k)->save(key_path);
                }
        }
        std::cout << "[ Keystore store_keys end ]\n";
}


/**
 * Saves the keystore and keymapping.
 *
 * Receives a user's directory.
 *
 * @param dir A path to the user's directory.
 */
void
Keystore::store(const Path& dir) {
        store_keys(dir);
        Path keymap_dir (dir);
        // Go up two levels:
        keymap_dir = keymap_dir.branch_path();
        keymap_dir = keymap_dir.branch_path();
        store_keymap(keymap_dir);
}

}
