/*
 * staticset.h
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef DIRECTORY_H
#define DIRECTORY_H

#include <map>
#include <list>
#include <string>
#include <exception>
#include <stdexcept>

#include <boost/shared_ptr.hpp>

namespace aleph0 {

/**
 * Manages a catalog of class instances.
 *
 * Make you class inherit from this one, and then you will be able
 * to store named instances of your class:
 *
 * class MyClass : public Directory<MyClass> { ... };
 *
 * boost::shared_ptr<MyClass> ptr (new MyClass);
 * MyClass::instance_register("instance#1", ptr);
 * ...
 * boost::shared_ptr<MyClass> ptr2;
 * ptr2 = MyClass::instance("instance#1");  // this is the exact instance you registered!
 *
 * If you have one class MyClass and several subclasses of it, then
 * make MyClass inherit from this, and you can register any of the subclasses
 * of MyClass:
 *
 * class MyClass : public Directory<MyClass> { ... };
 * class MyClassA : public MyClass  { ... };
 * class MyClassB : public MyClassA { ... };
 *
 * boost::shared_ptr<MyClass> ptr (new MyClassA);
 * boost::shared_ptr<MyClass> ptr (new MyClassB);
 * MyClass::instance_register("instance#1", ptrA);
 * MyClass::instance_register("instance#2", ptrB);
 */
template <typename T>
class Directory {
private:
	static std::map< std::string,boost::shared_ptr<T> > instance_list;
	static std::string instance_default;
public:
	static boost::shared_ptr<T> instance() {
		return instance (instance_default);
	}

	static boost::shared_ptr<T> instance(const std::string& name) {
		if (instance_list.find(name) == instance_list.end()) {
			// name not registered, so we register ir and create
			// a new instance of the class T:
			boost::shared_ptr<T> ptr (new T);
			instance_register(name, ptr);
		}
		return instance_list[name];
	}

	static void instance_register(const std::string& name, boost::shared_ptr<T> ptr) {
		instance_list[name] = ptr;
	}

	static void set_default(const std::string& name) {
		if (instance_list.find(name) == instance_list.end()) {
			// name not registered, so we register ir and create
			// a new instance of the class T:
			boost::shared_ptr<T> ptr (new T);
			instance_register(name, ptr);
		}
		instance_default = name;
	}

};

template <typename T> std::map< std::string,boost::shared_ptr<T> > Directory<T>::instance_list;
template <typename T> std::string Directory<T>::instance_default;

}

#endif

