/*
 * options.cpp
 *
 * Copyright (C) 2006 Jernimo Pellegrini
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *   The Free Software Foundation, Inc.,
 *   51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "options.h"


/**
 * Gets the id of the user running Apso.
 *
 * @return A string with the user ID.
 */
std::string
apso::Options::get_user_id() {
	return user_id;
}

/**
 * Gets the public key of the user running Apso.
 *
 * @return The path to the public key.
 */
Path
apso::Options::get_user_pubk() {
	return user_pubk;
}

/**
 * Sets the id of the user running Apso.
 */
void
apso::Options::set_user_id(const std::string& id) {
	user_id = id;
}

/**
 * Sets the public key of the user running Apso.
 *
 * @param keypath A string with the path to the key.
 */
void
apso::Options::set_user_pubk(const std::string& keypath) {
	Path p (keypath);
	user_pubk = p;
}


/**
 * Includes the version control system to be used.
 *
 * @vc a string with the name of the VC system
 */
void
apso::Options::set_vc_system(std::string vc) {
	vc_system = vc;
}

/**
 * Includes the version control system private key.
 *
 * Version control systems will ask the user for a private key in order to sign
 * any changes. Apso needs a key to commit to the public database.
 *
 * @param vc a string with the string as it would be passed to the VC system on the command line
 */
void
apso::Options::set_vc_prk (const std::string& k) {
	vc_priv_key = k;
}

/**
 * Includes Apso's private key.
 *
 * @param k the path to an Apso private key.
 */
void
apso::Options::set_apso_prk (const Path& k) {
        apso_priv_key = k;
}


void
apso::Options::set_branch(std::string br) {
	branch = br;
}


/**
 * Sets the path to the public repository.
 *
 * @param pr a string with the path
 */
void
apso::Options::set_pubrep(std::string pr) {
	Path rep (pr);
	pubrep = rep;
}

/**
 * Sets the path to the private repository.
 *
 * @param pr a string with the path
 */
void
apso::Options::set_privrep(std::string pr) {
	Path rep (pr);
	privrep = rep;
}

/**
 * Sets the path for a scratch tree that Apso will use.
 *
 * @param sc a string with the path
 */
void
apso::Options::set_scratch(std::string sc) {
	Path rep (sc);
	scratch = rep;
}

/**
 * Gets the name of the version control system configured.
 *
 * @return A string with the name of the VCS
 */
std::string
apso::Options::get_vc_system() {
	return vc_system;
}

/**
 * Returns the path the the private key.
 *
 * @return A path to the private key.
 */
Path
apso::Options::get_apso_prk() {
	return apso_priv_key;
}

/**
 * Returns the private key to be used with the VCS.
 *
 * This is different from the private key used with Apso!
 *
 * @return a string with the private key.
 */
std::string
apso::Options::get_vc_prk() {
        return vc_priv_key;
}


/**
 *
 */
std::string
apso::Options::get_branch() {
	return branch;
}


/**
 * Returns the path to the public repository.
 *
 * @return The path to the public repository.
 */
Path
apso::Options::get_pubrep() {
	return pubrep;
}

/**
 * Returns the path to the private repository.
 *
 * @return The path to the private repository.
 */
Path
apso::Options::get_privrep() {
	return privrep;
}

/**
 * Returns the path to scratch directory.
 *
 * @return The Path to the scratch directory.
 */
Path
apso::Options::get_scratch() {
	return scratch;
}

