//
//    cctools --- tools for the C++ programmer
//    Copyright (C) 2000 Jan Pfeifer 
//
//    This library is free software; you can redistribute it and/or
//    modify it under the terms of the GNU Lesser General Public
//    License as published by the Free Software Foundation; either
//    version 2 of the License, or (at your option) any later version.
//
//    This library is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//    Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public
//    License along with this library; if not, write to the Free Software
//    Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//

#include "cctools.h"

using namespace std;

namespace cctools {

// cctools.hh
//============================================================

//------------------------------------------ Execute shell command
//  like system from stdlib, but tests for exec errors and
//   throws exceptions
int system(const std::string& cmd) throw (std::runtime_error)
{
    int ret = std::system(cmd.c_str());
    if (ret == 127) 
        throw std::runtime_error ("cctools::system(): couldn't call shell");
    if (ret == -1) 
        throw std::runtime_error
            ("cctools::system(): error executing");
    return ret;
}


/**
 * Run command and return its output
 * 
 * @param cmd command to run
 * @returns string evaluated
 **/
int system( const std::string &cmd, 
		const std::string &input, 
		std::string &output,
		std::string &error,
		unsigned timeout ) throw ( std::bad_alloc, std::runtime_error, std::runtime_error )
{
	int ret;
	
	// get pipes opened
	int pipe_input[2], pipe_output[2], pipe_error[2];
	ret = pipe( pipe_input );
	if ( ret != 0 )
		throw std::runtime_error("cctools::system(cmd,in,out,err): couldn't pipe");
	ret = pipe( pipe_output );
	if ( ret != 0 )
		throw std::runtime_error("cctools::system(cmd,in,out,err): couldn't pipe");
	ret = pipe( pipe_error );
	if ( ret != 0 )
		throw std::runtime_error("cctools::system(cmd,in,out,err): couldn't pipe");
	
	// fork to run code
	pid_t child = fork();
	if ( child == -1 )
		throw std::runtime_error("cctools::system(cmd,in,out,err): couldn't fork");
	if ( child == 0 ) {
		// Child process: dup pipes to the standard fds and run command
		close(0); close( pipe_input[1] );
		ret = dup2( pipe_input[0], 0 );
		if ( ret == - 1 ) exit(126);		
		close(1); close( pipe_output[0] );
		ret = dup2( pipe_output[1], 1 );
		if ( ret == - 1 ) exit(126);
		close(2); close( pipe_error[0] );
		ret = dup2( pipe_error[1], 2 );
		if ( ret == - 1 ) exit(126);
		
		// Run code through shell
		execl( "/bin/sh", "/bin/sh", "-c", cmd.c_str(), (const char *)0 );
		exit(125); // should never get here
	}

	// close the other ends of the pipes
	close( pipe_input[0] );
	close( pipe_output[1] );
	close( pipe_error[1] );

	// create poll structures
	struct pollfd polls[] = {
		{ pipe_input[1], POLLOUT, 0 },
		{ pipe_output[0], POLLIN, 0 },
		{ pipe_error[0], POLLIN, 0 }
	};
	
  	// handle IO
	int status;
	pid_t wait_res = 0;
  	try {
	  	size_t in_left = input.length();
	  	const char *in = 0;
	  	if ( in_left == 0 ) {
			polls[0].events = 0;
			close( polls[0].fd );
			polls[0].fd = -1;
	  	} else
	  	   	in = input.c_str();
	  	output = string();
	  	error = string();
	  	ssize_t count;
	  	char buffer[ BUFFER_SIZE + 1 ];
	  	
		time_t inicio = time(0);
	  	while ( (timeout == 0) or (time(0) < inicio+timeout) ) {
			// poll
			int poll_timeout = (timeout == 0 ? -1 : (int)( inicio+timeout-time(0) )*1000 );
	  		ret = poll( polls, sizeof(polls)/sizeof(struct pollfd), poll_timeout );
	  		if ( ret == -1 )
	  			throw std::runtime_error ("cctools::system(cmd,in,out,err,time): couldn't poll");
			
			// check if child exited	  		
			wait_res = waitpid( child, &status, WNOHANG );
					
			//cerr << "polling ..." << polls[0].revents << ", " << polls[1].revents << ", " << polls[2].revents
			//	<< " - " << polls[0].events << ", " << polls[1].events << ", " << polls[2].events << endl;
	  		
	  		// test for input buffer
	  		if ( polls[0].revents & POLLOUT ) {
	 			count = write( polls[0].fd, in, in_left );
	 			if ( count == -1 && errno != EINTR )
	 				throw std::runtime_error("cctools::system(cmd,in,out,err,time): couldn't write to pipe");
				if ( count != -1 ) {
	 				in_left -= count;
	 				if ( in_left == 0 ) {
	 					polls[0].events = 0;
	 					close( polls[0].fd );
	 					polls[0].fd = -1;
	 				}
				}
	  		}
	  		
	  		// test output buffers
	  		if ( polls[1].revents & POLLIN ) {
	 			count = read( polls[1].fd, buffer, BUFFER_SIZE );
	 			if ( count == -1 && errno != EINTR )
	 				throw std::runtime_error("cctools::system(cmd,in,out,err): couldn't read from child output pipe");
				if ( count > 0 ) {
					buffer[ count ] = (char) 0;
					// cerr << "  reading " << count << " bytes: " << buffer;
					output.append( buffer );
				} else if ( count == 0 ) {
					polls[1].events = 0;
					close( polls[1].fd );
	 				polls[1].fd = -1;
				}
	  		}	

	  		// test error buffers
	  		if ( polls[2].revents & POLLIN ) {
	 			count = read( polls[2].fd, buffer, BUFFER_SIZE );
	 			if ( count == -1 && errno != EINTR )
	 				throw std::runtime_error("cctools::system(cmd,in,out,err): couldn't read from child error pipe");
				if ( count > 0 ) {
					buffer[ count ] = (char) 0;
					// cerr << "  reading " << count << " bytes: " << buffer;
					error.append( buffer );
				} else if ( count == 0 ) {
					polls[2].events = 0;
					close( polls[2].fd );
	 				polls[2].fd = -1;
				}
	  		}	
	  		
	  		// if child has finished, this is the end
	  		if ( wait_res == child ) break;
			
	  		// test output/error pipe for end of output:
	  		if ( ( polls[1].revents & POLLERR ) ||
	  			( polls[1].revents & POLLHUP ) ||
	  			( polls[1].revents & POLLNVAL ) ) {
				close( polls[1].fd );
				polls[1].events = 0;
 				polls[1].fd = -1;
				
			}
	  		if ( ( polls[2].revents & POLLERR ) ||
	  			( polls[2].revents & POLLHUP ) ||
	  			( polls[2].revents & POLLNVAL ) ) {
				close( polls[2].fd );
				polls[2].events = 0;
 				polls[2].fd = -1;
			}
			
			// if output/error are finished quit polling
			if ( polls[1].events == 0 && polls[2].events == 0 )
				break;
	  	}
		
		// if timeouted, kill child
  		if ( wait_res != child and timeout != 0 and time(0) >= inicio+timeout ) 
		{
			kill( child, 9 );
			while ( true ) {
				wait_res = waitpid( child, &status, 0 );
				if ( wait_res == -1 && errno == EINTR ) continue;
				if ( wait_res != child ) continue;
				break;
			}
			if ( wait_res == -1 )
				throw std::runtime_error("cctools::system(cmd,in,out,err,timeout): couldn't waitpid, after timeout");
			return -1;			
		}
		
  	} catch (...) {
  		if ( wait_res == child ) throw;
  		
  		// in case of problems, wait for child to finish first
		while ( true ) {
			wait_res = waitpid( child, &status, 0 );
			if ( wait_res == -1 && errno == EINTR ) continue;
			if ( wait_res != child ) continue;
			break;
		}	
		throw;
  	}
  	
	// wait for child
	while ( wait_res != child ) {
		wait_res = waitpid( child, &status, 0 );
		if ( wait_res == -1 && errno != EINTR ) break;
	}
	if ( wait_res == -1 )
		throw std::runtime_error("cctools::system(cmd,in,out,err,timeout): couldn't waitpid");
	
	return WEXITSTATUS( status );
}

}
//============================================================
// cctools.hh
