/*
  AntiRight
  (c) 2002-2007 Jeffrey Bedard
  antiright@gmail.com

  This file is part of AntiRight.

  AntiRight is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  AntiRight is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with AntiRight; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "gtkshell.h"

#ifdef HAVE_VTE
#define VSET(element, widget, ...)\
	vte_terminal_set_##element(VTE_TERMINAL(widget), __VA_ARGS__)
#endif /* HAVE_VTE */

#define GT term

#ifdef HAVE_VTE
#define VSETW(element, ...)\
	VSET(element, term->widget, __VA_ARGS__)
#endif /* HAVE_VTE */

#ifdef HAVE_VTE
static void
set_vte_specific_terminal_options(struct GSH_Terminal * term)
{
	VSETW(size, term->dimensions.width, term->dimensions.height);
	VSETW(allow_bold, TRUE);
	VSETW(backspace_binding, VTE_ERASE_ASCII_BACKSPACE);
	if(term->transparent)
	{
		/* Only the first terminal should be transparent,
		   to save memory.  */
		VSETW(background_transparent, TRUE);
		term->transparent=FALSE;
	}
}
#endif /* HAVE_VTE */

static void
set_terminal_options(struct GSH_Terminal * term)
{
#ifdef HAVE_VTE
	set_vte_specific_terminal_options(term);
#endif /* HAVE_VTE */
	if(term->widget)
		gsh_widget_set_font(term->widget, term->font);
}

static const gchar *
get_shell()
{	
	/* This fixes the failure of environment inheritance
	   under cygwin.  */
	const gchar * shell = getenv("SHELL");
	if(shell)
		return shell;
	else
		return (const gchar *)"/bin/sh";
}

static void
gsh_terminal_run(struct GSH_Terminal * term, const gchar * command)
{
	/* Set up line and column counts so that VI has correct display
	   in IDE terminal.  */
	gchar * height;
	gchar * width;
	/* Use format of a shell export statement.  */
	ar_asprintf(&height, "LINES=%d", term->dimensions.height);
	ar_asprintf(&width, "COLUMNS=%d", term->dimensions.width);
	{
		/* Null terminated list of environment elements.  */
		const gchar *envv[] = {height, width, NULL};
		/* Determine which shell to use.  */
		const gchar * shell = get_shell();
		/* Null terminated list of program arguments.  Use
		   the shell as a command processor, such to avoid
		   handling argument separation here.  */
		const gchar *argv[] = { shell, "-c", command, NULL };
		/* Start the command in the terminal.  */
#ifdef HAVE_VTE
		vte_terminal_fork_command(VTE_TERMINAL(term->widget), shell, 
			(char **)argv, (char **)envv, NULL, TRUE, TRUE, TRUE);
#endif /* HAVE_VTE */
	}
	/* Free after use.  */
	g_free(height);
	g_free(width);
}

static void
gsh_delete_GSH_Terminal(struct GSH_Terminal * term)
{
	g_free(term);
	/* Freeing the terminal font causes a segmentation fault under
	 * linux, so likely a double free.  */
	/* The terminal widget itself is freed by gobject system
	 * reference counting.  */
}

static void
title_change_cb(GtkWidget * widget, gpointer data)
{
	struct GSHCBData * cb;
	GSH * gsh;
	GtkWindow * window;

	cb=(struct GSHCBData *)data;
	gsh=cb->gsh;
	window=GTK_WINDOW(gsh->widgets.window);
	gtk_window_set_title(window, 
#ifdef HAVE_VTE
			vte_terminal_get_window_title(VTE_TERMINAL(widget))
#else /* not HAVE_VTE */
			"Terminal"
#endif /* HAVE_VTE */
			);
}

static void
status_changed_cb(GtkWidget * widget, gpointer data)
{
	struct GSHCBData * cb;
	GSH * gsh;
	GtkWidget * status;

	cb=(struct GSHCBData *)data;
	gsh=cb->gsh;
	if((status=gsh->widgets.app.status))
		gtk_statusbar_push(GTK_STATUSBAR(status),
			gtk_statusbar_get_context_id(GTK_STATUSBAR(status), 
			"terminal"), 
#ifdef HAVE_VTE
			vte_terminal_get_status_line(VTE_TERMINAL(widget))
#else /* not HAVE_VTE */
			"terminal"
#endif /* HAVE_VTE */
			);
}

static void
setup_terminal_signals(struct GSH_Terminal * term)
{
	GSH * gsh;
	GtkWidget * widget;

	gsh=(GSH *)term->gsh;
	widget=term->widget;
	/* Return if no widget was created.  */
	if(!widget)
		return;
	/* Exit from the application if the terminal process dies.  */
	GSHCONNECT(widget, "child-exited", gtk_main_quit, NULL);
	{
		struct GSHCBData * cb;

		cb=ARNEW(gsh, GSHCBData, gsh, NULL);
		GSHCONNECT(widget, "status-line-changed",
			status_changed_cb, cb);
		GSHCONNECT(widget, "window-title-changed", 
			title_change_cb, cb);
	}
}

static GtkWidget *
gsh_GSH_Terminal_add(struct GSH_Terminal * term, const gchar * command)
{
	/* Create and set properties of the VTE-based terminal.  */
#ifdef HAVE_VTE
	term->widget = vte_terminal_new();
#else /* not HAVE_VTE */
	term->widget = NULL;
#endif /* HAVE_VTE */
	setup_terminal_signals(term);
	set_terminal_options(term);
	gsh_terminal_run(term, command);

	return term->widget;
}

static void
assign_methods(struct GSH_Terminal * term)
{
	term->delete=&gsh_delete_GSH_Terminal;
	term->add=&gsh_GSH_Terminal_add;
}

static void
initialize_fields(struct GSH_Terminal * term)
{
	term->widget=NULL;
	term->font=NULL;
	term->dimensions.width=80;
	term->dimensions.height=24;
	term->transparent=TRUE;
}

static void
setup_GSH_Terminal(struct GSH_Terminal * term)
{
	initialize_fields(term);
	assign_methods(term);
}

struct GSH_Terminal *
gsh_new_GSH_Terminal(GSH * gsh)
{
	struct GSH_Terminal * term;

	term=xmalloc(sizeof(struct GSH_Terminal));
	setup_GSH_Terminal(term);
	term->gsh = (gpointer)gsh;

	return term;
}

