/*
  AntiRight
  (c) 2002-2007 Jeffrey Bedard
  antiright@gmail.com

  This file is part of AntiRight.

  AntiRight is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  AntiRight is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with AntiRight; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include "gtkshell.h"

static void
create_label_in_menu_item(GtkWidget * widget, const gchar * formatted)
{
	GtkWidget * label;

	gtk_container_add(GTK_CONTAINER(widget), 
		label = gtk_label_new(formatted));
	gtk_widget_show(label);
}

static void
setup_label(GtkWidget *widget, const gchar *command)
{
	gchar *formatted;

	formatted = antiright_beautified_label(command);
	GTK_IS_MENU_ITEM(widget) ? create_label_in_menu_item(widget, formatted)
		: gtk_button_set_label(GTK_BUTTON(widget), formatted);
	g_free(formatted);
}

static void
setup_callback(struct GSH * gsh, GtkWidget * button, const gchar * command)
{
	if(!GSH_FLAG(GSH_MENU_BUTTON))
	{
		struct GSHCBData * cb;

		cb=ARNEW(gsh, GSHCBData, gsh, g_strdup(command));
		(void)g_signal_connect(G_OBJECT(button), 
			GTK_IS_MENU_ITEM(button) 
			? "activate" : "clicked",
			G_CALLBACK(cb->system), cb);
	}
	else
		GSH_UNSET(GSH_MENU_BUTTON);
}

static void
create_button(struct GSH * gsh)
{
	if(!GTK_IS_MENU(gsh->rows->v) && !GTK_IS_MENU_BAR(gsh->rows->v))
		gsh->widgets.button=gtk_button_new();
	else
		gtk_widget_show(gsh->widgets.button=gtk_menu_item_new());
}

static void
common_button_setup(GSH * gsh, GtkWidget * button, const gchar * command)
{
	gsh_setup_drag_drop(gsh, button);
	setup_callback(gsh, button, command);
}

static GtkWidget *
label_to_icon_translation(const gchar * label)
{
	GtkWidget * image;
	gchar * cut_label;

	cut_label=(gchar *)label;
	g_strstrip(cut_label);
#define MATCHES(text) !strcmp(text, cut_label)
#define ICON(id)\
	gtk_widget_show(image=gtk_image_new_from_stock(GTK_STOCK_##id, \
		GTK_ICON_SIZE_SMALL_TOOLBAR))

	if(MATCHES("Exit"))
		ICON(QUIT);
	else if(MATCHES("About"))
		ICON(ABOUT);
	else if(MATCHES("Open..."))
		ICON(OPEN);
	else if(MATCHES("Save"))
		ICON(SAVE);
	else if(MATCHES("Save As..."))
		ICON(SAVE_AS);
	else if(MATCHES("Cut"))
		ICON(CUT);
	else if(MATCHES("Copy"))
		ICON(COPY);
	else if(MATCHES("Paste"))
		ICON(PASTE);
	else if(MATCHES("Close"))
		ICON(CLOSE);
	else
		ICON(EXECUTE);

	return image;
}

static void
setup_icon_for_toolbutton(GtkWidget * toolbutton, gchar * label_text)
{
	GtkWidget * icon;

	icon = label_to_icon_translation(label_text);
	if(icon)
		gtk_button_set_image(GTK_BUTTON(toolbutton), icon);
}

static void
setup_label_for_toolbutton(GtkWidget * toolbutton, const gchar * command)
{
	gchar * label_text;

	label_text=antiright_beautified_label(command);
	gtk_button_set_label(GTK_BUTTON(toolbutton), label_text);
	setup_icon_for_toolbutton(toolbutton, label_text);
	g_free(label_text);
}

static void
setup_toolbutton_from_command_button(GSH * gsh, const gchar * command)
{
	GtkWidget * toolbutton;

	/* FIXME:  change to use the gtk tool_button widget.  */
	toolbutton=gtk_button_new_with_label(command);
	common_button_setup(gsh, toolbutton, command);
	GSHCA(gsh->widgets.app.toolbar, toolbutton);
	setup_label_for_toolbutton(toolbutton, command);
	gtk_button_set_relief(GTK_BUTTON(toolbutton), GTK_RELIEF_NONE);
	gtk_widget_show(toolbutton);
}

static void
test_create_toolbutton(GSH * gsh, const gchar * command)
{
	if(	GSH_FLAG(GSH_APP_MODE) 
		&& gsh->widgets.app.toolbar
		&& GTK_IS_MENU_ITEM(gsh->widgets.button) 
		&& strcmp(command, "File")
		&& strcmp(command, "Edit") 
		&& strcmp(command, "View")
		&& strcmp(command, "Help") 
		&& strcmp(command, "Tools"))
	{
		setup_toolbutton_from_command_button(gsh, command);
	}
}

static void
setup_command_button(struct GSH * gsh, const gchar * command)
{
	GtkWidget * button;

	setup_label(button=gsh->widgets.button, command);
	common_button_setup(gsh, button, command);
	test_create_toolbutton(gsh, command);
	if(GTK_IS_BUTTON(button))
		gtk_button_set_alignment(GTK_BUTTON(button), 0.0, 0.5);
}

static gboolean
test_image_button(struct GSH * gsh, const gchar * string)
{
	gchar **pair;
	gboolean return_value;

	pair=g_strsplit(string, ",", 2);
#ifdef DEBUG
	ARPASSERT(pair);
	ARPASSERT(pair[0]);
#endif /* DEBUG */
	if((return_value=(pair[1] != NULL)))
		gsh_image_button(gsh, pair[0], pair[1]);
	g_strfreev(pair);

	return return_value;
}

static void
gsh_command_button(GSH * gsh, const gchar * command)
{
	create_button(gsh);
	setup_command_button(gsh, command);
	gsh_manage(gsh, gsh->widgets.button);
}

/* This is just a wrapper around gsh_new_button(), to
   follow the convention of returning a created widget.  */
GtkWidget *
gsh_add_button(struct GSH * gsh, const gchar * string)
{
	if(!test_image_button(gsh, string))
		gsh_command_button(gsh, string);

	return gsh->widgets.button;
}

