/*
 * gtk GUI functions
 *
 * This file is part of ANT (Ant is Not a Telephone)
 *
 * Copyright 2002 Roland Stigge
 *
 */

/* regular GNU system includes */
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>

/* GTK */
#include <gtk/gtk.h>

/* own header files */
#include "gtk.h"
#include "runtime.h"
#include "isdn.h"
#include "util.h"
#include "sound.h"

/* call timeout_callback each <this number> of milliseconds */
#define TIMER_DELAY 300

volatile int interrupted = 0; /* the caught signal will be stored here */

/* the quit (e.g. menu entry) callback
 * prototype is this way to standardize to GtkItemFactoryCallback2 */
static void quit(GtkWidget *widget, gpointer data, guint action) {
  gtk_main_quit();
}

/* handler for SIGTERM and SIGINT */
void terminate_signal_callback(int sig) {
  interrupted = sig;
}

/*
 * periodically (e.g. each 300 milliseconds) from gtk main loop called function
 */
gint timeout_callback(gpointer data) {
  if (interrupted) {
    switch(interrupted) {
    case SIGINT:
      if (debug)
	fprintf(stderr, "Ctrl-C caught.\n");
      break;
    case SIGTERM:
      if (debug)
	fprintf(stderr, "SIGTERM caught.\n");
      break;
    default:
      fprintf(stderr, "Warning: Unknown signal caught.\n");
    }
    quit(NULL, NULL, 0);
  }
  return TRUE; /* call it again */
}


/*
 * some GUI tools
 */

/*
 * return a dialog window with a (big) label and an ok button to close
 * (good for displaying a note)
 *
 * justification: justification of label (e.g. GTK_JUSTIFY_LEFT)
 *
 * NOTE: caller has to show the window itself with gtk_widget_show()
 *       and maybe want to make it modal with
 *         gtk_window_set_modal(GTK_WINDOW(window), TRUE);
 */
GtkWidget *get_ok_dialog(char *title, char *contents,
			 GtkJustification justification) {
  GtkWidget *window;
  GtkWidget *button;
  GtkWidget *label;

  window = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(window), title);

  /* vbox area */
  gtk_container_set_border_width(GTK_CONTAINER(GTK_DIALOG(window)->vbox), 0);
  label = gtk_label_new(contents);
  gtk_label_set_justify(GTK_LABEL(label), justification);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), label,
		     TRUE, FALSE, 0);
  gtk_misc_set_padding(GTK_MISC(label), 10, 10);
  gtk_widget_show(label);

  /* action area */
  button = gtk_button_new_with_label("OK");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), button,
		     TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(window));
  gtk_widget_show(button);

  /* caller has to show the window itself */

  return window;
}

/*
 * some GUI callbacks
 */

/* clicked "ok" at cb_settings */
static void cb_settings_ok(GtkWidget *widget, gpointer data) {
  struct session_t *session = (struct session_t *) data;
  GtkWidget *window;
  GtkWidget *entry;

  /* initialization is just for correctness to make safe state */
  char *msn = DEFAULT_MSN;
  char *msns = DEFAULT_MSNS;
  char *audio_device_name_in = DEFAULT_AUDIO_DEVICE_NAME_IN;
  char *audio_device_name_out = DEFAULT_AUDIO_DEVICE_NAME_OUT;

  /* save current settings */
  char *old_msn = strdup(session->msn);
  char *old_msns = strdup(session->msns);
  char *old_audio_device_name_in = strdup(session->audio_device_name_in);
  char *old_audio_device_name_out = strdup(session->audio_device_name_out);

  int successful = 1;
  GtkWidget *ok_window;

  /* msn */
  entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(widget), "msn_entry");
  if (entry)
    msn = gtk_entry_get_text(GTK_ENTRY(entry));
  else
    fprintf(stderr, "cb_settings_ok: Error getting msn.\n");

  /* msns */
  entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(widget), "msns_entry");
  if (entry)
    msns = gtk_entry_get_text(GTK_ENTRY(entry));
  else
    fprintf(stderr, "cb_settings_ok: Error getting msns.\n");
       

  /* audio_device_name_in */
  entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(widget),
					    "audio_device_name_in_entry");
  if (entry)
    audio_device_name_in = gtk_entry_get_text(GTK_ENTRY(entry));
  else
    fprintf(stderr, "cb_settings_ok: Error getting audio_device_name_in.\n");

  /* audio_device_name_out */
  entry = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(widget),
					    "audio_device_name_out_entry");
  if (entry)
    audio_device_name_out = gtk_entry_get_text(GTK_ENTRY(entry));
  else
    fprintf(stderr, "cb_settings_ok: Error getting audio_device_name_out.\n");
				   
  /* reset all this by closing and reopening isdn and sound devices */
  session_io_handlers_stop(session);
  deinit_session(session);
  if (init_session(session, audio_device_name_in, audio_device_name_out,
		   msn, msns)) { /* new settings are bad */
    /* restore and hope for the best */
    //deinit_session(session);
    init_session(session, old_audio_device_name_in, old_audio_device_name_out,
		 old_msn, old_msns);
    successful = 0;
  }
  session_io_handlers_start(session);

  if (successful) { /* changed settings */
    /* destroy the window (if possible - should be! ;)) */
    window = (GtkWidget *) gtk_object_get_data(GTK_OBJECT(widget),
					       "dialog_window");
    if (window) {
      gtk_widget_destroy(window);
    }
  } else { /* settings error -> restored old settings -> display note */
    ok_window = get_ok_dialog("Note",
			      "Bad option, please try again.",
			      GTK_JUSTIFY_CENTER);
    gtk_window_set_modal(GTK_WINDOW(ok_window), TRUE);
    gtk_widget_show(ok_window);
  }

  free(old_msn);
  free(old_msns);
  free(old_audio_device_name_in);
  free(old_audio_device_name_out);
}

/*
 * Options menu entries
 */

/* Settings menu entry callback */
static void cb_settings(GtkWidget *widget, gpointer data, guint action) {
  struct session_t *session = (struct session_t *) data;
  GtkWidget *window;/* the dialog window itself */
  GtkWidget *button;/* action area buttons */
  GtkWidget *table; /* tables in frames */
  GtkWidget *label; /* different labels */
  GtkWidget *frame; /* the frames to separate msn/msns from sound devices */

  GtkWidget *msn_entry;
  GtkWidget *msns_entry;
  GtkWidget *audio_device_name_in_entry;
  GtkWidget *audio_device_name_out_entry;

  window = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(window), "ANT Settings");

  /* vbox area */
  frame = gtk_frame_new("ISDN");
  gtk_container_set_border_width(GTK_CONTAINER(frame), 10);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), frame,
		     TRUE, TRUE, 0);
  gtk_widget_show(frame);

  table = gtk_table_new(2,2, FALSE); /* 2 rows, 2 columns, not homogeneous */
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_widget_show(table);

  label = gtk_label_new("Identifying MSN:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1,0,1);
  gtk_widget_show(label);

  msn_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), msn_entry, 1,2,0,1);
  gtk_entry_set_text(GTK_ENTRY(msn_entry), session->msn);
  gtk_widget_grab_focus(GTK_WIDGET(msn_entry));
  gtk_widget_show(msn_entry);

  label = gtk_label_new("Listen to MSNs:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1,1,2);
  gtk_widget_show(label);

  msns_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), msns_entry, 1,2,1,2);
  gtk_entry_set_text(GTK_ENTRY(msns_entry), session->msns);
  gtk_widget_show(msns_entry);

  frame = gtk_frame_new("OSS");
  gtk_container_set_border_width(GTK_CONTAINER(frame), 10);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->vbox), frame,
		     TRUE, TRUE, 0);
  gtk_widget_show(frame);

  table = gtk_table_new(2,2, FALSE); /* 2 rows, 2 columns, not homogeneous */
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_widget_show(table);

  label = gtk_label_new("Input sound device:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1,0,1);
  gtk_widget_show(label);

  audio_device_name_in_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), audio_device_name_in_entry,
			    1,2,0,1);
  gtk_entry_set_text(GTK_ENTRY(audio_device_name_in_entry),
		     session->audio_device_name_in);
  gtk_widget_show(audio_device_name_in_entry);

  label = gtk_label_new("Output sound device:");
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0,1,1,2);
  gtk_widget_show(label);

  audio_device_name_out_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table), audio_device_name_out_entry,
			    1,2,1,2);
  gtk_entry_set_text(GTK_ENTRY(audio_device_name_out_entry),
		     session->audio_device_name_out);
  gtk_widget_show(audio_device_name_out_entry);

  /* action area */
  button = gtk_button_new_with_label("OK");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), button,
		     TRUE, TRUE, 0);
  gtk_object_set_data(GTK_OBJECT(button), "msn_entry", (gpointer) msn_entry);
  gtk_object_set_data(GTK_OBJECT(button), "msns_entry", (gpointer) msns_entry);
  gtk_object_set_data(GTK_OBJECT(button), "audio_device_name_in_entry",
		      (gpointer) audio_device_name_in_entry);
  gtk_object_set_data(GTK_OBJECT(button), "audio_device_name_out_entry",
		      (gpointer) audio_device_name_out_entry);
  gtk_object_set_data(GTK_OBJECT(button), "dialog_window",
		      (gpointer) window);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(cb_settings_ok), session);
  gtk_widget_show(button);

  button = gtk_button_new_with_label("Cancel");
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), button,
		     TRUE, TRUE, 0);
  gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    GTK_OBJECT(window));
  gtk_widget_show(button);
  
  gtk_window_set_modal(GTK_WINDOW(window), TRUE);

  /* show everything */
  gtk_widget_show(window);
}

/*
 * Help menu entries
 */

/* the about menu entry callback */
static void cb_about(GtkWidget *widget, gpointer data, guint action) {
  GtkWidget *window = get_ok_dialog("About ANT",
			"ANT (ANT is Not a Telephone) Version " VERSION "\n"
			"Copyright 2002 Roland Stigge\n\n"
			"This is a small ISDN telephone application\n"
			"written for GNU/Linux and ISDN4Linux for\n"
			"communicating via a full duplex soundcard (or\n"
			"multiple sound devices if you like) and an\n"
			"audio capable I4L ISDN device", GTK_JUSTIFY_CENTER);

  gtk_widget_show(window);
}

/* the about menu entry callback */
static void cb_license(GtkWidget *widget, gpointer data, guint action) {
  GtkWidget *window = get_ok_dialog("ANT License",
    "ANT (ANT is Not a Telephone)\n"
    "Copyright (C) 2002 Roland Stigge\n"
    "\n"
    "This program is free software; you can redistribute it and/or\n"
    "modify it under the terms of the GNU General Public License\n"
    "as published by the Free Software Foundation; either version 2\n"
    "of the License, or (at your option) any later version.\n"
    "\n"
    "This program is distributed in the hope that it will be useful,\n"
    "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
    "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
    "GNU General Public License for more details.\n"
    "\n"
    "You should have received a copy of the GNU General Public License\n"
    "along with this program; if not, write to the Free Software\n"
    "Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, "
    "USA.", GTK_JUSTIFY_LEFT);

  gtk_widget_show(window);
}

/* main window delete_event callback */
gint delete_event(GtkWidget *widget,
		  GdkEvent *event,
		  gpointer data)
{
  quit(widget, data, 0);
  return FALSE; /* continue event handling */
}

/*
 * get the main menu widget
 *
 * input: window: parent window to integrate accelerators
 * returns: the widget yet to pack
 */
GtkWidget *get_main_menu(GtkWidget *window, struct session_t *session) {
  /* The main menu structure */
  GtkItemFactoryEntry main_menu_items[] = {
    /*path                  accel.    callb.       cb param. kind */
    { "/_Phone",            NULL,     NULL,        0,        "<Branch>"     },
    { "/Phone/_Quit",       "<alt>X", quit,        0,        NULL           },
    { "/_Options",          NULL,     NULL,        0,        "<Branch>"     },
    { "/Options/_Settings", NULL,     cb_settings, 0,        NULL           },
    { "/_Help",             NULL,     NULL,        0,        "<LastBranch>" },
    { "/Help/_About",       NULL,     cb_about,    0,        NULL           },
    { "/Help/_License",     NULL,     cb_license,  0,        NULL           },
  };

  GtkItemFactory *item_factory;
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof(main_menu_items) / sizeof(main_menu_items[0]);

  accel_group = gtk_accel_group_new();
  item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>",
				      accel_group);
  gtk_item_factory_create_items_ac(item_factory, nmenu_items, main_menu_items,
				session, 2);
  gtk_window_add_accel_group(GTK_WINDOW(window), accel_group);
  
  return gtk_item_factory_get_widget(item_factory, "<main>");
}

/*
 * main function for gtk gui
 *
 * returns int to be returned from main()
 */
int main_gtk(struct session_t *session) {
  GtkWidget *main_window;
  GtkWidget *main_vbox;
  GtkWidget *main_menu;

  GtkWidget *label;
  GtkWidget *hbox;

  /* the main window */
  main_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(main_window), "ANT");

  gtk_signal_connect(GTK_OBJECT(main_window), "delete_event",
		     GTK_SIGNAL_FUNC(delete_event), NULL);

  gtk_container_set_border_width(GTK_CONTAINER(main_window), 0);

  /* main vbox */
  main_vbox = gtk_vbox_new(FALSE, 0); /* not homogeneous, spacing = 0*/
  gtk_container_set_border_width(GTK_CONTAINER(main_vbox), 0);
  gtk_container_add(GTK_CONTAINER(main_window), main_vbox);
  gtk_widget_show(main_vbox);

  /* fill main vbox ... */

  /* main menu bar */
  main_menu = get_main_menu(main_window, session); /* main menu */
  gtk_box_pack_start(GTK_BOX(main_vbox), main_menu, FALSE, TRUE, 0);
  gtk_widget_show(main_menu);

  /* the dial hbox */
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
  gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
  gtk_widget_show(hbox);

  /* dial label */
  label = gtk_label_new("Number:");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* dial number entry */
  session->dial_number_entry = gtk_entry_new();
  gtk_entry_set_text(GTK_ENTRY(session->dial_number_entry), DEFAULT_DIALNUMBER);
  gtk_object_set(GTK_OBJECT(session->dial_number_entry), "width", 150, NULL);
  gtk_box_pack_start(GTK_BOX(hbox), session->dial_number_entry,
		     FALSE, FALSE, 0);
  gtk_widget_grab_focus(GTK_WIDGET(session->dial_number_entry));
  gtk_widget_show(session->dial_number_entry);

  /* pick up button */
  session->pick_up_button = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), session->pick_up_button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(session->pick_up_button), "clicked",
		     GTK_SIGNAL_FUNC(gtk_handle_pick_up_button),
		     (gpointer) session);
  gtk_object_set(GTK_OBJECT(session->pick_up_button), "width", 60, NULL);
  /* XXX: set automatically to maximum possibly needed size */
  gtk_widget_show(session->pick_up_button);

  /* pick up button label */
  session->pick_up_label = gtk_label_new(NULL);
  gtk_container_add(GTK_CONTAINER(session->pick_up_button),
		    session->pick_up_label);
  gtk_widget_show(session->pick_up_label);

   /* hang up button */
  session->hang_up_button = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(hbox), session->hang_up_button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(session->hang_up_button), "clicked",
		     GTK_SIGNAL_FUNC(gtk_handle_hang_up_button),
		     (gpointer) session);
  gtk_object_set(GTK_OBJECT(session->hang_up_button), "width", 60, NULL);
  /* XXX: set automatically to maximum possibly needed size */
  gtk_widget_show(session->hang_up_button);

  /* hang up button label */
  session->hang_up_label = gtk_label_new(NULL);
  gtk_container_add(GTK_CONTAINER(session->hang_up_button),
		    session->hang_up_label);
  gtk_widget_show(session->hang_up_label);
  
  /* status bar */
  session->status_bar = gtk_statusbar_new();
  gtk_box_pack_end(GTK_BOX(main_vbox), session->status_bar, TRUE, TRUE, 2);
  gtk_widget_show(session->status_bar);
  session->phone_context_id =
    gtk_statusbar_get_context_id(GTK_STATUSBAR(session->status_bar), "phone");
  gtk_statusbar_push(GTK_STATUSBAR(session->status_bar),
		     session->phone_context_id, "");

  /* set up for initial state */
  session_set_state(session, session->state); /* state is already in session */

  /* connect isdn and sound input handlers to gdk pseudo select */
  session_io_handlers_start(session);

  /* set up additional handlers */
  gtk_timeout_add(TIMER_DELAY, timeout_callback, NULL);
  signal(SIGINT, &terminate_signal_callback);
  signal(SIGTERM, &terminate_signal_callback);

  /* show everything */
  gtk_widget_show(main_window);

  /* gtk main loop */
  gtk_main();
  
  /* some deinit */
  session_io_handlers_stop(session);

  return 0;
}
