/*
 *   This file is part of AkariXB
 *   Copyright 2015-2019  JanKusanagi JRR <jancoding@gmx.com>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .
 */

#include "roommodule.h"


RoomModule::RoomModule(GlobalObject *globalObject,
                       QWidget *parent) : QWidget(parent)
{
    m_globalObject = globalObject;

    m_xmppClient = m_globalObject->getXmppClient();
    m_mucManager = m_globalObject->getMucManager();


    m_dataFile = new DataFile(m_globalObject->getDataDirectory()
                              + "/rooms.axb",
                              this);


    // List of configured rooms
    m_roomListWidget = new QListWidget(this);
    m_roomListWidget->setDragDropMode(QAbstractItemView::InternalMove); // Movable items
    connect(m_roomListWidget, &QListWidget::currentItemChanged, // 2nd arg ignored
            this, &RoomModule::showRoomDetails);



    // Buttons to manage rooms
    m_addRoomButton = new QPushButton(QIcon::fromTheme("list-add",
                                                       QIcon(":/images/list-add.png")),
                                      tr("Add &New Room"),
                                      this);
    m_addRoomButton->setFlat(true);
    connect(m_addRoomButton, &QAbstractButton::clicked,
            this, &RoomModule::addRoom);


    m_removeRoomButton = new QPushButton(QIcon::fromTheme("list-remove",
                                                          QIcon(":/images/list-remove.png")),
                                         tr("&Remove Room"),
                                         this);
    m_removeRoomButton->setFlat(true);
    m_removeRoomButton->setDisabled(true);
    connect(m_removeRoomButton, &QAbstractButton::clicked,
            this, &RoomModule::removeRoom);


    m_buttonsLayout = new QHBoxLayout();
    m_buttonsLayout->addWidget(m_addRoomButton);
    m_buttonsLayout->addStretch();
    m_buttonsLayout->addWidget(m_removeRoomButton);


    // Room details
    m_roomJidLineEdit = new QLineEdit(this);
    m_roomJidLineEdit->setPlaceholderText(tr("Room address"));

    m_nickLineEdit = new QLineEdit(this);
    m_nickLineEdit->setPlaceholderText(tr("Nickname"));
    connect(m_roomJidLineEdit, SIGNAL(returnPressed()), // Leaving old-style
            m_nickLineEdit, SLOT(setFocus()));          // connect() due to overload

    m_roomPasswordLineEdit = new QLineEdit(this);
    m_roomPasswordLineEdit->setPlaceholderText(tr("Password, if needed"));
    m_roomPasswordLineEdit->setEchoMode(QLineEdit::Password);
    connect(m_nickLineEdit, SIGNAL(returnPressed()),   // Same as previous connect()
            m_roomPasswordLineEdit, SLOT(setFocus()));

    m_autojoinCheckbox = new QCheckBox(tr("Autojoin"), this);


    // Edit / Update
    m_editRoomButton = new QPushButton(QIcon::fromTheme("document-edit",
                                                        QIcon(":/images/button-edit.png")),
                                       tr("&Edit"),
                                       this);
    m_editRoomButton->setDisabled(true);
    connect(m_editRoomButton, &QAbstractButton::clicked,
            this, &RoomModule::editRoom);

    m_updateRoomButton = new QPushButton(QIcon::fromTheme("view-refresh",
                                                          QIcon(":/images/button-refresh.png")),
                                         tr("&Update"),
                                         this);
    connect(m_updateRoomButton, &QAbstractButton::clicked,
            this, &RoomModule::updateRoom);
    connect(m_roomPasswordLineEdit, SIGNAL(returnPressed()), // Same as previous
            m_updateRoomButton, SLOT(setFocus()));


    // Join / Leave
    m_joinButton = new QPushButton(QIcon::fromTheme("list-add",
                                                    QIcon(":/images/list-add.png")),
                                   tr("Join"),
                                   this);
    m_joinButton->hide();
    connect(m_joinButton, &QAbstractButton::clicked,
            this, &RoomModule::joinSelectedRoom);

    m_leaveButton = new QPushButton(QIcon::fromTheme("list-remove",
                                                     QIcon(":/images/list-remove.png")),
                                    tr("Leave"),
                                    this);
    m_leaveButton->hide();
    connect(m_leaveButton, &QAbstractButton::clicked,
            this, &RoomModule::leaveSelectedRoom);




    // Stacked widget to hold ChatWidgets
    m_firstPageLabel = new QLabel("<big>"
                                  + tr("Select a room from the list above")
                                  + "</big>",
                                  this);
    m_firstPageLabel->setAlignment(Qt::AlignCenter);

    m_chatStackedWidget = new QStackedWidget(this);
    m_chatStackedWidget->addWidget(m_firstPageLabel);


    m_detailsUpperLayout = new QHBoxLayout();
    m_detailsUpperLayout->addWidget(m_roomJidLineEdit,      3);
    m_detailsUpperLayout->addWidget(m_nickLineEdit,         2);
    m_detailsUpperLayout->addWidget(m_roomPasswordLineEdit, 2);
    m_detailsUpperLayout->addWidget(m_autojoinCheckbox);
    m_detailsUpperLayout->addStretch();
    m_detailsUpperLayout->addSpacing(8);
    m_detailsUpperLayout->addWidget(m_editRoomButton);
    m_detailsUpperLayout->addWidget(m_updateRoomButton);
    m_detailsUpperLayout->addWidget(m_joinButton);
    m_detailsUpperLayout->addWidget(m_leaveButton);


    m_detailsLayout = new QVBoxLayout();
    m_detailsLayout->addLayout(m_detailsUpperLayout);
    m_detailsLayout->addWidget(m_chatStackedWidget);

    m_detailsGroupbox = new QGroupBox(tr("Room Details"), this);
    m_detailsGroupbox->setLayout(m_detailsLayout);


    m_mainLayout = new QVBoxLayout();
    m_mainLayout->setContentsMargins(0, 0, 0, 0);
    m_mainLayout->addWidget(m_roomListWidget,  1);
    m_mainLayout->addLayout(m_buttonsLayout);
    m_mainLayout->addWidget(m_detailsGroupbox, 3);
    this->setLayout(m_mainLayout);


    this->loadRoomInfo();
    this->toggleRoomDetails(false);

    qDebug() << "RoomModule created";
}


RoomModule::~RoomModule()
{
    qDebug() << "RoomModule destroyed";
}



void RoomModule::loadRoomInfo()
{
    QVariantList roomDataList = m_dataFile->loadData();
    QStringList configuredRoomsList;

    foreach (QVariant listItem, roomDataList)
    {
        QVariantMap map = listItem.toMap();

        const QString roomJid = map.value("jid").toString();
        const QString roomNickname = map.value("nickname").toString();
        const QString roomPassword = QByteArray::fromBase64(map.value("password")
                                                               .toByteArray());
        const bool roomAutojoin = map.value("autojoin").toBool();
        const QString itemString = this->roomDescriptionLine(roomJid,
                                                             roomNickname,
                                                             !roomPassword.isEmpty(),
                                                             roomAutojoin);

        QListWidgetItem *item = new QListWidgetItem(QIcon::fromTheme("user-offline",
                                                                     QIcon(":/images/button-offline.png")),
                                                    itemString);
        item->setData(Qt::UserRole,     roomJid);
        item->setData(Qt::UserRole + 1, roomNickname);
        item->setData(Qt::UserRole + 2, roomPassword);
        item->setData(Qt::UserRole + 3, roomAutojoin);
        item->setData(Qt::UserRole + 4, false); // Joined status, initially false

        m_roomListWidget->addItem(item);

        configuredRoomsList.append(roomJid);
    }

    // Keep track of the configured rooms, for some checks
    m_globalObject->setConfiguredRoomsList(configuredRoomsList);

    m_globalObject->addToLog(tr("Loaded %Ln room(s).",
                                "",
                                roomDataList.length()));
}


void RoomModule::saveRoomInfo()
{
    QVariantList roomDataList;
    QStringList configuredRoomsList;

    for (int counter = 0; counter < m_roomListWidget->count(); ++counter)
    {
        QListWidgetItem *item = m_roomListWidget->item(counter);

        QString roomJid = item->data(Qt::UserRole).toString().trimmed();
        QString roomNick = item->data(Qt::UserRole + 1).toString().trimmed();

        // Don't save data for invalid rooms
        if (Helpers::isValidJid(roomJid) && !roomNick.isEmpty())
        {
            QVariantMap roomMap;
            roomMap.insert("jid",        roomJid);
            roomMap.insert("nickname",   roomNick);
            roomMap.insert("password",   item->data(Qt::UserRole + 2).toByteArray()
                                                                     .toBase64());
            roomMap.insert("autojoin",   item->data(Qt::UserRole + 3).toBool());

            roomDataList.append(roomMap);

            configuredRoomsList.append(roomJid);
        }
    }

    m_dataFile->saveData(roomDataList);

    m_globalObject->setConfiguredRoomsList(configuredRoomsList);


    m_globalObject->addToLog(tr("Saved %Ln room(s).", "",
                                 roomDataList.length()),
                             true,
                             GlobalObject::LogGood);
}



QString RoomModule::roomDescriptionLine(QString roomJid, QString nickname,
                                        bool hasPassword, bool autojoin)
{
    // FIXME: this list needs to be converted into a proper table
    QString description = tr("%1 as %2").arg(roomJid).arg(nickname);

    if (description.length() < 30) // Until then, try to align it a bit
    {
        description.append("\t");
        if (description.length() < 20)
        {
            description.append("\t");
            if (description.length() < 10)
            {
                description.append("\t");
            }
        }
    }

    if (hasPassword)
    {
        description.append(QString::fromUtf8("\t \342\232\277 ") // Squared key
                           + "("
                           + tr("Password")
                           + ")");
    }

    if (autojoin)
    {
        description.append(QString::fromUtf8("\t \342\234\224  ") // Check mark
                           + "("
                           + tr("Autojoin")
                           + ")");
    }

    return description;
}


/*
 * Find out which item from the list matches a given JID
 *
 */
QListWidgetItem *RoomModule::listItemFromJid(QString roomJid)
{
    for (int count = 0; count < m_roomListWidget->count(); ++count)
    {
        QListWidgetItem *item = m_roomListWidget->item(count);

        if (item->data(Qt::UserRole).toString() == roomJid)
        {
            return item;
        }
    }

    return nullptr;
}



/*
 * Get the page number in the QStackedWidget that matches
 * the ChatWidget with the given room JID
 *
 */
int RoomModule::stackIndexFromJid(QString roomJid)
{
    qDebug() << "RoomModule::stackIndexFromJid()" << roomJid;
    qDebug() << m_openedRooms;

    int page = 1; // Starts with 1, since 0 is an empty page with no chat
    foreach (ChatWidget *chat, m_openedRooms)
    {
        if (chat->getRoomJid() == roomJid)
        {
            return page;
        }

        ++page;
    }

    return 0;
}



void RoomModule::joinRoom(QString roomJid, QString nickname, QString password)
{
    QXmppMucRoom *room = m_mucManager->addRoom(roomJid);
    room->setNickName(nickname);
    if (!password.isEmpty())
    {
        room->setPassword(password);
    }

    room->join();


    const QString message = tr("Joining room %1...")
                            .arg("<b>" + room->jid().toHtmlEscaped() + "</b>");
    m_globalObject->addToLog(message);


    ChatWidget *newChatWidget = new ChatWidget(m_globalObject, this);
    newChatWidget->setRoom(room);
    connect(newChatWidget, &ChatWidget::roomJoined,
            this, &RoomModule::onRoomJoined);
    connect(newChatWidget, &ChatWidget::roomLeft,
            this, &RoomModule::onRoomLeft);


    m_chatStackedWidget->addWidget(newChatWidget);
    m_openedRooms.append(newChatWidget);

    m_globalObject->addJoinedRoom(roomJid);
}


void RoomModule::leaveRoom(QString roomJid, QString reason)
{
    QXmppMucRoom *room = m_globalObject->roomFromJid(roomJid);
    if (room != nullptr)
    {
        const QString message = tr("Leaving room %1...")
                                .arg("<b>" + room->jid().toHtmlEscaped() + "</b>");
        m_globalObject->addToLog(message);

        room->leave(reason);
    }
    else
    {
        qDebug() << "Trying to leave: " << roomJid;
        qDebug() << "Room NOT FOUND via globalObject!";
    }

    qDebug() << "RoomModule::leaveRoom() ended" << roomJid;
}



/*
 * Join rooms configured for autojoin
 *
 */
void RoomModule::autojoinRooms()
{
    for (int count = 0; count < m_roomListWidget->count(); ++count)
    {
        QListWidgetItem *item = m_roomListWidget->item(count);

        QString roomJid = item->data(Qt::UserRole).toString();
        QString roomNick = item->data(Qt::UserRole + 1).toString();
        QString roomPass = item->data(Qt::UserRole + 2).toString();

        if (item->data(Qt::UserRole + 3).toBool()               // autojoin=true
         || m_globalObject->getJoinedRooms().contains(roomJid)) // or previously joined
        {
            this->joinRoom(roomJid, roomNick, roomPass);
        }
    }
}


bool RoomModule::newRoomIsValid(bool showErrors)
{
    const QString roomJid = m_roomJidLineEdit->text().trimmed();
    if (!Helpers::isValidJid(roomJid))
    {
        if (showErrors)
        {
            QMessageBox::warning(this,
                                 tr("Error"),
                                 tr("Room address is not valid."));
        }

        m_roomJidLineEdit->setFocus();

        return false;
    }

    if (m_nickLineEdit->text().trimmed().isEmpty())
    {
        if (showErrors)
        {
            QMessageBox::warning(this, tr("Error"),
                                 tr("Nickname is not valid."));
        }

        m_nickLineEdit->setFocus();

        return false;
    }

    return true;
}


void RoomModule::clearRoomDetails()
{
    m_roomJidLineEdit->clear();
    m_nickLineEdit->clear();
    m_roomPasswordLineEdit->clear();
    m_autojoinCheckbox->setChecked(false);
}


void RoomModule::toggleJoinLeaveButtons(QListWidgetItem *item)
{
    // UserRole+4 contains true if joined
    if (item->data(Qt::UserRole + 4).toBool())
    {
        m_joinButton->hide();
        m_leaveButton->show();
    }
    else
    {
        m_joinButton->show();
        m_leaveButton->hide();
    }
}



//////////////////////////////////////////////////////////////////////////////
///////////////////////////////////// SLOTS //////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



void RoomModule::addRoom()
{
    m_roomListWidget->addItem(" --- " + tr("New Room") + " ---");

    // Select this new item
    m_roomListWidget->setCurrentRow(m_roomListWidget->count() - 1);

    this->clearRoomDetails();

    this->editRoom();
}


/*
 * Enable fields for room editing, and Update button
 *
 */
void RoomModule::editRoom()
{
    this->toggleRoomDetails(true);
}


void RoomModule::updateRoom()
{
    if (m_roomListWidget->currentRow() < 0)
    {
        return;
    }

    const QString roomJid = m_roomJidLineEdit->text().trimmed();
    const QString roomNickname = m_nickLineEdit->text().trimmed();
    const QString roomPassword = m_roomPasswordLineEdit->text();
    const bool roomAutojoin = m_autojoinCheckbox->isChecked();

    if (newRoomIsValid(true)) // Showing errors = true
    {
        QListWidgetItem *item = m_roomListWidget->currentItem();

        QString itemString = this->roomDescriptionLine(roomJid,
                                                       roomNickname,
                                                       !roomPassword.isEmpty(),
                                                       roomAutojoin);
        // Icon depends on current joined status
        if (item->data(Qt::UserRole + 4).toBool())
        {
            item->setIcon(QIcon::fromTheme("user-online",
                                           QIcon(":/images/button-online.png")));

            // For online rooms, check if nick needs changing
            if (item->data(Qt::UserRole + 1).toString() != roomNickname)
            {
                this->changeNick();
            }
        }
        else
        {
            item->setIcon(QIcon::fromTheme("user-offline",
                                           QIcon(":/images/button-offline.png")));
        }

        item->setText(itemString);
        item->setData(Qt::UserRole,     roomJid);
        item->setData(Qt::UserRole + 1, roomNickname);
        item->setData(Qt::UserRole + 2, roomPassword);
        item->setData(Qt::UserRole + 3, roomAutojoin);

        this->toggleRoomDetails(false);

        this->saveRoomInfo();
    }
}


void RoomModule::removeRoom()
{
    if (m_roomListWidget->currentRow() >= 0) // Ensure something's selected
    {
        // Don't takeItem(), otherwise leaveSelectedRoom() will act on another one
        QListWidgetItem *selectedItem = m_roomListWidget
                                        ->item(m_roomListWidget->currentRow());
        const QString roomJid = selectedItem->data(Qt::UserRole).toString();

        const int confirm = QMessageBox::question(this,
                             tr("Remove room?") + " - AkariXB",
                             tr("Do you want to remove room %1 from your list?")
                             .arg("<b>'" + roomJid + "'</b>")
                             + "<br />",
                             tr("Yes, remove it", "It = A chat room"), tr("No"),
                             QString(), 1, 1);
        if (confirm == 1) // Not sure, so bail
        {
            return;
        }

        if (m_globalObject->isRoomFullyJoined(roomJid))
        {
            qDebug() << "Leaving room" << roomJid << "before removing it";
            this->leaveSelectedRoom();
        }

        delete selectedItem;
        this->toggleRoomDetails(false);

        if (m_roomListWidget->count() == 0)
        {
            m_removeRoomButton->setDisabled(true);

            // Remove any trace of the last removed room, and disable any
            // button that allows editing or joining nonexistent rooms
            this->clearRoomDetails();
            m_editRoomButton->setDisabled(true);
            m_updateRoomButton->setDisabled(true);
            m_joinButton->setDisabled(true);
            m_leaveButton->setDisabled(true);
        }

        this->saveRoomInfo();
    }
}



void RoomModule::joinSelectedRoom()
{
    this->joinRoom(m_roomJidLineEdit->text().trimmed(),
                   m_nickLineEdit->text().trimmed(),
                   m_roomPasswordLineEdit->text());

    // Room's ChatWidget will be shown later, via onRoomJoined()

    m_joinButton->setDisabled(true);
}


void RoomModule::leaveSelectedRoom()
{
    const QString roomJid = m_roomJidLineEdit->text();
    this->leaveRoom(roomJid, "Bye!"); // TMP reason FIXME

    // Remove it from joined rooms, since we're not leaving due to errors
    m_globalObject->removeJoinedRoom(roomJid);
    m_globalObject->removeFullyJoinedRoom(roomJid);

    QListWidgetItem *item = m_roomListWidget->currentItem();
    this->showRoomDetails(item);
}


void RoomModule::changeNick()
{
    QXmppMucRoom *room = m_globalObject->roomFromJid(m_roomJidLineEdit->text());
    if (room != nullptr)
    {
        room->setNickName(m_nickLineEdit->text().trimmed());
    }
}



void RoomModule::showRoomDetails(QListWidgetItem *item)
{
    // item comes from call to listItemFromJid(), it will be nullptr if invalid
    if (item == nullptr)
    {
        return;
    }

    //qDebug() << "showRoomDetails()" << item->data(Qt::UserRole).toString();
    m_roomJidLineEdit->setText(item->data(Qt::UserRole).toString());
    m_nickLineEdit->setText(item->data(Qt::UserRole + 1).toString());
    m_roomPasswordLineEdit->setText(item->data(Qt::UserRole + 2).toString());
    m_autojoinCheckbox->setChecked(item->data(Qt::UserRole + 3).toBool());

    this->toggleRoomDetails(false);

    m_editRoomButton->setEnabled(true);
    m_removeRoomButton->setEnabled(true);


    this->toggleJoinLeaveButtons(item);


    const int stackPos = this->stackIndexFromJid(m_roomJidLineEdit->text());
    m_chatStackedWidget->setCurrentIndex(stackPos);
}


void RoomModule::toggleRoomDetails(bool state)
{
    // Room JID field will only be enabled if the room is not currently joined
    bool roomJoined = false;
    if (m_roomListWidget->currentRow() >= 0) // Ensure something's selected
    {
        roomJoined = m_roomListWidget->currentItem()
                                     ->data(Qt::UserRole + 4).toBool();
    }

    m_roomJidLineEdit->setEnabled(state && !roomJoined);
    m_nickLineEdit->setEnabled(state);
    m_roomPasswordLineEdit->setEnabled(state);
    m_autojoinCheckbox->setEnabled(state);

    m_updateRoomButton->setVisible(state);
    m_updateRoomButton->setEnabled(state);
    m_editRoomButton->setHidden(state); // Reversed state!


    // Ensure join button isn't available for invalid rooms, or when offline
    m_joinButton->setEnabled(m_globalObject->connectedToServer()
                             && this->newRoomIsValid());

    m_leaveButton->setEnabled(m_globalObject->connectedToServer());


    if (state)
    {
        if (roomJoined)  // Jump directly to the nickname field
        {
            m_nickLineEdit->setFocus();
        }
        else             // Room JID is editable, so jump to that one first
        {
            m_roomJidLineEdit->setFocus();
        }
    }
    else
    {
        m_roomListWidget->setFocus();
    }
}



void RoomModule::onRoomJoined(QString jid)
{
    QListWidgetItem *item = this->listItemFromJid(jid);
    if (item != nullptr)
    {
        item->setIcon(QIcon::fromTheme("user-online",
                                       QIcon(":/images/button-online.png")));

        item->setData(Qt::UserRole + 4, true); // Mark as joined

        if (item->isSelected())
        {
            this->toggleJoinLeaveButtons(item);
            m_joinButton->setEnabled(true);

            // Ensure room is visible, even after reconnections
            this->showRoomDetails(item);
        }

        // TODO: ability to query participants at any time, shown in UI
        qDebug() << "Joined" << jid << ";\nCurrent participants:"
                 << m_globalObject->roomFromJid(jid)->participants();
    }
}


void RoomModule::onRoomLeft(QString jid)
{
    qDebug() << "RoomModule::onRoomLeft()" << jid;

    for (int count = 0; count < m_openedRooms.size(); ++count)
    {
        if (m_openedRooms.at(count)->getRoomJid() == jid)
        {
            m_openedRooms.at(count)->deleteLater();
            m_openedRooms.removeAt(count);
            // break; ?
            // What if for some reason the user added same room twice,
            // for different nicks or whatever?
            // FIXME: this shouldn't be allowed
        }
    }

    QListWidgetItem *item = this->listItemFromJid(jid);
    if (item != nullptr) // In case of leaving due to room removal
    {
        item->setData(Qt::UserRole + 4, false); // Set as not-joined
        item->setIcon(QIcon::fromTheme("user-offline",
                                       QIcon(":/images/button-offline.png")));
        this->showRoomDetails(item);
    }
}
